#include "gfx/font.hpp"
#include "snd/stream.hpp"
#include "thr/dispatch.hpp"
#include "ui/console.hpp"
#include "ui/console_state.hpp"
#include "ui/text_area.hpp"
#include "ui/ui_state.hpp"
#include "ui/ui_stack.hpp"

#include <boost/program_options.hpp>

namespace po = boost::program_options;

/** Test string content. */
static const std::wstring	test_string(ui::wstr_utf8("This is a test text box."));

/** Test string content. */
static const std::wstring naju_latin(ui::wstr_utf8("Naju"));

/** Test string content. */
static const std::wstring naju_cyrillic(ui::wstr_utf8("Наю"));

/** Test string content. */
static const std::wstring naju_hiragana(ui::wstr_utf8("なゆ"));

/** Console output content. */
static const char *usage = ""
"Usage: test_2d <options>\n"
"Naju graphics engine pixel-perfect 2d renderer test program.\n"
"Not for any sensible use.\n"
"\n";

/** \brief Data for 2D-test.
 */
class TestData
{
	private:
		/** Font used for tests. */
		gfx::Font _fnt;

		/** Sample for sound shit. */
		snd::Sample m_sample;

		/** Image used for tests. */
		gfx::Texture2D _tex;

		/** Console test. */
		ui::Console _console;

	public:
		inline ui::Console& getConsole()
		{
			return _console;
		}

		inline const gfx::Font& getFont() const
		{
			return _fnt;
		}

		inline snd::Sample& getSample()
		{
			return m_sample;
		}

		inline const gfx::Texture2D& getTexture() const
		{
			return _tex;
		}

	public:
		TestData(const gfx::SurfaceScreen &pscreen) :
			_fnt("fnt/default.xml"),
			m_sample("snd/ob_nuke.wav"),
			_tex("gfx/general/ob_console_background_2.png"),
			_console(_fnt, 0.03f, pscreen, "gfx/general/ob_console_background_1.png")
		{
			_console.setProgressMax(20);
 		}
};

/** \brief Graphics test class. */
class Test2D : public ui::UiState
{
	private:
		/** Maximum test index. */
		static const int TEST_COUNT = 7;

		/** Static test count. */
		static const int STATIC_TESTS = 6;

	private:
		/** Current test index. */
		int _current_test;

		/** Has the test changed this frame? */
		bool _test_changed;

		/** Gravity for test text box. */
		ui::TextGravity _grav;

		/** Test data. */
		TestData &_data;

		/** Stream pointer. */
		snd::StreamSptr m_stream;

	public:
		/** \brief Constructor.
		 */
		Test2D(TestData &data) :
			_current_test(0),
			_test_changed(true),
			_grav(ui::GRAVITY_DOWN_LEFT),
			_data(data) { }

		virtual ~Test2D()
		{
			// Do nothing.
		}

	private:
		/** \brief Convert a screen coordinate into a color.
		 * 
		 * @param px X coordinate.
		 * @param py Y coordinate.
		 * @param pw Width of the screen.
		 * @param ph Height of the screen.
		 * @param cmyk Use CMYK instead of RGB.
		 * @return Color.
		 */
		static gfx::Color coord2Color(int px, int py, int pw, int ph, bool cmyk)
		{
			int ww = pw - 1,
					hh = ph - 1;
			math::vec2f d0(static_cast<float>(px), static_cast<float>(py)),
				d1(static_cast<float>(ww - px), static_cast<float>(py)),
				d2(static_cast<float>(px), static_cast<float>(hh - py)),
				d3(static_cast<float>(ww - px), static_cast<float>(hh - py));
			float len0 = length(d0),
						len1 = length(d1),
						len2 = length(d2),
						len3 = length(d3),
						lensum = len0 + len1 + len2 + len3;

			len0 = 1.0f - len0 / lensum;
			len1 = 1.0f - len1 / lensum;
			len2 = 1.0f - len2 / lensum;
			len3 = 1.0f - len3 / lensum;

			if(!cmyk)
			{
				return gfx::Color(len0, len1, len2, 1.0f);
			}
			return gfx::Color::fromCMYK(len0, len1, len2, len3, 1.0f);
		}

		/** \brief Render everything.
		 *
		 * @param screen Screen to use.
		 * @param st UI stack.
		 */
		void render(gfx::SurfaceScreen &screen, ui::UiStack &st)
		{
			int ww = screen.getWidth(),
					hh = screen.getHeight();
			float fww = static_cast<float>(ww),
						fhh = static_cast<float>(hh);

			if((_current_test < STATIC_TESTS) && !_test_changed)
			{
				return;
			}

			screen.clear(true, true);

			switch(_current_test)
			{
				case 0:
					screen.select2D();
					gfx::bind_shader_2d();
					{
						for(int ii = 0; (ii < ww); ++ii)
						{
							for(int jj = 0; (jj < hh); ++jj)
							{
								gfx::draw_pixel(ii, jj, coord2Color(ii, jj, ww, hh, false));
							}
						}
					}
					break;

				case 1:
					screen.select2D();
					gfx::bind_shader_2d();
					{
						int wwdiv = ww / 6,
								hhdiv = hh / 6;

						for(int ii = 0; (ii < wwdiv); ++ii)
						{
							float relpos = static_cast<float>(ii) / static_cast<float>(wwdiv - 1);
							int x = static_cast<int>(relpos * (fww - 1.0f)),
									y = static_cast<int>(relpos * (fhh - 1.0f)),
									rx = ww - 1 - x,
									ry = hh - 1 - y;
							gfx::draw_line(x, 0, ww - 1, y,
									coord2Color(x, 0, ww, hh, true),
									coord2Color(ww - 1, y, ww, hh, true));
							gfx::draw_line(rx, hh - 1, 0, ry,
									coord2Color(rx, hh - 1, ww, hh, true),
									coord2Color(0, ry, ww, hh, true));
						}
						for(int ii = 0; (ii < hhdiv); ++ii)
						{
							float relpos = static_cast<float>(ii) / static_cast<float>(hhdiv - 1);
							int x = static_cast<int>(relpos * (fww - 1.0f)),
									y = static_cast<int>(relpos * (fhh - 1.0f)),
									rx = ww - 1 - x,
									ry = hh - 1 - y;
							gfx::draw_line(0, ry, x, 0,
									coord2Color(0, ry, ww, hh, true),
									coord2Color(x, 0, ww, hh, true));
							gfx::draw_line(ww - 1, y, rx, hh - 1,
									coord2Color(ww - 1, y, ww, hh, true),
									coord2Color(rx, hh - 1, ww, hh, true));
						}
					}
					break;

				case 2:
					screen.select2D();
					gfx::bind_shader_2d();
					{
						int div = 32,
								wwdiv = ww / div,
								hhdiv = hh / div;

						for(int ii = 0; (ii <= div); ++ii)
						{
							for(int jj = 0; (jj <= div); ++jj)
							{
								gfx::draw_rect(ii * wwdiv, jj * hhdiv, wwdiv + 1, hhdiv + 1,
										coord2Color(ii * wwdiv, jj * hhdiv, ww, hh, true));
							}
						}
					}
					break;

				case 3:
					screen.select2D();
					gfx::bind_shader_2d();
					{
						int wwhh = std::min(ww, hh);
						for(int ii = 0; (ii < wwhh / 2); ii += 2)
						{
							gfx::draw_rect_contour(ii, ii, ww - ii * 2, hh - ii * 2,
									coord2Color(ii, ii, ww, hh, false));
						}
					}
					break;

				case 4:
					{
						screen.select2D();
						gfx::bind_shader_2d_texture();

						gfx::draw_rect_textured(0, 0, ww, hh,
								gfx::Color(1.0f, 1.0f, 1.0f, 1.0f),
								_data.getTexture());
					}
					break;

				case 5:
					screen.select2D();
					gfx::bind_shader_2d();
					{
						float wwf = fww * gfx::Surface::get_mul_2d(),
									hhf = fhh * gfx::Surface::get_mul_2d();
						ui::TextArea tarea(
								math::rect2f(math::vec2f(0.05f, 0.05f), math::vec2f(0.5f, 0.5f)),
								0.07f, _data.getFont(), _grav);
						tarea.setMargins(0.5f, 0.5f, 0.6f, 0.4f);
						tarea.setContent(test_string);

						gfx::Color margincolor(1.0f, 0.6f, 0.6f, 1.0f);
						math::rect2f rect(tarea.getContentArea());
						gfx::draw_line(0.0f, rect.y1(), wwf, rect.y1(), margincolor);
						gfx::draw_line(0.0f, rect.y2(), wwf, rect.y2(), margincolor);
						gfx::draw_line(rect.x2(), 0.0f, rect.x2(), hhf, margincolor);
						gfx::draw_line(rect.x1(), 0.0f, rect.x1(), hhf, margincolor);

						tarea.render(gfx::Color(0.8f, 0.8f, 0.8f, 1.0f),
								gfx::Color(0.3f, 0.9f, 0.3f, 1.0f));
					}
					break;

				case 6:
					screen.select2D();
					gfx::bind_shader_2d_font();
					{
						float ffrnum = static_cast<float>(st.getFrameCount());
						gfx::Color fntcolor(0.9f, 0.9f, 0.9f, 1.0f);
						const gfx::Font &fnt = _data.getFont();

						gfx::draw_text(ww * 2 / 5, hh * 2 / 9,
								static_cast<int>(fww * 4.0f / 9.0f * (0.5f + 0.5f * math::sin(ffrnum * 0.0052f))),
								naju_latin, fnt, fntcolor, gfx::CENTER);

						gfx::draw_text(ww * 3 / 9, hh * 5 / 6,
								static_cast<int>(fww / 2.0f * (0.5f + 0.5f * math::sin(ffrnum * 0.0065f))),
								naju_cyrillic, fnt, fntcolor, gfx::CENTER);

						gfx::draw_text(ww * 5 / 7, hh * 4 / 7,
								static_cast<int>(fww / 3.0f * (0.5f + 0.5f * math::sin(ffrnum * 0.004f))),
								naju_hiragana, fnt, fntcolor, gfx::CENTER);
					}
					break;

				default:
					{
						std::stringstream err;
						err << "invalid test: " << _current_test;
						BOOST_THROW_EXCEPTION(std::runtime_error(err.str()));
					}
					break;
			}

			screen.update();
			_test_changed = false;
		}

	public:
		/** \cond */
		virtual bool handle(ui::EventKey &ev, ui::UiStack &st)
		{
			if(ev.isPress())
			{
				switch(ev.getCode())
				{
					case SDLK_KP1:
						_grav = ui::GRAVITY_DOWN_LEFT;
						_test_changed = true;
						break;

					case SDLK_KP2:
						_grav = ui::GRAVITY_DOWN;
						_test_changed = true;
						break;

					case SDLK_KP3:
						_grav = ui::GRAVITY_DOWN_RIGHT;
						_test_changed = true;
						break;

					case SDLK_KP4:
						_grav = ui::GRAVITY_LEFT;
						_test_changed = true;
						break;

					case SDLK_KP5:
						_grav = ui::GRAVITY_CENTER;
						_test_changed = true;
						break;

					case SDLK_KP6:
						_grav = ui::GRAVITY_RIGHT;
						_test_changed = true;
						break;

					case SDLK_KP7:
						_grav = ui::GRAVITY_UP_LEFT;
						_test_changed = true;
						break;

					case SDLK_KP8:
						_grav = ui::GRAVITY_UP;
						_test_changed = true;
						break;

					case SDLK_KP9:
						_grav = ui::GRAVITY_UP_RIGHT;
						_test_changed = true;
						break;

					case SDLK_LEFT:
					case SDLK_UP:
						if(--_current_test < 0)
						{
							_current_test = TEST_COUNT - 1;
						}
						_test_changed = true;
						break;

					case SDLK_RIGHT:
					case SDLK_DOWN:
						if(++_current_test >= TEST_COUNT)
						{
							_current_test = 0;
						}
						_test_changed = true;
						break;

					case SDLK_BACKQUOTE:
					case SDLK_WORLD_7:
						st.pushState(new ui::ConsoleState(_data.getConsole()));
						break;

					case SDLK_o:
						m_stream = snd::StreamSptr(new snd::Stream(std::string("test.ogg")));
						break;

					case SDLK_p:
						snd::set_listener(math::vec3f(0.0f, 0.0f, 0.0f),
								math::vec3f(0.0f, 0.0f, -1.0f),
								math::vec3f(0.0f, 1.0f, 0.0f));
						snd::play(_data.getSample(), math::vec3f(0.0f, 0.0f, -20.0f));
						break;

					case SDLK_ESCAPE:
					case SDLK_q:
						_alive = false;
						break;

					default:
						std::cout << "Keyboard press: " << ev.getCode() << std::endl;
						break;
				}
			}
			else
			{
				std::cout << "Keyboard raise: " << ev.getCode() << std::endl;
			}
			return true;
		}

		virtual bool handle(ui::EventMisc &ev, ui::UiStack &st)
		{
			switch(ev.getType())
			{
				case ui::QUIT:
					_alive = false;
					break;

				default:
					std::cout << "Event type: " << ev.getType() << std::endl;
					break;
			}
			return true;
		}	

		virtual bool handle(ui::EventMouseButton &ev, ui::UiStack &st)
		{
			if(ev.isPress())
			{
				std::cout << "Mouse press: " << ev.getCode() << std::endl;
			}
			else
			{
				std::cout << "Mouse raise: " << ev.getCode() << std::endl;
			}
			return true;
		}

		virtual bool handle(ui::EventMouseMotion &ev, ui::UiStack &st)
		{
			return true;
		}

		virtual void reinstate(ui::UiStack &st)
		{
			this->UiState::reinstate(st);
			_test_changed = true;
		}

		virtual void update(gfx::SurfaceScreen &screen, ui::UiStack &st, bool prender)
		{
			if(prender)
			{
				thr::wait_privileged(boost::bind(&Test2D::render, this,
							boost::ref(screen), boost::ref(st)));
			}
		}
		/** \endcond */
};

int main(int argc, char **argv)
{
	int w = 800,
			h = 600,
			b = 32;
	bool fullscreen = false;

	if(argc > 0)
	{
		po::options_description desc("Options");
		desc.add_options()
			("fullscreen,f", "Full-screen mode instead of window.")
			("help,h", "Print help text.")
			("resolution,r", po::value<std::string>(), "Resolution to use.");

		po::variables_map vmap;
		po::store(po::parse_command_line(argc, argv, desc), vmap);
		po::notify(vmap);

		if(vmap.count("help"))
		{
			std::cout << usage << desc << std::endl;
			return 0;
		}
		if(vmap.count("fullscreen"))
		{
			fullscreen = true;
		}
		if(vmap.count("resolution"))
		{
			boost::tie(w, h, b) = gfx::SurfaceScreen::parseResolution(
          vmap["resolution"].as<std::string>());
 		}
	}

	// Initialize.
	thr::thr_init();
	gfx::SurfaceScreen scr(w, h, b, fullscreen);
	snd::snd_init(16);

	// Block.
	{
		ui::UiStack stack(scr, 100);
		TestData data(scr);
		stack.pushState(new Test2D(data));
		boost::thread stack_thread(boost::bind(&ui::UiStack::run, &stack));
		thr::thr_main(1);
		stack_thread.join();
	}
	snd::snd_quit();

	return 0;
}

