#include "gfx/font.hpp"
#include "gfx/entity_camera.hpp"
#include "gfx/entity_object.hpp"
#include "gfx/mesh_ball.hpp"
#include "gfx/surface_screen.hpp"
#include "thr/dispatch.hpp"
#include "ui/ui_stack.hpp"
#include "ui/ui_state.hpp"

#include <boost/program_options.hpp>
#include <boost/exception/diagnostic_information.hpp>

#include <iostream>

using namespace gfx;
using namespace ui;
using namespace math;
namespace po = boost::program_options;
/** \cond */

static const float CAMERA_SPEED = 0.01f,
						 CAMERA_RSPEED = 0.001f;
static SDL_Cursor *cursor_blank = NULL,
									*cursor_default = NULL;
math::vec2i mouse_last(0, 0),
	mouse_diff(0, 0);

/** Console output content. */
static const char *usage = ""
"Usage: test_3d <options>\n"
"Naju graphics engine 2d renderer test program.\n"
"Not for any sensible use.\n"
"\n";

/** \brief Data for 3D-test.
 */
class TestData
{
	private:
		Shader _sh;
		MeshBall _ball;
		Font _fnt;

	public:
		inline const Shader& getShader() const
		{
			return _sh;
		}

		inline const Mesh& getBall() const
		{
			return _ball;
		}

		inline const Font& getFont() const
		{
			return _fnt;
		}

	public:
		TestData() :
			_sh("shader/3d_static.xml"),
			_ball(3, 2),
			_fnt("fnt/default.xml")
		{
			_ball.addTexture("texture", new Texture2D("gfx/textures/texture_stub.png"));
		}
};

/** \brief Graphics test class. */
class Test3D : public UiState
{
	private:
		static const int TEST_COUNT = 1;

	private:
		TestData &_data;
		LightDirectional _light_dir;
		EntityCamera _camera;
		EntityObject _obj_ball;
		int _current_test;
		bool _mv_bk;
		bool _mv_dn;
		bool _mv_fw;
		bool _mv_lt;
		bool _mv_rt;
		bool _mv_up;
		bool _mode_fps;
		bool _mode_wireframe;

	public:
		Test3D(TestData &pdata) :
			_data(pdata),
			_light_dir(0.1f, 0.1f, 0.1f, 0.9f, 0.9f, 0.9f, normalize(math::vec3f(-1.0f, -1.0f, -1.0f))),
			_camera(vec3d(0.0, 0.0, 5.0), vec3d(0.0, 0.0, 0.0)),
			_obj_ball(&(_data.getBall())),
			_current_test(0),
			_mv_bk(false),
			_mv_dn(false),
			_mv_fw(false),
			_mv_lt(false),
			_mv_rt(false),
			_mv_up(false),
			_mode_fps(false),
			_mode_wireframe(false) { }

		virtual ~Test3D() { }

	public:
		/** \brief Render everything.
		 *
		 * @param screen Screen to use.
		 * @param st UI stack.
		 */
		void render(SurfaceScreen &screen, UiStack &st)
		{
			screen.clear(true, true);

			switch(_current_test)
			{
				case 0:
					{
						glEnable(GL_CULL_FACE);
						glDisable(GL_DEPTH_TEST);
						glPolygonMode(GL_FRONT, _mode_wireframe ? GL_LINE : GL_FILL);

						screen.select3D(static_cast<float>(M_PI * 0.25f), 1.0f, 0.01f, 1024.0f);
						const Shader &sh = _data.getShader();
						bind_shader_3d(sh);
						
						mat4f wm(_camera.getWm());
						load_light(_light_dir, wm);

						_obj_ball.loadIdentity();
						_obj_ball.draw(sh, wm);
					}
					break;

				default:
					{
						std::stringstream err;
						err << "invalid test: " << _current_test;
						BOOST_THROW_EXCEPTION(std::runtime_error(err.str()));
					}
					break;
			}

			screen.select2D();
			bind_shader_2d_font();
			if(_mode_fps)
			{
				glPolygonMode(GL_FRONT, GL_FILL);
				std::stringstream fstr;
				fstr << st.getFps();
				std::wstring fps = wstr_utf8(fstr.str());
				const Font &fnt = _data.getFont();
				draw_text(0.052f, 0.048f, 0.05f, fps, fnt, Color(0.0f, 0.0f, 0.0f, 1.0f));
				draw_text(0.05f, 0.05f, 0.05f, fps, fnt, Color(1.0f, 1.0f, 1.0f, 1.0f));

				std::stringstream cstr;
				cstr << Lod::clear_poly_count();
				std::wstring pcnt = wstr_utf8(cstr.str());
				draw_text(0.052f, 0.898f, 0.05f, pcnt, fnt, Color(0.0f, 0.0f, 0.0f, 1.0f));
				draw_text(0.05f, 0.9f, 0.05f, pcnt, fnt, Color(1.0f, 1.0f, 1.0f, 1.0f));
			}

			screen.update();
		}

	public:
		virtual bool handle(EventKey &ev, UiStack &st)
		{
			switch(ev.getCode())
			{
				case SDLK_a:
					_mv_lt = ev.isPress();
					break;

				case SDLK_d:
					_mv_rt = ev.isPress();
					break;

				case SDLK_e:
					_mv_up = ev.isPress();
					break;

				case SDLK_q:
					_mv_dn = ev.isPress();
					break;

				case SDLK_s:
					_mv_bk = ev.isPress();
					break;

				case SDLK_w:
					_mv_fw = ev.isPress();
					break;

				case SDLK_F1:
					if(ev.isPress())
					{
						_mode_fps = !_mode_fps;
					}
					break;

				case SDLK_F2:
					if(ev.isPress())
					{
						_mode_wireframe = !_mode_wireframe;
					}
					break;

				case SDLK_LEFT:
				case SDLK_UP:
					if(ev.isPress() && (--_current_test < 0))
					{
						_current_test = TEST_COUNT - 1;
					}
					break;

				case SDLK_RIGHT:
				case SDLK_DOWN:
					if(ev.isPress() && (++_current_test >= TEST_COUNT))
					{
						_current_test = 0;
					}
					break;

				case SDLK_ESCAPE:
					if(ev.isPress())
					{
						_alive = false;
					}
					break;

				default:
					std::cout << "Keyboard " << (ev.isPress() ? "press" : "raise") <<
						": " << ev.getCode() << std::endl;
					break;
			}
			return true;
		}

		virtual bool handle(EventMisc &ev, UiStack &st)
		{
			switch(ev.getType())
			{
				case QUIT:
					_alive = false;
					break;

				default:
					std::cout << "Event type: " << ev.getType() << std::endl;
					break;
			}
			return true;
		}	

		virtual bool handle(EventMouseButton &ev, UiStack &st)
		{
			switch(ev.getCode())
			{
				case 1:
					if(ev.isPress())
					{
						SDL_WM_GrabInput(SDL_GRAB_ON);
						SDL_SetCursor(cursor_blank);
						mouse_last = ev.getCoord();
					}
					else
					{
						SDL_WM_GrabInput(SDL_GRAB_OFF);
						SDL_SetCursor(cursor_default);
						SDL_EventState(SDL_MOUSEMOTION, SDL_IGNORE);
						SDL_WarpMouse(static_cast<uint16_t>(mouse_last.x()),
								static_cast<uint16_t>(mouse_last.y()));
						SDL_EventState(SDL_MOUSEMOTION, SDL_ENABLE);
					}
					break;

				default:
					std::cout << "Mouse " << (ev.isPress() ? "press" : "raise") <<
						": " << ev.getCode() << std::endl;
			}
			return true;
		}

		virtual bool handle(EventMouseMotion &ev, UiStack &st)
		{
			if(SDL_WM_GrabInput(SDL_GRAB_QUERY) == SDL_GRAB_ON)
			{
				mouse_diff += ev.getDelta();
				SDL_EventState(SDL_MOUSEMOTION, SDL_IGNORE);
				SDL_WarpMouse(static_cast<uint16_t>(mouse_last.x()),
						static_cast<uint16_t>(mouse_last.y()));
				SDL_EventState(SDL_MOUSEMOTION, SDL_ENABLE);
			}
			return true;
		}

		virtual void update(gfx::SurfaceScreen &screen, UiStack &st, bool prender)
		{
			const math::vec3d &crot(_camera.getRot());
			_camera.setRot(crot +
					math::vec3d(static_cast<double>(-mouse_diff.y()) * CAMERA_RSPEED,
						static_cast<double>(-mouse_diff.x()) * CAMERA_RSPEED,
						0.0));
			mouse_diff = math::vec2i(0, 0);

			const math::mat4f &cm(_camera.getWm());
			const math::vec3d &cpos(_camera.getPos()),
						row0(static_cast<double>(cm(0, 0)), static_cast<double>(cm(0, 1)), static_cast<double>(cm(0, 2))),
						row1(static_cast<double>(cm(1, 0)), static_cast<double>(cm(1, 1)), static_cast<double>(cm(1, 2))),
						row2(static_cast<double>(cm(2, 0)), static_cast<double>(cm(2, 1)), static_cast<double>(cm(2, 2)));
			double mbk = _mv_bk ? CAMERA_SPEED : 0.0,
						 mdn = _mv_dn ? CAMERA_SPEED : 0.0,
						 mfw = _mv_fw ? CAMERA_SPEED : 0.0,
						 mlt = _mv_lt ? CAMERA_SPEED : 0.0,
						 mrt = _mv_rt ? CAMERA_SPEED : 0.0,
						 mup = _mv_up ? CAMERA_SPEED : 0.0;
			_camera.setPos(cpos +
					(mrt - mlt) * row0 +
					(mup - mdn) * row1 +
					(mbk - mfw) * row2);

			_camera.loadZXY();

			if(prender)
			{
				thr::wait_privileged(boost::bind(&Test3D::render, this,
							boost::ref(screen), boost::ref(st)));
			}
		}
};

int main(int argc, char **argv)
{
	try
	{
		int w = 800,
				h = 600,
				b = 32;
		bool fullscreen = false;

		if(argc > 0)
		{
			po::options_description desc("Options");
			desc.add_options()
				("fullscreen,f", "Full-screen mode instead of window.")
				("help,h", "Print help text.")
				("resolution,r", po::value<std::string>(), "Resolution to use.");

			po::variables_map vmap;
			po::store(po::parse_command_line(argc, argv, desc), vmap);
			po::notify(vmap);

			if(vmap.count("help"))
			{
				std::cout << usage << desc << std::endl;
				return 0;
			}
			if(vmap.count("fullscreen"))
			{
				fullscreen = true;
			}
			if(vmap.count("resolution"))
			{
				boost::tie(w, h, b) = SurfaceScreen::parseResolution(
						vmap["resolution"].as<std::string>());
			}
		}

		// Initialize.
		thr::thr_init();
		SurfaceScreen scr(w, h, b, fullscreen);
		{
			uint8_t cdata[8] = { 0, 0, 0, 0, 0, 0, 0, 0 };
			cursor_blank = SDL_CreateCursor(cdata, cdata, 8, 1, 0, 0);
			cursor_default = SDL_GetCursor();
		}

		// Block.
		UiStack stack(scr, 100);
		TestData tdata;
		stack.pushState(new Test3D(tdata));
		boost::thread stack_thread(boost::bind(&UiStack::run, &stack));
		thr::thr_main(1);
		stack_thread.join();

		SDL_FreeCursor(cursor_blank);
	}
	catch(const boost::exception & e)
	{
		std::cerr<< boost::diagnostic_information(e) << std::endl;
		return EXIT_FAILURE;
	}
	catch(const std::exception & e)
	{
		std::cerr<<"Exception: "<<e.what()<<std::endl;
		return EXIT_FAILURE;
	}
	catch(...)
	{
		std::cerr<<"Unknown exception caught!"<<std::endl;
		return EXIT_FAILURE;
	}
	return EXIT_SUCCESS;
}
/** \endcond */

