#include "thr/dispatch.hpp"

#include "defaults.hpp"

#include <boost/thread/condition_variable.hpp>

using namespace thr;

/** Thread list. */
std::map<boost::thread::id, ThreadSptr> threads;

/** Task list. */
std::list<Task> tasks_normal;

/** Privileged task list. */
std::list<Task> tasks_privileged;

/** The dispatcher needs to be guarded. */
boost::mutex mut;

/** Sleeping area for normal threads. */
boost::condition_variable cond_normal;

/** Sleeping area for privileged thread. */
boost::condition_variable cond_privileged;

/** Sleeping area for anyone waiting for normal tasks. */
boost::condition_variable cond_wait_normal;

/** Sleeping area for anyone waiting for privileged tasks. */
boost::condition_variable cond_wait_privileged;

/** Number of normal threads sleeping. */
unsigned num_sleeping_normal = 0;

/** Number of privileged threads sleeping. */
unsigned num_sleeping_privileged = 0;

/** Number of waiting threads. */
unsigned num_waiting = 0;

/** Tell if this is quitting. */
bool quitting = false;

/** \brief Inner common implementation of dispatch.
 *
 * Must be called from a locked context.
 *
 * @param pfunctor Task to execute.
 */
static void inner_dispatch(const Task &pfunctor)
{
	tasks_normal.push_back(pfunctor);

	if(0 < num_sleeping_normal)
	{
		cond_normal.notify_one();
		return;
	}
	if(0 < num_sleeping_privileged)
	{
		cond_privileged.notify_one();
	}
}

/** \brief Inner common implementation of dispatch_privileged.
 *
 * @param pfunctor Task to execute.
 * @return Task id created.
 */
static void inner_dispatch_privileged(const Task &pfunctor)
{
	tasks_privileged.push_back(pfunctor);

	if(0 < num_sleeping_privileged)
	{
		cond_privileged.notify_one();
	}
}

/** \brief Wake up anyone waiting for normal tasks to complete.
 *
 * Must be called from a locked context.
 *
 * @param tid Thread id.
 */
static bool wake_normal(boost::thread::id tid)
{
	if(tid != privileged_thread_id)
	{
		if((0 >= tasks_normal.size()) &&
				(num_waiting + num_sleeping_normal + 1 == threads.size()))
		{
			cond_wait_normal.notify_all();
			return true;
		}
	}
	else
	{
		if((0 >= tasks_normal.size()) &&
				(num_waiting + num_sleeping_normal == threads.size()))
		{
			cond_wait_normal.notify_all();
			return true;
		}
	}
	return false;
}

/** \brief Wake up anyone waiting for privileged tasks to complete.
 *
 * Must be called from a locked context.
 *
 * Must be only called by the privileged executor thread.
 */
static void wake_privileged()
{
	if(0 >= tasks_privileged.size())
	{
		cond_wait_privileged.notify_all();
	}
}

/** \brief Inner task running, normal tasks.
 *
 * @param tlist Task list.
 * @param scope Previously created scoped lock.
 * @param tid Id of this thread.
 * @return True if executed something, false if not.
 */
static bool inner_run_normal(boost::mutex::scoped_lock &scope,
		boost::thread::id tid)
{
	if(tasks_normal.empty())
	{
		return false;
	}
	Task tk = tasks_normal.front();
	tasks_normal.pop_front();
	scope.unlock();
	tk();
	scope.lock();
	wake_normal(tid);
	return true;
}

/** \brief Inner task running, privileged tasks.
 *
 * @param tlist Task list.
 * @param scope Previously created scoped lock.
 * @return True if executed something, false if not.
 */
static bool inner_run_privileged(boost::mutex::scoped_lock &scope)
{
	if(tasks_privileged.empty())
	{
		return false;
	}
	Task tk = tasks_privileged.front();
	tasks_privileged.pop_front();
	scope.unlock();
	tk();
	scope.lock();
	wake_privileged();
	return true;
}

/** \brief Clean up privileged jobs if we're a privileged thread.
 *
 * Ensures that a privileged job can't queue up other privileged jobs that
 * would potentially create deadlocks.
 *
 * @param pfunctor Task to execute.
 * @param scope Previously created scoped lock.
 * @param tid Id of this thread.
 * @return True if cleaned up the queue, false if not.
 */
static bool cleanup_privileged(const Task &pfunctor, boost::mutex::scoped_lock &scope, boost::thread::id tid)
{
	if(tid == privileged_thread_id)
	{
		while(inner_run_privileged(scope));
		scope.unlock();
		pfunctor();
		scope.lock();
		return true;
	}
	return false;
}

/** \brief Run in this dispatcher.
 *
 * Execute in this dispatcher, waiting for normal jobs and executing them
 * whenever available.
 */
static void run_normal()
{
	boost::thread::id tid = boost::this_thread::get_id();
	boost::mutex::scoped_lock scope(mut);
	--num_sleeping_normal;

	while(!quitting)
	{
		if(inner_run_normal(scope, tid))
		{
			continue;
		}
		++num_sleeping_normal;
		cond_normal.wait(scope);
		--num_sleeping_normal;
	}
}

void thr::dispatch_ext(const Task &pfunctor)
{
	boost::mutex::scoped_lock scope(mut);

	inner_dispatch(pfunctor);
}

void thr::dispatch_privileged_ext(const Task &pfunctor)
{
	boost::thread::id tid = boost::this_thread::get_id();
	boost::mutex::scoped_lock scope(mut);

	if(cleanup_privileged(pfunctor, scope, tid))
	{
		return;
	}

	inner_dispatch_privileged(pfunctor);
}

void thr::thr_main(unsigned nthreads)
{
	if(!is_primary_thread())
	{
		std::stringstream sstr;
		sstr << "trying to enter main loop from a non-privileged thread";
		BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str()));
	}

	if(0 >= nthreads)
	{
		std::stringstream sstr;
		sstr << "thread count autodetect not implemented yet";
		BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str()));
	}

	// Lock this scope only.
	{
		boost::mutex::scoped_lock scope(mut);

		// Last slot is reserved for callee.
		for(unsigned ii = 0; (ii < nthreads); ++ii)
		{
			ThreadSptr ts(new boost::thread(run_normal));
			threads[ts->get_id()] = ts;
			++num_sleeping_normal;
		}

		while(!quitting)
		{
			if(inner_run_privileged(scope))
			{
				continue;
			}
			if(inner_run_normal(scope, privileged_thread_id))
			{
				continue;
			}
			++num_sleeping_privileged;
			cond_privileged.wait(scope);
			--num_sleeping_privileged;
		}
	}

	// FOREACH does not work due to MSVC.
	{
		std::map<boost::thread::id, ThreadSptr>::iterator ii;
		std::map<boost::thread::id, ThreadSptr>::iterator ee;
		for(ii = threads.begin(), ee = threads.end(); (ii != ee); ++ii)
		{
			(*ii).second->join();
		}
	}
	threads.clear();
}

void thr::thr_quit()
{
	boost::mutex::scoped_lock scope(mut);

	quitting = true;
	cond_normal.notify_all();
	cond_privileged.notify_all();
	cond_wait_normal.notify_all();
	cond_wait_privileged.notify_all();
	tasks_normal.clear();
	tasks_privileged.clear();
}

void thr::wait()
{
	boost::thread::id tid = boost::this_thread::get_id();
	boost::mutex::scoped_lock scope(mut);

	if(threads.end() != threads.find(tid))
	{
		while(inner_run_normal(scope, tid));
		if(wake_normal(tid))
		{
			return;
		}
		++num_waiting;
		cond_wait_normal.wait(scope);
		--num_waiting;
		return;
	}
	else if(privileged_thread_id == tid)
	{
		while(0 < tasks_privileged.size() + tasks_normal.size())
		{
			if(inner_run_privileged(scope))
			{
				continue;
			}
			inner_run_normal(scope, tid);
		}
		return;
	}

	if((0 >= tasks_normal.size()) &&
			(num_sleeping_normal + num_waiting == threads.size()))
	{
		return;
	}
	cond_wait_normal.wait(scope);
}

void thr::wait_privileged_ext(const Task &pfunctor)
{
	boost::thread::id tid = boost::this_thread::get_id();
	boost::mutex::scoped_lock scope(mut);

	if(cleanup_privileged(pfunctor, scope, tid))
	{
		return;
	}

	inner_dispatch_privileged(pfunctor);;
	cond_wait_privileged.wait(scope);
}

