#include "ui/console.hpp"

using namespace ui;

static const float AREA_MARGIN = 0.6f;
static const float AREA_MARGIN_BOTTOM = AREA_MARGIN * 1.4f;
static const float AREA_BORDER = 0.18f;

Console::Console(const gfx::Font &fnt, float fs,
		const math::rect2f &prestrict, const char *pfname) :
	_font(fnt),
	_font_size(fs),
	_area(prestrict),
	_progress(0),
	_progress_max(0)
{
	this->setDefaultColors();
	this->updateArea();

	if(NULL != pfname)
	{
		_background.load(pfname);
	}
}

Console::Console(const gfx::Font &fnt, float fs,
		const gfx::SurfaceScreen &screen, const char *pfname) :
	_font(fnt),
	_font_size(fs),
	_area(screen.getArea()),
	_progress(0),
	_progress_max(0)
{
	this->setDefaultColors();
	this->updateArea();

	if(NULL != pfname)
	{
		_background.load(pfname);
	}
}

void Console::addRow(const std::wstring &op)
{
	TextRect row(math::vec2f(_area_text.w(), _area_text.h()));
	std::wstring remaining = row.setContent(op, _font, _font_size);
	if(0 < remaining.length())
	{
		// TODO: ?
	}

	boost::mutex::scoped_lock scope(_mut);

	_lines.push_front(row);

	float total_h = 0.0f;

	for(std::list<TextRect>::iterator ii = _lines.begin(), ee = _lines.end();
			(ii != ee);)
	{
		std::list<TextRect>::iterator jj = ii;
		++ii;

		if(total_h > _area_text.h())
		{
			_lines.erase(jj);
			continue;
		}
		total_h += jj->getDimensions().y();
	}
}

void Console::execute()
{
	this->addRow(m_input.getLine());
	m_input.reset();
}

void Console::render(gfx::SurfaceScreen &screen)
{
	screen.select2D();

	if(0 != _background.id())
	{
		gfx::bind_shader_2d_texture();
		gfx::draw_rect_textured_fill(_area, m_color_background, _background);
	}

	m_input.draw(m_color_input_bottom, m_color_input_top, _font_size, _font);

	gfx::draw_fill(0, m_color_text_bottom);
	gfx::draw_fill(1, m_color_text_bottom);
	gfx::draw_fill(2, m_color_text_top);
	gfx::draw_fill(3, m_color_text_top);
	this->renderText(screen);

	this->renderProgress(screen);
}

void Console::renderProgress(const gfx::SurfaceScreen &screen) const
{
	math::rect2i parea = screen.toPixelArea(_area_bar);
	int bw = math::lround(_font_size * AREA_BORDER *
			static_cast<float>(math::max(parea.w(), parea.h())));

	parea.x1() += bw;
	parea.y1() += bw;
	parea.w() -= bw * 2;
	parea.h() -= bw;

	float ppos = static_cast<float>(_progress) / static_cast<float>(_progress_max);
	int pw = math::lround(static_cast<float>(parea.w()) * ppos);

	gfx::bind_shader_2d();

	gfx::draw_rect(parea.x1(), parea.y1(), pw + 1, parea.h() + 1,
			m_color_progress_fill);

	gfx::draw_rect(parea.x1() + pw, parea.y1(), parea.w() - pw + 1, parea.h() + 1,
			m_color_progress_empty);

	gfx::draw_rect(parea.x1(), parea.y1() + parea.h(), parea.w() + 1, bw + 1,
			m_color_border);
}

void Console::renderText(const gfx::SurfaceScreen &screen)
{
	math::rect2i parea = screen.toPixelArea(_area);
	int bw = math::lround(_font_size * AREA_BORDER *
			static_cast<float>(math::max(parea.w(), parea.h())));

	gfx::bind_shader_2d_font();
	{
		boost::mutex::scoped_lock scope(_mut);

		float cy = _area_text.y1();
		BOOST_FOREACH(const TextRect &vv, _lines)
		{
			math::rect2f carea(_area_text.x1(), cy, vv.getDimensions());
			vv.renderText(carea, _font, _font_size);
			cy += vv.getDimensions().y();
		}
	}

	gfx::bind_shader_2d();
	// OpenGL rounding system requires +1 here.
	gfx::draw_rect_contour(parea.x1(), parea.y1(),
			parea.w() + 1, parea.h() + 1, bw, m_color_border);
}

void Console::setDefaultColors()
{
	m_color_background = gfx::Color(0.3f, 0.3f, 0.3f, 1.0f);
	m_color_border = gfx::Color(0.4f, 0.8f, 0.5f, 0.3f);
	m_color_progress_empty = gfx::Color(0.2f, 0.2f, 0.2f, 0.4f);
	m_color_progress_fill = gfx::Color("e09d1aa0");
	m_color_input_bottom = gfx::Color(0.8f, 0.9f, 0.9f, 0.8f);
	m_color_input_top = gfx::Color(0.8f, 0.9f, 0.9f, 0.8f);
	m_color_text_bottom = gfx::Color(0.9f, 0.8f, 0.8f, 0.8f);
	m_color_text_top = gfx::Color(0.9f, 0.8f, 0.8f, 0.8f);
}

void Console::updateArea()
{
	_area_bar = math::rect2f(_area.x1(), _area.y1(), _area.w(), _font_size * 2.0f);

	math::rect2f area_input(_area.x1() + _font_size * AREA_MARGIN,
			_area_bar.y2() + _font_size * AREA_MARGIN_BOTTOM,
			_area.w() - _font_size * AREA_MARGIN * 2.0f,
			_font_size);
	m_input.setArea(area_input);

	_area_text = math::rect2f(area_input.x1(),
			area_input.y2(),
			area_input.w(),
			_area.y2() - area_input.y2() - _font_size * AREA_MARGIN);
}

