#ifndef UI_EVENT_HPP
#define UI_EVENT_HPP

#include "math/vec.hpp"

namespace ui
{
	/** Event type. */
	enum EventType
	{
		/** Key event. */
		KEY,

		/** Miscellaneous event. */
		MISC,

		/** Mouse button event. */
		MOUSE_BUTTON,

		/** Mouse motion event. */
		MOUSE_MOTION,

		/** Quit event. */
		QUIT
	};

	/** \brief Generic event class.
	 *
	 * Actually contains all the required elements for all the events. The
	 * subsequent events implement a specific type for virtual method call
	 * reasons.
	 */
	struct Event
	{
		/** Coordinates (if applicable). */
		math::vec2i _coord;

		/** Delta coordinates (if applicable). */
		math::vec2i _delta;

		/** Type of this event. */
		EventType _type;

		/** Button, keycode or special code (depends on the event). */
		int _code;

		/** Is this a press (instead of a raise)? */
		bool _press;

		/** \brief Constructor.
		 *
		 * @param ptype Event type.
		 */
		inline Event(EventType ptype)
		{
			_type = ptype;
		}

		/** \brief Accessor.
		 *
		 * @return Event code of this.
		 */
		inline int getCode() const
		{
			return _code;
		}

		/** \brief Get the coordinates of this.
		 *
		 * @return Vector reference.
		 */
		inline const math::vec2i& getCoord() const
		{
			return _coord;
		}

		/** \brief Get the coordinates of this.
		 *
		 * @return Vector reference.
		 */
		inline const math::vec2i& getDelta() const
		{
			return _delta;
		}

		/** \brief Accessor.
		 *
		 * @return Event type of this.
		 */
		inline EventType getType() const
		{
			return _type;
		}

		/** \brief Tell if this is a press event.
		 * 
		 * Note that this might not apply unless this is a key or button event.
		 *
		 * @return True if yes, false if no.
		 */
		inline bool isPress() const
		{
			return _press;
		}

		/** \brief Tell if this is a raise event.
		 * 
		 * Note that this might not apply unless this is a key or button event.
		 *
		 * @return True if yes, false if no.
		 */
		inline bool isRaise() const
		{
			return !_press;
		}
	};
}

#endif
