#include "ui/fps_counter.hpp"

#include "thr/generic.hpp"

#include <sstream>

using namespace ui;

FpsCounter::FpsCounter(int maxrate) :
	_frame_counter(0),
	_last_ticks(thr::usec_get_timestamp()),
	_excess_ticks(0.0)
{
	this->assignFramerates(maxrate, std::max(maxrate / 4, 1));
}

FpsCounter::FpsCounter(int maxrate, int minrate) :
	_frame_counter(0),
	_last_ticks(thr::usec_get_timestamp()),
	_excess_ticks(0.0)
{
	this->assignFramerates(maxrate, minrate);
}

void FpsCounter::appendFrame(int64_t stamp)
{
	_frame_list.push_front(stamp);

	while(_frame_list.back() <= stamp - 1000000)
	{
		_frame_list.pop_back();
	}
}

void FpsCounter::assignFramerates(int maxrate, int minrate)
{
	if(maxrate <= 0)
	{
		std::ostringstream str;
		str << "invalid maximum framerate " << maxrate << " maxrate";
		BOOST_THROW_EXCEPTION(std::runtime_error(str.str()));
	}
	if(minrate > maxrate)
	{
		std::ostringstream str;
		str << "minimum framerate can not be larger than maximum framerate " <<
			minrate << " minrate";
		BOOST_THROW_EXCEPTION(std::runtime_error(str.str()));
	}
	if(minrate <= 0)
	{
		std::ostringstream str;
		str << "invalid minimum framerate " << minrate << " minrate";
		BOOST_THROW_EXCEPTION(std::runtime_error(str.str()));
	}

	_desired_framerate = maxrate;
	_minimum_framerate = minrate;
	_tick_us = 1000000.0 / (double)maxrate;
}

int FpsCounter::check(bool allow_suspend)
{
	int64_t current_ticks = this->getCurrentTicks();

	if(allow_suspend && (_excess_ticks < 0.0))
	{
		thr::usec_sleep(static_cast<uint64_t>(-_excess_ticks));
		current_ticks = this->getCurrentTicks();
	}

	// Below minimum FPS must draw in any case
	if(this->getCurrentFps() <= _minimum_framerate)
	{
		++_frame_counter;
		// Must modify excess tick numeber to prevent later delay rush.
		this->appendFrame(current_ticks);
		_excess_ticks = _tick_us;
		return 2;
	}

	// Fallback to check with normal methods.
	if(_excess_ticks >= 0.0)
	{
		++_frame_counter;
		_excess_ticks -= _tick_us;
		//std::cout << _excess_ticks << '\n';
		if(_excess_ticks > 0.0)
		{
			return 1;
		}
		this->appendFrame(current_ticks);
		return 2;
	}
	return 0;
}

int64_t FpsCounter::getCurrentTicks()
{
	int64_t ret = thr::usec_get_timestamp();
	_excess_ticks += static_cast<double>(ret - _last_ticks);
	_last_ticks = ret;
	return ret;
}

void FpsCounter::reset()
{
	this->getCurrentTicks();

	_frame_list.clear();
	_excess_ticks = 0.0;
	_frame_counter = 0;
}

