#ifndef UI_FPS_COUNTER_HPP
#define UI_FPS_COUNTER_HPP

#include "ui/generic.hpp"

#include <list>

namespace ui
{
	/** \brier Class to keep track of the current framerate<
	 *
	 * TODO: Should this be adjusted to work with something else than SDL?
	 */
	class FpsCounter
	{
		private:
			/** Frame counter of frames (supposed to be) rendered. */
			int64_t _frame_counter;

			/** Ticks when last checked. */
			int64_t _last_ticks;;

			/** \brief The desired framerate.
			 *
			 * Limited to integer values for clarity of framerate calculation.
			 */
			int _desired_framerate;

			/** \brief Minimum framerate.
			 *
			 * The framerate draw request calculation will not go below this number,
			 * instead causing slowdown.
			 */
			int _minimum_framerate;

			/** Microseconds per tick. */
			double _tick_us;

			/** Excess ticks to use. */
			double _excess_ticks;

			/** Listing of latest rendered frames to keep track of the framerate. */
			std::list<int64_t> _frame_list;

		public:
		  /** \brief Default constructor.
			 *
			 * Per default behavior, the minimum (slowdown) framerate is specified to
			 * one fourth of the maximum framerate.
			 * 
			 * @param maxrate Maximum framerate.
			 */
			FpsCounter(int maxrate);

			/** \brief Alternate constructor.
			 *
			 * @param maxrate Maximum framerate.
			 * @param minrate Maximum framerate.
			 */
			FpsCounter(int maxrate, int minrate);

		private:
			/** \brief Append a frame.
			 *
			 * @param stamp Current timestamp.
			 */
			void appendFrame(int64_t stamp);

			/** \brief Assign given framerates and throw an exception on error.
			 *
			 * @param maxrate Maximum framerate.
			 * @param minrate Minimum framerate.
			 */
			void assignFramerates(int maxrate, int minrate);

			/** \brief Get the current ticks in a wrap-safe manner.
			 *
			 * @return Tick count as long.
			 */
			int64_t getCurrentTicks();

		public:
			/** \brief Check if a frame should be drawn.
			 *
			 * If there is time and it is allowed, this method will suspend.
			 *
			 * The return value of 0 is system inaccuracies, and perfectly reasonable.
			 * 
			 * @param allow_suspend True to allow sleeping.
			 * @return 0: do nothing, 1: essentials, 2: everything.
			 */
			int check(bool allow_suspend);

			/** \brief Reset the frame calculation.
			 *
			 * Should be done after a long period of inactivity in framerate
			 * calculation to prevent catch-up behavior.
			 */
			void reset();

		public:
			/** \brief Get the current framerate.
			 *
			 * @return Current framerate as integer.
			 */
			inline int getCurrentFps() const
			{
				return static_cast<int>(_frame_list.size());
			}

			/** \brief Returns the number of frames this has allowed to be rendered.
			 *
			 * @return Frame count.
			 */
			inline int64_t getFrameCount() const
			{
				return _frame_counter;
			}

			/** \brief Get the target framerate.
			 *
			 * @return Target fps.
			 */
			inline int getTargetFps() const
			{
				return _desired_framerate;
			}
	};
}

#endif
