#ifndef UI_TEXT_AREA_HPP
#define UI_TEXT_AREA_HPP

#include "gfx/color.hpp"
#include "ui/generic.hpp"
#include "ui/text_rect.hpp"

namespace ui
{
	class TextWord;

	/** \brief Portrays a text box.
	*/
	class TextArea :
		public TextRect
	{
		protected:
			/** Restrict area. */
			math::rect2f _boundary;

			/** Margin value. */
			float _margin_left;

			/** Margin value. */
			float _margin_right;

			/** Margin value. */
			float _margin_top;

			/** Margin value. */
			float _margin_bottom;

			/** General font size of this. */
			float _font_size;

			/** Font used in this. */
			const gfx::Font &_font;

			/** Justification mode. */
			TextGravity _gravity;

		public:
			/** \brief Constructor.
			 *
			 * @param prestrict Restrictions.
			 * @param fs Font size for this text box.
			 * @param fnt Font to use.
			 * @param pjust Justification schema.
			 */
			TextArea(const math::rect2f &prestrict, float fs, const gfx::Font &fnt,
					TextGravity grav = GRAVITY_CENTER);

			/** \brief Destructor.
			 */
			virtual ~TextArea() { }

		public:
			/** \brief Calculate the borders of the encapsulated text content.
			 *
			 * Uses the internal gravity variable to determine the area.
			 * 
			 * @return Size rectangle including position.
			 */
			math::rect2f getContentArea() const;

		protected:
			/** \brief Render text in this.
			 *
			 * Fill color must have been set beforehand.
			 *
			 * @param col Color to render with.
			 * @param carea Content area calculated beforehand.
			 */
			void renderText(const math::rect2f &carea) const;

			/** \brief Wrapper for text render.
			 *
			 * Calculates the content size by itself.
			 *
			 * @param col Color to render with.
			 */
			void renderText() const;

		private:
			/** \brief Update the text rectangle boundary.
			 *
			 * Takes the margins into account to update the boudnary.
			 */
			void updateBoundary();

		public:
			/** \brief Render the text area.
			 *
			 * The default render is intentionally crude. It can be used as an
			 * example to create better-looking renders (overwriting this method or
			 * not).
			 *
			 * @param tcol Text color.
			 * @param bcol Border color.
			 */
			virtual void render(const gfx::Color &tcol, const gfx::Color &bcol) const;

		public:
			/** \brief Get the font.
			 *
			 * @return Font in this.
			 */
			inline const gfx::Font& getFont() const
			{
				return _font;
			}

			/** \brief Get the font size.
			 *
			 * @return Font size.
			 */
			inline float getFontSize() const
			{
				return _font_size;
			}

			/** \brief Get justification mode.
			 *
			 * @return Justification mode.
			 */
			inline TextGravity getGravity() const
			{
				return _gravity;
			}

			/** \brief Margin accessor.
			 *
			 * @return Margin value.
			 */
			inline float getMarginBottom() const
			{
				return _margin_left;
			}

			/** \brief Margin accessor.
			 *
			 * @return Margin value.
			 */
			inline float getMarginLeft() const
			{
				return _margin_left;
			}

			/** \brief Margin accessor.
			 *
			 * @return Margin value.
			 */
			inline float getMarginRight() const
			{
				return _margin_right;
			}

			/** \brief Margin accessor.
			 *
			 * @return Margin value.
			 */
			inline float getMarginTop() const
			{
				return _margin_left;
			}

			/** \brief Get a text line.
			 *
			 * NOT checked. Use hasLine() to check.
			 *
			 * @param idx Line index.
			 * @return Line at index.
			 */
			inline TextRow& getLine(unsigned idx)
			{
				return _content[idx];
			}

			/** \brief Get the position.
			 *
			 * @return Position vector.
			 */
			inline const math::vec2f& getPosition() const
			{
				return _boundary.pos();
			}

			/** \brief Get the size.
			 *
			 * @return Size vector.
			 */
			inline const math::vec2f& getSize() const
			{
				return _boundary.size();
			}

			/** \brief Tell if line exists in here.
			 *
			 * @param idx Line index.
			 * @return True if yes, false if no.
			 */
			inline bool hasLine(unsigned idx)
			{
				return (idx < _content.size());
			}

			/** \brief Move this.
			 *
			 * Alters the restriction rectangle to move this.
			 *
			 * Note that the move anchor is always the lower left corner of the
			 * textarea.
			 *
			 * @param mvec New position.
			 */
			inline void move(const math::vec2f &mvec)
			{
				_boundary = math::rect2f(mvec, _boundary.size());
			}

			/** \brief Move this.
			 *
			 * Alters the restriction rectangle to move this.
			 *
			 * Note that the move anchor is always the lower left corner of the
			 * textarea.
			 *
			 * @param px New x coordinate.
			 * @param py New y coordinate.
			 */
			inline void move(float px, float py)
			{
				this->move(math::vec2f(px, py));
			}

			/** \brief Set the text content of this.
			 *
			 * @param op String to set in.
			 * @return Rest of the string that didn't fit.
			 */
			inline std::wstring setContent(const std::wstring &op)
			{
				return this->TextRect::setContent(op, _font, _font_size);
			}

			/** \brief Set bottom margin.
			 *
			 * Note: does not update the boundary.
			 *
			 * @param op New margin.
			 */
			inline void setMarginBottom(float op)
			{
				_margin_bottom = op * _font_size;
			}

			/** \brief Set left margin.
			 *
			 * Note: does not update the boundary.
			 *
			 * @param op New margin.
			 */
			inline void setMarginLeft(float op)
			{
				_margin_left = op * _font_size;
			}

			/** \brief Set right margin.
			 *
			 * Note: does not update the boundary.
			 *
			 * @param op New margin.
			 */
			inline void setMarginRight(float op)
			{
				_margin_right = op * _font_size;
			}

			/** \brief Set top margin.
			 *
			 * Note: does not update the boundary.
			 *
			 * @param op New margin.
			 */
			inline void setMarginTop(float op)
			{
				_margin_top = op * _font_size;
			}

			/** \brief Set margins.
			 *
			 * Order: left right bottom top.
			 *
			 * @param pleft Left margin.
			 * @param pright Right margin.
			 * @param pbottom Bottom margin.
			 * @param ptop Top margin.
			 */
			inline void setMargins(float pleft, float pright, float pbottom,
					float ptop)
			{
				this->setMarginLeft(pleft);
				this->setMarginRight(pright);
				this->setMarginBottom(pbottom);
				this->setMarginTop(ptop);
				this->updateBoundary();
			}
	};
}

#endif
