#include "ui/text_rect.hpp"

#include "gfx/surface.hpp"
#include "ui/text_word.hpp"

using namespace ui;

TextRect::TextRect(const math::vec2f &plimits) :
	_limits(plimits),
	_dimensions(0.0f, 0.0f) { }

bool TextRect::add(const TextRow &op, float fs)
{
	float new_height = static_cast<float>(_content.size() + 1) * fs;

	if(new_height > _limits.y())
	{
		return false;
	}

	if(_dimensions.x() < op.getWidth())
	{
		_dimensions.x() = op.getWidth();
	}
	_dimensions.y() = new_height;
	_content.push_back(op);
	return true;
}

void TextRect::clear()
{
	_dimensions = math::vec2f(0.0f, 0.0f);
	_content.clear();
}

void TextRect::renderText(const math::rect2f &carea, const gfx::Font &font,
		float font_size, gfx::TextJustify justify) const
{
	float cy = carea.y2() - font_size;
	switch(justify)
	{
		case gfx::LEFT:
			BOOST_FOREACH(const TextRow &vv, _content)
			{
				gfx::draw_text_line(carea.x1(), cy,
						font_size, vv.getContent(), font);
				cy -= font_size;
			}
			return;

		case gfx::RIGHT:
			BOOST_FOREACH(const TextRow &vv, _content)
			{
				float linewid = vv.calcWidth(font, font_size);
				gfx::draw_text_line(carea.x1() + (carea.w() - linewid), cy,
						font_size, vv.getContent(), font);
				cy -= font_size;
			}
			return;

		case gfx::CENTER:
		default:
			BOOST_FOREACH(const TextRow &vv, _content)
			{
				float linewid = vv.calcWidth(font, font_size);
				gfx::draw_text_line(carea.x1() + (carea.w() - linewid) * 0.5f, cy,
						font_size, vv.getContent(), font);
				cy -= font_size;
			}
			return;
	}
}

std::wstring TextRect::setContent(const std::wstring &str, const gfx::Font &fnt,
		float fs)
{
	TextRow row;
	TextWord word;
	wchar_t last_space = 0;
	float last_space_width = 0.0f;
	int ret_idx = 0,
			last_inc = 0;

	for(unsigned ii = 0; (ii <= str.length()); ++ii)
	{
		// Newline is the same as zero here, it also forces commit.
		wchar_t cc = 0;
		float cwid = 0.0f;
		if(ii < str.length())
		{
			cc = str[ii];
			if(is_newline(cc))
			{
				cc = 0;
			}
			else
			{
				cwid = fnt.getGlyph(cc).getAdvanceX() * fs;
			}
		}

		// Whitespace detected, must commit.
		if(is_whitespace(cc))
		{
			// If no word to commit, do nothing.
			if(word.isEmpty())
			{
				last_space = cc;
				last_space_width = cwid;
			}
			// If appending to row succeeds, it seems all good.
			else if(row.add(word, _limits.x(), last_space, last_space_width))
			{
				last_space = cc;
				last_space_width = cwid;
				last_inc = ii;
				word.clear();
			}
			// If can't add to row, must create a new row. If can't create a new
			// row, abort.
			else if(!this->add(row, fs))
			{
				return str.substr(ret_idx);
			}
			// Could add new row.
			else
			{
				row.clear();
				// If can't add word into new row, abort.
				if(!row.add(word, _limits.x(), 0, 0.0f))
				{
					return str.substr(ret_idx);
				}
				last_space = cc;
				last_space_width = cwid;
				ret_idx = last_inc;
				word.clear();
			}

			// If newline, must additionally commit. Word is always empty now.
			if(0 == cc)
			{
				// If can't create a new row, exit.
				if(!this->add(row, fs))
				{
					return str.substr(ret_idx);
				}
				row.clear();
				last_space = 0;
				last_space_width = 0.0f;
				ret_idx = last_inc = ii;
			}
		}
		// Not whitespace, append to row.
		else
		{
			word.add(cc, cwid);
		}
	}

	return std::wstring();
}

