#ifndef UI_TEXT_ROW_HPP
#define UI_TEXT_ROW_HPP

#include "gfx/font.hpp"

#include <sstream>

namespace ui
{
	class TextWord;

	/** Represents one text row within a textarea.
	*/
	class TextRow
	{
		private:
			/** Width using the parameters specified in the addition. */
			float _width;

			/** String builder, to be unreferenced when ready. */
			std::wstringstream _builder;

		public:
			/** \brief Constructor.
			 */
			inline TextRow() :
				_width(0.0f) { }

			/** \brief Copy constructor.
			 *
			 * @param op Source.
			 */
			inline TextRow(const TextRow &op) :
				_width(op._width)
			{
				_builder.str(op._builder.str());
			}

		public:
			/** \brief Add one character to this row.
			 *
			 * @param cc Character to add.
			 * @param mwid Maximum width.
			 * @param cwid Character width.
			 * @return Success value.
			 */
			bool add(wchar_t cc, float mwid, float cwid);

			/** \brief Add one word to this row.
			 *
			 * @param word Word to add.
			 * @param mwid Maximum width.
			 * @param schar Space character.
			 * @param swid Space width.
			 * @return Success value.
			 */
			bool add(const TextWord &word, float mwid, wchar_t schar, float swid);

			/** \brief Get the width of this row.
			 *
			 * @param fnt Font used.
			 * @param font size used.
			 * @return Width of this row of text.
			 */
			float calcWidth(const gfx::Font &fnt, float font_size) const;

		public:
			/** \brief Clear all content of this row.
			 */
			inline void clear()
			{
				_width = 0.0f;
				_builder.str(std::wstring());
			}

			/** \brief Accessor.
			 *
			 * @return content.
			 */
			inline std::wstring getContent() const
			{
				return _builder.str();
			}

			/** Accessor.
			 *
			 * @return Width.
			 */
			inline float getWidth() const
			{
				return _width;
			}

		public:
			/** \brief Assignment operator.
			 *
			 * @param op Source.
			 */
			inline const TextRow& operator=(const TextRow &op)
			{
				_width = op._width;
				_builder.str(op._builder.str());
				return *this;
			}
	};
}

#endif
