#ifndef UI_UI_STATE_HPP
#define UI_UI_STATE_HPP

#include "ui/event_key.hpp"
#include "ui/event_misc.hpp"
#include "ui/event_mouse_button.hpp"
#include "ui/event_mouse_motion.hpp"

namespace gfx
{
	class SurfaceScreen;
}

namespace ui
{
	class UiStack;

	/** \brief UI state class.
	 *
	 * Implements behavior in a certain UI state.
	 */
	class UiState
	{
		protected:
			/** Is this alive? */
			bool _alive;

		public:
			/** \brief Constructor. */
			UiState();

			/** \brief Destructor. */
			virtual ~UiState() { }

		public:
			/** \brief Virtual method for keyboard events.
			 *
			 * @param ev Event struct.
			 * @param st UI stack reference.
			 * @return True if handled, false if not.
			 */
			virtual bool handle(EventKey &ev, UiStack &st);

			/** \brief Virtual method for misc events.
			 *
			 * @param ev Event struct.
			 * @param st UI stack reference.
			 * @return True if handled, false if not.
			 */
			virtual bool handle(EventMisc &ev, UiStack &st);

			/** \brief Virtual method for mouse button events.
			 *
			 * @param ev Event struct.
			 * @param st UI stack reference.
			 * @return True if handled, false if not.
			 */
			virtual bool handle(EventMouseButton &ev, UiStack &st);

			/** \brief Virtual method for mouse motion events.
			 *
			 * @param ev Event struct.
			 * @param st UI stack reference.
			 * @return True if handled, false if not.
			 */
			virtual bool handle(EventMouseMotion &ev, UiStack &st);

			/** \brief Called whenever this UI state gets focus.
			 *
			 * Called when the UI state gets focus for the first time after being
			 * out of focus for some time. This means both getting initialized and
			 * being rerevealed from upper UI states.
			 *
			 * For practical purposes, the number of times this method gets called
			 * should have no effect.
			 *
			 * The default implementation simply turns key repeat off. For the most
			 * part, users will probably not want to change this.
			 */
			virtual void reinstate(UiStack &st);

			/** \brief Wrapper for update.
			 *
			 * If a state decides to stop executing, it should turn alive flag to
			 * false (or call die) before returning from an implementation of this
			 * method.
			 *
			 * @param screen Screen to render to.
			 * @param st UI stack reference.
			 * @param prender True if should render too, not just update state.
			 * @return False if dead, otherwise true.
			 */
			virtual void update(gfx::SurfaceScreen &screen, UiStack &st,
					bool prender) = 0;

		public:
			/** \brief Tell if this is alive.
			 *
			 * @return True if yes, false if no.
			 */
			inline bool isAlive() const
			{
				return _alive;
			}

			/** \brief Turn alive flag off.
			 */
			inline void die()
			{
				_alive = false;
			}
	};
}

#endif
