// < OBJECTS >

// < 3D >
/**
* 3D Vector object for x,y,z coordinates
**/
function Vector3D(x,y,z){
	this.x = x;
	this.y = y;
	this.z = z;
}

/**
* 3D line containing start and end point of the line
**/
function Line3D(coordinate, dcoordinate){
	this.coordinate = coordinate;
	this.dcoordinate = dcoordinate;
}

/**
* Object for 3D objects such as cubes
* lines = array of 3D lines in object
* cam = Vector3D containing camera location
* angle = Vector3D containing angle of object
* piv = Vector3D containing pivot point
* position = Vector3D containing poisition
* type = type of object as string, for example "cube" or "pyramid"
**/
function Object3D(cam,angle,piv,position,size,scale,type){
	this.lines = new Array();
	this.cam = cam;
	this.angle = angle;
	this.piv = piv;
	this.size = size;
	this.position = position;
	this.scale = scale;
	if (type == "cube"){
		this.make_cube();
	}
	if (type == "pyramid"){
		this.make_pyramid();
	}
	if (type == "hyper"){
		this.make_hyper();
	}
	this.lines2D = new Array();
	this.calc_lines2D();
}

Object3D.prototype.make_cube = function(){
	this.lines[0] = new Line3D(new Vector3D(-this.size,-this.size,this.size), new Vector3D(this.size,-this.size,this.size));
	this.lines[1] = new Line3D(new Vector3D(-this.size,this.size,this.size), new Vector3D(this.size,this.size,this.size));
	this.lines[2] = new Line3D(new Vector3D(-this.size,-this.size,-this.size), new Vector3D(this.size,-this.size,-this.size));
	this.lines[3] = new Line3D(new Vector3D(-this.size,this.size,-this.size), new Vector3D(this.size,this.size,-this.size));
	this.lines[4] = new Line3D(new Vector3D(-this.size,-this.size,this.size), new Vector3D(-this.size,this.size,this.size));
	this.lines[5] = new Line3D(new Vector3D(this.size,-this.size,this.size), new Vector3D(this.size,this.size,this.size));
	this.lines[6] = new Line3D(new Vector3D(-this.size,-this.size,-this.size), new Vector3D(-this.size,this.size,-this.size));
	this.lines[7] = new Line3D(new Vector3D(this.size,-this.size,-this.size), new Vector3D(this.size,this.size,-this.size));
	this.lines[8] = new Line3D(new Vector3D(this.size,-this.size,this.size), new Vector3D(this.size,-this.size,-this.size));
	this.lines[9] = new Line3D(new Vector3D(-this.size,-this.size,this.size), new Vector3D(-this.size,-this.size,-this.size));
	this.lines[10] = new Line3D(new Vector3D(-this.size,this.size,this.size), new Vector3D(-this.size,this.size,-this.size));
	this.lines[11] = new Line3D(new Vector3D(this.size,this.size,this.size), new Vector3D(this.size,this.size,-this.size));
}

Object3D.prototype.make_hyper = function(){
	this.lines[0] = new Line3D(new Vector3D(-this.size,-this.size,this.size), new Vector3D(this.size,-this.size,this.size));
	this.lines[1] = new Line3D(new Vector3D(-this.size,this.size,this.size), new Vector3D(this.size,this.size,this.size));
	this.lines[2] = new Line3D(new Vector3D(-this.size,-this.size,-this.size), new Vector3D(this.size,-this.size,-this.size));
	this.lines[3] = new Line3D(new Vector3D(-this.size,this.size,-this.size), new Vector3D(this.size,this.size,-this.size));
	this.lines[4] = new Line3D(new Vector3D(-this.size,-this.size,this.size), new Vector3D(-this.size,this.size,this.size));
	this.lines[5] = new Line3D(new Vector3D(this.size,-this.size,this.size), new Vector3D(this.size,this.size,this.size));
	this.lines[6] = new Line3D(new Vector3D(-this.size,-this.size,-this.size), new Vector3D(-this.size,this.size,-this.size));
	this.lines[7] = new Line3D(new Vector3D(this.size,-this.size,-this.size), new Vector3D(this.size,this.size,-this.size));
	this.lines[8] = new Line3D(new Vector3D(this.size,-this.size,this.size), new Vector3D(this.size,-this.size,-this.size));
	this.lines[9] = new Line3D(new Vector3D(-this.size,-this.size,this.size), new Vector3D(-this.size,-this.size,-this.size));
	this.lines[10] = new Line3D(new Vector3D(-this.size,this.size,this.size), new Vector3D(-this.size,this.size,-this.size));
	this.lines[11] = new Line3D(new Vector3D(this.size,this.size,this.size), new Vector3D(this.size,this.size,-this.size));
	
	
	var b = 1.7;
	
	// Bigger
	this.lines[12] = new Line3D(new Vector3D(-b * this.size,-b * this.size,b * this.size), new Vector3D(b * this.size,-b * this.size,b * this.size));
	this.lines[13] = new Line3D(new Vector3D(-b * this.size,b * this.size,b * this.size), new Vector3D(b * this.size,b * this.size,b * this.size));
	this.lines[14] = new Line3D(new Vector3D(-b * this.size,-b * this.size,-b * this.size), new Vector3D(b * this.size,-b * this.size,-b * this.size));
	this.lines[15] = new Line3D(new Vector3D(-b * this.size,b * this.size,-b * this.size), new Vector3D(b * this.size,b * this.size,-b * this.size));
	this.lines[16] = new Line3D(new Vector3D(-b * this.size,-b * this.size,b * this.size), new Vector3D(-b * this.size,b * this.size,b * this.size));
	this.lines[17] = new Line3D(new Vector3D(b * this.size,-b * this.size,b * this.size), new Vector3D(b * this.size,b * this.size,b * this.size));
	this.lines[18] = new Line3D(new Vector3D(-b * this.size,-b * this.size,-b * this.size), new Vector3D(-b * this.size,b * this.size,-b * this.size));
	this.lines[19] = new Line3D(new Vector3D(b * this.size,-b * this.size,-b * this.size), new Vector3D(b * this.size,b * this.size,-b * this.size));
	this.lines[20] = new Line3D(new Vector3D(b * this.size,-b * this.size,b * this.size), new Vector3D(b * this.size,-b * this.size,-b * this.size));
	this.lines[21] = new Line3D(new Vector3D(-b * this.size,-b * this.size,b * this.size), new Vector3D(-b * this.size,-b * this.size,-b * this.size));
	this.lines[22] = new Line3D(new Vector3D(-b * this.size,b * this.size,b * this.size), new Vector3D(-b * this.size,b * this.size,-b * this.size));
	this.lines[23] = new Line3D(new Vector3D(b * this.size,b * this.size,b * this.size), new Vector3D(b * this.size,b * this.size,-b * this.size));
	
	
	// Connecting lines
	//this.lines[24] = new Line3D(new Vector3D(-b * this.size,-b * this.size,b * this.size), new Vector3D(-this.size,-this.size,this.size));
	//this.lines[25] = new Line3D(new Vector3D(-b * this.size,b * this.size,b * this.size), new Vector3D(-this.size,this.size,this.size));
	//this.lines[26] = new Line3D(new Vector3D(b * this.size,b * this.size,b * this.size), new Vector3D(this.size,this.size,this.size));
	//this.lines[27] = new Line3D(new Vector3D(-b * this.size,b * this.size,-b * this.size), new Vector3D(-this.size,this.size,-this.size));
	//this.lines[28] = new Line3D(new Vector3D(b * this.size,-b * this.size,-b * this.size), new Vector3D(this.size,-this.size,-this.size));
	//this.lines[29] = new Line3D(new Vector3D(b * this.size,b * this.size,-b * this.size), new Vector3D(this.size,this.size,-this.size));
	//this.lines[30] = new Line3D(new Vector3D(-b * this.size,-b * this.size,-b * this.size), new Vector3D(-this.size,-this.size,-this.size));
	//this.lines[31] = new Line3D(new Vector3D(b * this.size,-b * this.size,b * this.size), new Vector3D(this.size,-this.size,this.size));

}


Object3D.prototype.make_pyramid = function(){
	var factor = 1.4; //Height factor

	this.lines[0] = new Line3D(new Vector3D(this.size,0,this.size), new Vector3D(this.size,0,-this.size));
	this.lines[1] = new Line3D(new Vector3D(this.size,0,this.size), new Vector3D(-this.size,0,this.size));
	this.lines[2] = new Line3D(new Vector3D(this.size,0,this.size), new Vector3D(0,factor*this.size,0));
	this.lines[3] = new Line3D(new Vector3D(this.size,0,-this.size), new Vector3D(-this.size,0,-this.size));
	this.lines[4] = new Line3D(new Vector3D(this.size,0,-this.size), new Vector3D(0,factor*this.size,0));
	this.lines[5] = new Line3D(new Vector3D(-this.size,0,this.size), new Vector3D(-this.size,0,-this.size));
	this.lines[6] = new Line3D(new Vector3D(-this.size,0,this.size), new Vector3D(0,factor*this.size,0));
	this.lines[7] = new Line3D(new Vector3D(-this.size,0,-this.size), new Vector3D(0,factor*this.size,0));
}

/**
* Calculates 2D lines of object
**/
Object3D.prototype.calc_lines2D = function(){
	for(i=0;i<this.lines.length;i++){
		dxy = this.calc_point(this.lines[i].coordinate);		
		xy = this.calc_point(this.lines[i].dcoordinate);
		var line = new Line(dxy, xy);
		this.lines2D[i] = line;
	}
}

/**
* Updates cube
**/
Object3D.prototype.update = function(){
	this.calc_lines2D();
}

/**
* Rotates cube by given angle, adds given angle to exsisting!
* angle = Vector 3D angle on each axis
**/
Object3D.prototype.rotate = function(angle){
	this.angle.x += angle.x;
	this.angle.y += angle.y;
	this.angle.z += angle.z;
	this.update();
}

/**
* Calculates 2D projection to 3D point
* point = Vector3D point to project
* Returns Vector2D containing poitns x,y coordinates
**/
Object3D.prototype.calc_point = function(point){
	var xd = point.x - this.piv.x; 
	var yd = point.y - this.piv.y; 
	var zd = point.z - this.piv.z;
	
	var zx = xd * Math.cos(this.angle.z) - yd * Math.sin(this.angle.z) - xd;
	var zy = xd * Math.sin(this.angle.z) + yd * Math.cos(this.angle.z) - yd;
	
	var yx = (xd+zx) * Math.cos(this.angle.y) - zd * Math.sin(this.angle.y) - (xd+zx);
	var yz = (xd+zx) * Math.sin(this.angle.y) + zd * Math.cos(this.angle.y) - zd;
	
	var xy = (yd+zy) * Math.cos(this.angle.x) - (zd+yz) * Math.sin(this.angle.x) - (yd+zy);
	var xz = (yd+zy) * Math.sin(this.angle.x) + (zd+yz) * Math.cos(this.angle.x) - (zd+yz);
	
	var xrotoffset = yx + zx;
	var yrotoffset = zy + xy;

	
	var x = (point.x + xrotoffset + this.cam.x) * this.scale + this.position.x;
	var y = (point.y + yrotoffset + this.cam.y) * this.scale + this.position.y;
	
	var dot = new Vector2D(48 + Math.floor(x), 27 + Math.floor(y));
	return dot;	
}

/**
* Draws cube
**/
Object3D.prototype.draw = function(){	
	for (i in this.lines2D){
		this.lines2D[i].draw();
	}
}


// < 3D />

// < 2D >
/**
* 2D Vector object for x,y coordinates
**/
function Vector2D(x,y){
	this.x = Math.floor(x);
	this.y = Math.floor(y);
}

/**
* Returns Vector2D in 'x,y' form
**/ 
Vector2D.prototype.id = function() {
	return this.x + ',' + this.y;
}

/**
* Draws Vector2D aka checks one point
**/
Vector2D.prototype.draw = function(){
	try{
		document.getElementById(this.x + ',' + this.y).checked = true;}
	catch (err){
		//Ignores pixels out of range
	}
}

/**
* Line object
* xy = Vector 2D starting point
* dxy = Vector 2D ending point
**/
function Line(xy,dxy){
	this.xy = xy;
	this.dxy = dxy;
	this.dots = new Array();
	this.make_dots();	
}

/**
* Makes all dots needed in the line into array (this.dots)
* Uses Bresenham's line algorithm 
**/
Line.prototype.make_dots = function(){
	var x0 = this.xy.x;
	var x1 = this.dxy.x;
	var y0 = this.xy.y;
	var y1 = this.dxy.y;
	
	var steep = (Math.abs(y1 - y0) > Math.abs(x1 -x0));
	
	var swap = 0;
	if (steep){
		swap = x0; x0 = y0; y0 = swap;
		swap = x1; x1 = y1; y1 = swap;
	}
	if (x0 > x1){
		swap = x0; x0 = x1; x1 = swap;
		swap = y0; y0 = y1; y1 = swap;
	}
	
	var deltax = x1 - x0;
	var deltay = Math.abs(y1-y0);
	var error = deltax / 2;
	var ystep;
	var y = y0;
	if (y0 < y1){
		ystep = 1;
	}
	else{
		ystep = -1;
	}
	
	var i = 0;
	for (x=x0; x <= x1; x++){
		var dot;
		if (steep){ dot = new Vector2D(y,x);}
		else {dot = new Vector2D(x,y);}
		this.dots[i++] = dot;
		error -= deltay;
		if (error < 0){
			y += ystep;
			error += deltax;
		}
	}
}

/**
* Draws line by drawing all points in this.dots
**/
Line.prototype.draw = function(){
	for (i in this.dots){
		this.dots[i].draw();
	}
}


/**
* Sin wave object
* position = position where to draw
* heigth = heigth of the wave
* width = width of the wave
* offset = change this to make wave move
**/
function Sin_wave(position, heigth, width, offset, thickness){	
	this.position = position;
	this.heigth = heigth;
	this.width = width;
	this.offset = offset;
	this.thickness = thickness;
}

/**
* Draws beatiful sin wave
**/
Sin_wave.prototype.draw = function(){
	for (i=1; i <= 96; i++){
			var x = i;
			var y = this.position + Math.floor(Math.sin((i + this.offset) * this.width) * this.heigth);
			for (j=0; j < this.thickness;j++){
				var d = new Vector2D(x,y-j);
				d.draw();
			}
	}
		
	
}

/**
* Creates regular polygon with given paramets
* position = Vector 2D position
* corners = amount of corners in polygon
* size = size of polygon
**/
function Polygon(position, corners, size){
	this.position = position;
	this.corners = corners;
	this.size = size;
	this.lines = new Array();
	this.make_lines();
}

/**
* Makes all lines needed in polygon
**/
Polygon.prototype.make_lines = function(){
	var xcoords = new Array();
	var ycoords = new Array();
	var angle = (Math.PI * 2) / this.corners;
	var calc_angle = 0; 
	
	for (i=0; i < this.corners; i++){
		xcoords[i] = Math.round(this.position.x + this.size * Math.sin(calc_angle));
		ycoords[i] = Math.round(this.position.y + this.size * Math.cos(calc_angle));
		calc_angle += angle;
	}
	
	
	var last = xcoords.length - 1
	for (i=0; i < last; i++){
		this.lines[i] = new Line(new Vector2D(xcoords[i],ycoords[i]), new Vector2D(xcoords[i+1],ycoords[i+1]));	
	}	
	this.lines[last] = new Line(new Vector2D(xcoords[last],ycoords[last]), new Vector2D(xcoords[0],ycoords[0]));
}

/**
* Draws polygon
**/
Polygon.prototype.draw = function(){
	for (i in this.lines){
		this.lines[i].draw();
	}
	
}

/**
* Constractor for Tron text
* startpoint = Vector2D startpoint of tron
* route = rout for tron text, Array containing arrays where index0= line length and index1= next direction, 0=up, 1=upright, 2=right, 3=downright, 4=down, 5=downleft, 6=left, 7=upleft; 
**/
function Tron_text(startpoint,route){
	this.dot = startpoint;
	this.route = route;
	this.index = 0;
	this.ticks = route[0][0];
	this.direction = route[0][1];
}


/**
* Updates Tron text according to route and draws new point
**/
Tron_text.prototype.update = function(){
	for (i=0; i < 3; i++){
	this.dot.draw();
	
		if (this.ticks == 0){
			if (this.index < this.route.length){
				this.index++;
				this.ticks = this.route[this.index][0];
				this.direction = this.route[this.index][1];
			}
			else return 1;
		}
		
		if (this.direction == 0){
			this.dot.y++;
		}
		if (this.direction == 1){
			this.dot.y++;
			this.dot.x++;
		}
		if (this.direction == 2){
			this.dot.x++;
		}
		if (this.direction == 3){
			this.dot.x++;
			this.dot.y--;
		}
		if (this.direction == 4){
			this.dot.y--;
		}
		if (this.direction == 5){
			this.dot.y--;
			this.dot.x--;
		}
		if (this.direction == 6){
			this.dot.x--;
		}
		if (this.direction == 7){
			this.dot.y++;
			this.dot.x--;
		}
		this.ticks--;
	} 
	return 0;
}

/**
* Counts from 8 to 0 with binary numbers
* position = Vector2D position of left corner of counter
* scale = size of the counter
**/
function Binary_counter(position, scale){
	this.position = position;
	this.scale = scale;
	this.space = 4;
	this.bits = new Array(1,0,0,0);
	this.lines = new Array();
	this.n = 8;
	this.make_lines();
} 

Binary_counter.prototype.update = function(){
	switch(--this.n){
		case 8:
			this.bits = new Array(1,0,0,0);
			break;
		case 7:
			this.bits = new Array(0,1,1,1);
			break;
		case 6:
			this.bits = new Array(0,1,1,0);
			break;
		case 5:
			this.bits = new Array(0,1,0,1);
			break;
		case 4:
			this.bits = new Array(0,1,0,0);	
			break;
		case 3:
			this.bits = new Array(0,0,1,1);
			break;
		case 2:
			this.bits = new Array(0,0,1,0);
			break;
		case 1:
			this.bits = new Array(0,0,0,1);
			break;
	
	}
	this.make_lines();
}

Binary_counter.prototype.make_lines = function(){
	var x = this.position.x;
	var y = this.position.y;
	
	for (i in this.bits){
		if (this.bits[i] == 1){
			x += this.scale;
			this.lines[i] = new Array( new Line( new Vector2D(x,y), new Vector2D(x,y+2*this.scale) ) );
		}
		else {
			var t = new Array(
				new Line( new Vector2D(x,y), new Vector2D(x+this.scale,y) ),
				new Line( new Vector2D(x,y), new Vector2D(x,y+2*this.scale) ),
				new Line( new Vector2D(x,y+2*this.scale), new Vector2D(x+this.scale,y+2*this.scale) ),
				new Line( new Vector2D(x+this.scale,y+2*this.scale), new Vector2D(x+this.scale,y) )
			);
			this.lines[i] = t;
			x += this.scale;
		}

		x += this.space;
	}
}

Binary_counter.prototype.draw = function(){
	for (j in this.lines){
		for (i in this.lines[j]){
			this.lines[j][i].draw();
		}
	}

}


/**
* Constructor for cool Star object
* position = Vector2D position of star
* scale = start scale of the star
* grow = how much star grows on each update
**/
function Star(position,scale,grow){
	this.position = position;
	this.scale = scale;
	this.grow = grow;
	this.lines = new Array();
	this.make_lines();
}

/**
* Makes all lines needed in star
**/
Star.prototype.make_lines = function(){
	var x = this.position.x -  1.5*this.scale;
	var y = this.position.y +  1.5*this.scale;
	
	
	var lines = new Array();
	
	lines[0] = new Line( new Vector2D(x,y), new Vector2D(x+this.scale,y) );
	lines[1] = new Line( new Vector2D(x,y), new Vector2D(x + this.scale * 0.5, y+this.scale) );
	lines[2] = new Line( new Vector2D(x+this.scale,y), new Vector2D(x + this.scale * 0.5 , y+this.scale) );
		
	y += 2/3 * this.scale;
	
	lines[3] = new Line( new Vector2D(x,y), new Vector2D(x+this.scale,y) );
	lines[4] = new Line( new Vector2D(x,y), new Vector2D(x + this.scale * 0.5, y-this.scale) );
	lines[5] = new Line( new Vector2D(x+this.scale,y), new Vector2D(x + this.scale * 0.5 , y-this.scale) );
	
	this.lines = lines;
}

/**
* Makes star little bit larger
**/
Star.prototype.update = function(){
	this.scale += this.grow;
	this.make_lines();
}

/**
* Draws the star
**/
Star.prototype.draw = function(){
	for (i in this.lines){
		this.lines[i].draw();
	}
}


/**
* Creates cool hexagon web filling whole screen
**/
function Hexaweb(){
	this.width = 14;
	this.heigth = 17;
	this.offset = new Vector2D(0,0);
	this.start = new Vector2D(1,1);
	this.lines = new Array();
	this.make_lines();
}


/**
* Creates all lines needed in hexagonweb
**/
Hexaweb.prototype.make_lines = function(){
	var x = this.start.x + this.offset.x - 90;
	var y = this.start.y + this.offset.y -60;
	
	var start_x = x;
	
	this.lines = new Array();
	
	for (var j=0; j < 10; j++){
		
		if (j % 2 != 0){
			x = start_x +  0.5 * this.width;
		} 
		else {
			x = start_x;
		}
		
		
		for (i=0; i < 15; i++){
		
			hex = new Array(
				new Line( new Vector2D(x+0.5*this.width,y), new Vector2D(x, y + 1/3 * this.heigth) ),
				new Line( new Vector2D(x+0.5*this.width,y), new Vector2D(x + this.width, y + 1/3 * this.heigth) ),
				new Line( new Vector2D(x, y + 1/3 * this.heigth), new Vector2D(x, y + 2/3 * this.heigth) ),
				new Line( new Vector2D(x + this.width, y + 1/3 * this.heigth), new Vector2D(x + this.width, y + 2/3 * this.heigth) ),
				new Line( new Vector2D(x+0.5*this.width, y+this.heigth), new Vector2D(x, y + 2/3 * this.heigth) ),
				new Line( new Vector2D(x+0.5*this.width, y+this.heigth), new Vector2D(x +this. width, y + 2/3 * this.heigth) )
			);
		
			this.lines = this.lines.concat(hex);
			x += this.width;
		}
	
		y += 2/3*this.heigth;
	
	}
	
}

/**
* Incereases or decreases hexagons' width and/or heigth
* width = new value for width
* heigth = new value for heigth
* offset = Vector2D value, changing this chances position of hexagons
**/
Hexaweb.prototype.update = function(width, heigth, offset){
	this.width += width;
	this.heigth += heigth;
	this.offset = new Vector2D(this.offset.x + offset.x, this.offset.y + offset.y);
	this.make_lines();
}

Hexaweb.prototype.draw = function(){
	for (i in this.lines){
		this.lines[i].draw();
	}
}



/**
* (supercool) Spring object wich bounces in sine wave
* position = position of spring
* offset = offset value for sine
* letter = letter index of word awesome
**/
function Sine_spring(position, offset, letter){
	this.position = new Vector2D(position,56);
	this.letter = letter;
	this.sine_value = 0;
	this.i = offset;
	this.width = 0.2;
	this.heigth = 8;
	this.z = 7;
	this.lines = new Array();
	this.make_lines();
	
	
}


/**
* Creates all lines needed in this animation
**/
Sine_spring.prototype.make_lines = function(){
	var cube_anchor = new Vector2D(this.position.x, this.sine_value);

	var z = this.z;
	var t = Math.round((54 - cube_anchor.y -7) / 5);
	var v = Math.round( Math.sqrt( ((z*z) - (t*t)) ));
	
	var spring_start = new Vector2D(this.position.x, this.position.y-3);
	var spring_end = new Vector2D(cube_anchor.x, cube_anchor.y+2);
	
	
	this.lines = new Array(
		new Line ( this.position, spring_start),
		new Line ( spring_start, new Vector2D(spring_start.x + 0.5*v, spring_start.y -t)),
		new Line ( new Vector2D(spring_start.x + 0.5*v, spring_start.y -t), new Vector2D(spring_start.x -v, spring_start.y -2*t)),
		new Line ( new Vector2D(spring_start.x -v, spring_start.y -2*t), new Vector2D(spring_start.x +v, spring_start.y -3*t)),
		new Line ( new Vector2D(spring_start.x +v, spring_start.y -3*t), new Vector2D(spring_start.x -v, spring_start.y -4*t)),
		new Line ( new Vector2D(spring_start.x -v, spring_start.y -4*t), new Vector2D(spring_start.x +v, spring_start.y -5*t)),
		new Line ( new Vector2D(spring_start.x +v, spring_start.y -5*t), spring_end),
		new Line ( cube_anchor, spring_end)
		
	);

	var letter_lines = new Array();
	
	// A
	if (this.letter == 0){
		letter_lines[0] = new Line(cube_anchor, new Vector2D(cube_anchor.x+3, cube_anchor.y-7));
		letter_lines[1] = new Line(cube_anchor, new Vector2D(cube_anchor.x-3, cube_anchor.y-7));
		letter_lines[2] = new Line(new Vector2D(cube_anchor.x+3, cube_anchor.y-7), new Vector2D(cube_anchor.x-3, cube_anchor.y-7));
		letter_lines[3] = new Line(new Vector2D(cube_anchor.x-3, cube_anchor.y-7), new Vector2D(cube_anchor.x-3, cube_anchor.y-13));
		letter_lines[4] = new Line(new Vector2D(cube_anchor.x+3, cube_anchor.y-7), new Vector2D(cube_anchor.x+3, cube_anchor.y-13));
	}
	// W
	if (this.letter == 1){
		letter_lines[0] = new Line(cube_anchor, new Vector2D(cube_anchor.x+3, cube_anchor.y-13));
		letter_lines[1] = new Line(cube_anchor, new Vector2D(cube_anchor.x-3, cube_anchor.y-13));
		letter_lines[2] = new Line(new Vector2D(cube_anchor.x+3, cube_anchor.y-13), new Vector2D(cube_anchor.x+6, cube_anchor.y));
		letter_lines[3] = new Line(new Vector2D(cube_anchor.x-3, cube_anchor.y-13), new Vector2D(cube_anchor.x-6, cube_anchor.y));
	}
	// E
	if (this.letter == 2 || this.letter == 6){
		letter_lines[0] = new Line(new Vector2D(cube_anchor.x+3, cube_anchor.y), new Vector2D(cube_anchor.x-3, cube_anchor.y));
		letter_lines[1] = new Line(new Vector2D(cube_anchor.x-3, cube_anchor.y-13), new Vector2D(cube_anchor.x-3, cube_anchor.y));
		letter_lines[2] = new Line(new Vector2D(cube_anchor.x+3, cube_anchor.y-13), new Vector2D(cube_anchor.x-3, cube_anchor.y-13));
		letter_lines[3] = new Line(new Vector2D(cube_anchor.x-3, cube_anchor.y-6), new Vector2D(cube_anchor.x, cube_anchor.y-6));
	}
	// S
	if (this.letter == 3){
		letter_lines[0] = new Line(new Vector2D(cube_anchor.x+3, cube_anchor.y), new Vector2D(cube_anchor.x-3, cube_anchor.y));
		letter_lines[1] = new Line(new Vector2D(cube_anchor.x+3, cube_anchor.y-6), new Vector2D(cube_anchor.x-3, cube_anchor.y-6));
		letter_lines[2] = new Line(new Vector2D(cube_anchor.x+3, cube_anchor.y-13), new Vector2D(cube_anchor.x-3, cube_anchor.y-13));
		letter_lines[3] = new Line(new Vector2D(cube_anchor.x-3, cube_anchor.y), new Vector2D(cube_anchor.x-3, cube_anchor.y-6));
		letter_lines[4] = new Line(new Vector2D(cube_anchor.x+3, cube_anchor.y-7), new Vector2D(cube_anchor.x+3, cube_anchor.y-13));
	}
	// O
	if (this.letter == 4){
		letter_lines[0] = new Line(new Vector2D(cube_anchor.x+3, cube_anchor.y), new Vector2D(cube_anchor.x-3, cube_anchor.y));
		letter_lines[1] = new Line(new Vector2D(cube_anchor.x+3, cube_anchor.y), new Vector2D(cube_anchor.x+3, cube_anchor.y-13));
		letter_lines[2] = new Line(new Vector2D(cube_anchor.x-3, cube_anchor.y), new Vector2D(cube_anchor.x-3, cube_anchor.y-13));
		letter_lines[3] = new Line(new Vector2D(cube_anchor.x-3, cube_anchor.y-13), new Vector2D(cube_anchor.x+3, cube_anchor.y-13));
	}
	// M
	if (this.letter == 5){
		letter_lines[0] = new Line(new Vector2D(cube_anchor.x+3, cube_anchor.y), new Vector2D(cube_anchor.x-3, cube_anchor.y));
		letter_lines[1] = new Line(new Vector2D(cube_anchor.x+3, cube_anchor.y), new Vector2D(cube_anchor.x+3, cube_anchor.y-13));
		letter_lines[2] = new Line(new Vector2D(cube_anchor.x-3, cube_anchor.y), new Vector2D(cube_anchor.x-3, cube_anchor.y-13));
		letter_lines[3] = new Line(new Vector2D(cube_anchor.x, cube_anchor.y), new Vector2D(cube_anchor.x, cube_anchor.y-4));
	}

	
	
	this.lines = this.lines.concat(letter_lines);
	
}


/**
* Updates sine value
**/
Sine_spring.prototype.update = function(){
	this.sine_value = Math.floor(Math.sin((++this.i) * this.width) * this.heigth) + 25;
	this.make_lines();

}

/**
* Draws the spring
**/
Sine_spring.prototype.draw = function(){
	for (i in this.lines){
		this.lines[i].draw();
	}
}

/**
* Creates 2D text with given parametrs
* input = text to create!
* position = Vector2D position of leftdown corer of first letter
* width = width of one letter
* height = height of one letter
**/
function Text2D(input, position, width, height){
	this.input = input.toUpperCase();
	this.position = position;
	this.width = width;
	this.height = height;
	this.lines = new Array();
	this.make_lines();


}


/**
* Creates all lines needed in word
**/
Text2D.prototype.make_lines = function(){
	var letter = "A";
	var h = this.height;
	var w = this.width;
	var x = this.position.x;
	var y = this.position.y;
	var lines = new Array();
	

	for (i=0; i<this.input.length; i++){
		letter = this.input.charAt(i);
		lines = new Array();
		
		if ( letter == "A"){
			lines[0] = new Line ( new Vector2D(x,y), new Vector2D(x,y+h) );
			lines[1] = new Line ( new Vector2D(x,y+h), new Vector2D(x+w,y+h) );
			lines[2] = new Line ( new Vector2D(x+w,y+h), new Vector2D(x+w,y) );
			lines[3] = new Line ( new Vector2D(x,y + 2/3*h), new Vector2D(x+w,y + 2/3*h) );
		}
		
		if ( letter == "B"){
			lines[0] = new Line ( new Vector2D(x,y), new Vector2D(x,y+h) );
			lines[1] = new Line ( new Vector2D(x,y+h), new Vector2D(x+ 2/3*w,y+h) );
			lines[2] = new Line ( new Vector2D(x+ 2/3*w,y+h), new Vector2D(x+ 2/3*w,y+2/3*h) );
			lines[3] = new Line ( new Vector2D(x,y+2/3*h), new Vector2D(x+w,y+2/3*h) );
			lines[4] = new Line ( new Vector2D(x+w,y+2/3*h), new Vector2D(x+w,y) );
			lines[5] = new Line ( new Vector2D(x,y), new Vector2D(x+w,y) );
		}
		
		if ( letter == "C"){
			lines[0] = new Line ( new Vector2D(x,y), new Vector2D(x,y+h) );
			lines[1] = new Line ( new Vector2D(x,y+h), new Vector2D(x+w,y+h) );
			lines[2] = new Line ( new Vector2D(x,y), new Vector2D(x+w,y) );
		}
		
		if ( letter == "D"){
			lines[0] = new Line ( new Vector2D(x,y), new Vector2D(x,y+h) );
			lines[1] = new Line ( new Vector2D(x,y+h), new Vector2D(x+2/3*w,y+h) );
			lines[2] = new Line ( new Vector2D(x+2/3*w,y+h), new Vector2D(x+w,y+2/3*h) );
			lines[3] = new Line ( new Vector2D(x+w,y+2/3*h), new Vector2D(x+w,y+1/3*h) );
			lines[4] = new Line ( new Vector2D(x+w,y+1/3*h), new Vector2D(x+2/3*w,y) );
			lines[5] = new Line ( new Vector2D(x+2/3*w,y), new Vector2D(x,y) );
		}
		
		if ( letter == "E"){
			lines[0] = new Line ( new Vector2D(x,y), new Vector2D(x,y+h) );
			lines[1] = new Line ( new Vector2D(x,y+h), new Vector2D(x+w,y+h) );
			lines[2] = new Line ( new Vector2D(x,y+2/3*h), new Vector2D(x+2/3*w,y+2/3*h) );
			lines[3] = new Line ( new Vector2D(x,y), new Vector2D(x+w,y) );
		}
		
		if ( letter == "F"){
			lines[0] = new Line ( new Vector2D(x,y), new Vector2D(x,y+h) );
			lines[1] = new Line ( new Vector2D(x,y+h), new Vector2D(x+w,y+h) );
			lines[2] = new Line ( new Vector2D(x,y+2/3*h), new Vector2D(x+2/3*w,y+2/3*h) );
			
		}
		
		if ( letter == "G"){
			lines[0] = new Line ( new Vector2D(x,y), new Vector2D(x,y+h) );
			lines[1] = new Line ( new Vector2D(x,y+h), new Vector2D(x+w,y+h) );
			lines[2] = new Line ( new Vector2D(x,y), new Vector2D(x+w,y) );
			lines[3] = new Line ( new Vector2D(x+w,y), new Vector2D(x+w,y+1/3*h) );
			lines[4] = new Line ( new Vector2D(x+2/3*w,y+1/3*h), new Vector2D(x+w,y+1/3*h) );
		}
		
		if ( letter == "H"){
			lines[0] = new Line ( new Vector2D(x,y), new Vector2D(x,y+h) );
			lines[1] = new Line ( new Vector2D(x+w,y), new Vector2D(x+w,y+h) );
			lines[2] = new Line ( new Vector2D(x,y+2/3*h), new Vector2D(x+w,y+2/3*h) );
		}
		
		if ( letter == "I" || letter == "1"){
			lines[0] = new Line ( new Vector2D(x+1/2*w,y), new Vector2D(x+1/2*w,y+h) );
		}
		
		if ( letter == "J"){
			lines[0] = new Line ( new Vector2D(x+w,y), new Vector2D(x+w,y+h) );
			lines[1] = new Line ( new Vector2D(x+w,y), new Vector2D(x,y) );
			lines[2] = new Line ( new Vector2D(x,y+1/3*h), new Vector2D(x,y) );
		}
		
		if ( letter == "K"){
			lines[0] = new Line ( new Vector2D(x,y), new Vector2D(x,y+h) );
			lines[1] = new Line ( new Vector2D(x,y+1/2*h), new Vector2D(x+w,y+h) );
			lines[2] = new Line ( new Vector2D(x,y+1/2*h), new Vector2D(x+w,y) );
		}
		
		if ( letter == "L"){
			lines[0] = new Line ( new Vector2D(x,y), new Vector2D(x,y+h) );
			lines[1] = new Line ( new Vector2D(x,y), new Vector2D(x+w,y) );
		}
		
		if ( letter == "M"){
			lines[0] = new Line ( new Vector2D(x,y), new Vector2D(x,y+h) );
			lines[1] = new Line ( new Vector2D(x+w,y), new Vector2D(x+w,y+h) );
			lines[2] = new Line ( new Vector2D(x,y+h), new Vector2D(x+w,y+h) );
			lines[3] = new Line ( new Vector2D(x+1/2*w,y+2/3*h), new Vector2D(x+1/2*w,y+h) );
		}
		
		if ( letter == "N"){
			lines[0] = new Line ( new Vector2D(x,y), new Vector2D(x,y+h) );
			lines[1] = new Line ( new Vector2D(x+w,y), new Vector2D(x+w,y+h) );
			lines[2] = new Line ( new Vector2D(x,y+h), new Vector2D(x+w,y) );
		}
		
		if ( letter == "O" || letter == "0"){
			lines[0] = new Line ( new Vector2D(x,y), new Vector2D(x,y+h) );
			lines[1] = new Line ( new Vector2D(x+w,y), new Vector2D(x+w,y+h) );
			lines[2] = new Line ( new Vector2D(x,y+h), new Vector2D(x+w,y+h) );
			lines[3] = new Line ( new Vector2D(x,y), new Vector2D(x+w,y) );
		}
		
		if ( letter == "P"){
			lines[0] = new Line ( new Vector2D(x,y), new Vector2D(x,y+h) );
			lines[1] = new Line ( new Vector2D(x,y+h), new Vector2D(x+w,y+h) );
			lines[2] = new Line ( new Vector2D(x,y+2/3*h), new Vector2D(x+w,y+2/3*h) );
			lines[3] = new Line ( new Vector2D(x+w,y+2/3*h), new Vector2D(x+w,y+h) );
		}
		
		if ( letter == "R"){
			lines[0] = new Line ( new Vector2D(x,y), new Vector2D(x,y+h) );
			lines[1] = new Line ( new Vector2D(x,y+h), new Vector2D(x+w,y+h) );
			lines[2] = new Line ( new Vector2D(x,y+2/3*h), new Vector2D(x+w,y+2/3*h) );
			lines[3] = new Line ( new Vector2D(x+w,y+2/3*h), new Vector2D(x+w,y+h) );
			lines[4] = new Line ( new Vector2D(x,y+2/3*h), new Vector2D(x+w,y) );
		}
		
		if ( letter == "S" || letter == "5"){
			lines[0] = new Line ( new Vector2D(x,y), new Vector2D(x+w,y) );
			lines[1] = new Line ( new Vector2D(x+w,y+1/2*h), new Vector2D(x+w,y) );
			lines[2] = new Line ( new Vector2D(x+w,y+1/2*h), new Vector2D(x,y+1/2*h) );
			lines[3] = new Line ( new Vector2D(x,y+h), new Vector2D(x,y+1/2*h) );
			lines[4] = new Line ( new Vector2D(x,y+h), new Vector2D(x+w,y+h) );
		}
		
		if ( letter == "T"){
			lines[0] = new Line ( new Vector2D(x,y+h), new Vector2D(x+w,y+h) );
			lines[1] = new Line ( new Vector2D(x+1/2*w,y), new Vector2D(x+1/2*w,y+h) );
		}
		
		if ( letter == "U"){
			lines[0] = new Line ( new Vector2D(x,y), new Vector2D(x,y+h) );
			lines[1] = new Line ( new Vector2D(x+w,y), new Vector2D(x+w,y+h) );
			lines[2] = new Line ( new Vector2D(x,y), new Vector2D(x+w,y) );
		}
		
		if ( letter == "W"){
			lines[0] = new Line ( new Vector2D(x,y), new Vector2D(x,y+h) );
			lines[1] = new Line ( new Vector2D(x+w,y), new Vector2D(x+w,y+h) );
			lines[2] = new Line ( new Vector2D(x,y), new Vector2D(x+w,y) );
			lines[3] = new Line ( new Vector2D(x+1/2*w,y), new Vector2D(x+1/2*w,y+1/3*h) );
		}
		
		if ( letter == "V"){
			lines[0] = new Line ( new Vector2D(x,y+h), new Vector2D(x+1/2*w,y) );
			lines[1] = new Line ( new Vector2D(x+w,y+h), new Vector2D(x+1/2*w,y) );
		}
		
		if ( letter == "Y"){
			lines[0] = new Line ( new Vector2D(x,y+h), new Vector2D(x+1/2*w,y+2/3*h) );
			lines[1] = new Line ( new Vector2D(x+w,y+h), new Vector2D(x+1/2*w,y+2/3*h) );
			lines[2] = new Line ( new Vector2D(x+1/2*w,y), new Vector2D(x+1/2*w,y+2/3*h) );
		}
		
		if ( letter == "X"){
			lines[0] = new Line ( new Vector2D(x,y+h), new Vector2D(x+w,y) );
			lines[1] = new Line ( new Vector2D(x+w,y+h), new Vector2D(x,y) );
		}
		
	
		if ( letter == "!"){
			lines[0] = new Line ( new Vector2D(x+1/2*w,y+3), new Vector2D(x+1/2*w,y+h) );
			lines[1] = new Line ( new Vector2D(x+1/2*w-1,y+3), new Vector2D(x+1/2*w-1,y+h) );
			lines[2] = new Line ( new Vector2D(x+1/2*w,y+1), new Vector2D(x+1/2*w,y) );
			lines[3] = new Line ( new Vector2D(x+1/2*w-1,y+1), new Vector2D(x+1/2*w-1,y) );
		}
		
		if ( letter == "<"){
			lines[0] = new Line ( new Vector2D(x+w,y+h), new Vector2D(x,y+1/2*h) );
			lines[1] = new Line ( new Vector2D(x,y+1/2*h), new Vector2D(x+w,y) );
		}
		
		
		if (letter == "2"){
			lines[0] = new Line ( new Vector2D(x+w,y), new Vector2D(x,y) );
			lines[1] = new Line ( new Vector2D(x,y+1/2*h), new Vector2D(x,y) );
			lines[2] = new Line ( new Vector2D(x,y+1/2*h), new Vector2D(x+w,y+1/2*h) );
			lines[3] = new Line ( new Vector2D(x+w,y+h), new Vector2D(x+w,y+1/2*h) );
			lines[4] = new Line ( new Vector2D(x+w,y+h), new Vector2D(x,y+h) );
		}
		
		if (letter == "3"){
			lines[0] = new Line ( new Vector2D(x+w,y), new Vector2D(x+w,y+h) );
			lines[1] = new Line ( new Vector2D(x,y), new Vector2D(x+w,y) );
			lines[2] = new Line ( new Vector2D(x,y+h), new Vector2D(x+w,y+h) );
			lines[3] = new Line ( new Vector2D(x+1/2*w,y+1/2*h), new Vector2D(x+w,y+1/2*h) );
		}
		
		if (letter == "4"){
			lines[0] = new Line ( new Vector2D(x+w,y), new Vector2D(x+w,y+h) );
			lines[1] = new Line ( new Vector2D(x+w,y+1/2*h), new Vector2D(x,y+1/2*h) );
			lines[2] = new Line ( new Vector2D(x,y+1/2*h), new Vector2D(x,y+h) );
		}
		
		if (letter == "6"){
			lines[0] = new Line ( new Vector2D(x,y), new Vector2D(x,y+h) );
			lines[1] = new Line ( new Vector2D(x,y+h), new Vector2D(x+w,y+h) );
			lines[2] = new Line ( new Vector2D(x,y), new Vector2D(x+w,y) );
			lines[3] = new Line ( new Vector2D(x,y+1/2*h), new Vector2D(x+w,y+1/2*h) );
			lines[4] = new Line ( new Vector2D(x+w,y), new Vector2D(x+w,y+1/2*h) );
		}
		
		if (letter == "7"){
			lines[0] = new Line ( new Vector2D(x+w,y+h), new Vector2D(x,y+h) );
			lines[1] = new Line ( new Vector2D(x,y+h), new Vector2D(x,y) );
		}
		
		if ( letter == "8"){
			lines[0] = new Line ( new Vector2D(x,y), new Vector2D(x,y+h) );
			lines[1] = new Line ( new Vector2D(x+w,y), new Vector2D(x+w,y+h) );
			lines[2] = new Line ( new Vector2D(x,y+h), new Vector2D(x+w,y+h) );
			lines[3] = new Line ( new Vector2D(x,y), new Vector2D(x+w,y) );
			lines[4] = new Line ( new Vector2D(x,y+1/2*h), new Vector2D(x+w,y+1/2*h) );
		}
		
		if (letter == "9"){
			lines[0] = new Line ( new Vector2D(x+w,y), new Vector2D(x+w,y+h) );
			lines[1] = new Line ( new Vector2D(x,y+h), new Vector2D(x+w,y+h) );
			lines[2] = new Line ( new Vector2D(x,y), new Vector2D(x+w,y) );
			lines[3] = new Line ( new Vector2D(x,y+1/2*h), new Vector2D(x+w,y+1/2*h) );
			lines[4] = new Line ( new Vector2D(x,y+1/2*h), new Vector2D(x,y+h) );
		}
		
		x += w + 3;
		
		this.lines = this.lines.concat(lines);

	}
}


/**
* Updates texts parameters
* move = Vector2D to be added current position
* width = new value to be adden current width
* height = new value to be adden current height
**/
Text2D.prototype.update = function(move, width, height){
	this.position = new Vector2D( this.position.x + move.x, this.position.y + move.y);
	this.width += width;
	this.height += height;
	this.make_lines();
	
}


/**
* Draws all lines
**/
Text2D.prototype.draw = function(){
	for (j in this.lines){
		this.lines[j].draw();
	}
}

// < 2D />

// < OBJECTS />

// < UTILS >
/**
* Clears window
**/
function clear(){
	var pixels = document.getElementsByName('pixel');
	for (i = 0; i < pixels.length; i++){
		pixels[i].checked = false;
	}
}

/**
* Clears area between given points
* start = Vector 2D start point
* end = Vector 2D end point
**/
function clear_area(start, end){
	for( y = start.y; y <= end.y; y++){
		for (x = start.x; x <= end.x; x++){
			document.getElementById(x + ',' + y).checked = false;
		}
	}
}

/**
* Changes background color
* color = color to change, either name of color e.g. 'white' of the hex value e.g. '#ffffff'
**/
function bgColor(color){
	var pixels = document.getElementsByName('pixel');
	for (i = 0; i < pixels.length; i++){
		pixels[i].style.backgroundColor = color;
	}
}


function start_music(){
	var music = document.getElementById('music');
	music.src = "oldschool_summer.ogg";
}

// < UTILS />


//Debug dot:



// < EFFECTS >

/**
* Countdown
**/
function Countdown(){
	this.counter = new Binary_counter(new Vector2D(20,18), 10);
	this.bpm = 17;
	this.tick = this.bpm;
	this.first = true;
}

Countdown.prototype.update = function(){
	if (this.first){
		this.first = false;
		start_music();
	}

	if (--this.tick == 0){
		this.tick = this.bpm;
		this.counter.update();
	}
}


Countdown.prototype.draw = function(){
	clear();
	this.counter.draw();
}

/**
* Sine
**/ 
function Sine(){
	this.dynamic = false;
	this.wave = new Sin_wave(30,19,0.1,0,4);
	this.text1 = new Text2D("96x54 px", new Vector2D(14,20), 6, 12);
	this.text2 = new Text2D("checkbox", new Vector2D(24,28), 4, 10);
	this.text3 = new Text2D("screen", new Vector2D(31,12), 4, 10);
	this.ticks = 0;
}


/**
* Makes sin wave to change
**/
Sine.prototype.toggle_wave = function(){
	this.dynamic = true;
}


Sine.prototype.update = function(){
	this.wave.offset++;
	if (this.dynamic){
		this.wave.width -= 0.001;
	}
}


Sine.prototype.draw = function(){
	clear();
	this.wave.draw();
	if (++this.ticks > 10 && this.ticks < 80){
		clear_area(new Vector2D(13,19),new Vector2D(84,33));
		this.text1.draw();
	}
	if (this.ticks > 80 && this.ticks < 150){
		clear_area(new Vector2D(23,27),new Vector2D(78,39));
		this.text2.draw();
		clear_area(new Vector2D(30,11),new Vector2D(71,23));
		this.text3.draw();
	}
}


/**
* Stars
**/
function Stars(){
	this.min = 7;
	this.max = 15;
	this.scale = 1.6;
	this.grow = 0.5;
	this.stars = new Array();
	this.add_stars();
	this.bpm = 6;
	this.tick = this.bpm;
	this.ticks = 0;
	this.text1 = new Text2D("if you know", new Vector2D(11,30), 4, 8);
	this.text2 = new Text2D("javascript", new Vector2D(16,15), 4, 8);
	this.text3 = new Text2D("nothing else", new Vector2D(9,30), 4, 8);
	this.text4 = new Text2D("matters", new Vector2D(27,15), 4, 8);
}

/**
* Adds random amount of star in random locations
**/
Stars.prototype.add_stars = function(){
	this.stars = new Array();
	var amount = Math.floor((this.max - this.min - 1)*Math.random()) + this.min;
	for (i=0; i < amount; i++){
		var position = new Vector2D( Math.ceil( 96 * Math.random()), Math.ceil( 54 * Math.random()));
		this.stars[i] = new Star(position, this.scale, this.grow);
	} 
}

Stars.prototype.update = function(){
	if (--this.tick == 0){
		this.tick = this.bpm;
		this.add_stars();
	}
	
	for (i in this.stars){
		this.stars[i].update();
	}
	
}

Stars.prototype.draw = function(){
	clear();
	for (i in this.stars){
		this.stars[i].draw();
	}
	if (++this.ticks > 50 && this.ticks < 110){
		clear_area(new Vector2D(10,29), new Vector2D(85,39));
		this.text1.draw();
		clear_area(new Vector2D(16,14), new Vector2D(82,24));
		this.text2.draw();
	}
	if (this.ticks > 110 && this.ticks < 170){
		clear_area(new Vector2D(8,29), new Vector2D(91,39));
		this.text3.draw();
		clear_area(new Vector2D(26,14), new Vector2D(74,24));
		this.text4.draw();
	}
	
}


/**
*  Cubes
**/
function Cubes(){
	this.cam = new Vector3D(0,0,10);
	this.angle = new Vector3D(0.3,0,0);
	this.piv = new Vector3D(0,0,0);
	this.pos = new Vector3D(0,0,0);
	this.rotation = new Vector3D(0,0.05,0);
	this.cube = new Object3D(this.cam,this.angle,this.piv,this.pos,10,0.1,"cube");
	this.ticks = 250;
	this.tick = this.ticks;
	this.size = 0;
	this.grow = 0.008;
	

}

Cubes.prototype.update = function(){
	if (--this.tick == 0){
		this.ticks = 80;
		this.tick = this.ticks;
		this.rotation = new Vector3D(0.08,0.05,0.04);
		this.grow *= -1;
	}
	
	this.cube.rotate(this.rotation);	
	this.cube.scale += this.grow;
	

}

Cubes.prototype.draw = function(){
	clear();
	this.cube.draw();
}


/**
* Hexagons
**/
function Hexagons(){
	this.hexas = new Hexaweb();
	this.ticks = 12;
	this.tick = this.ticks;
	this.directions = new Array(
		new Vector2D(1,1),
		new Vector2D(1,0),
		new Vector2D(0,1),
		new Vector2D(-1,-1),
		new Vector2D(1,0),
		new Vector2D(0,-1)
	);
	
	this.direction = this.directions[0];
	this.index = 0;
	this.start_scaling = false;
	this.text0 = new Text2D("linkin", new Vector2D(29,34), 4, 8);
	this.text1 = new Text2D("koodauskerho", new Vector2D(8,23), 4, 8);
	this.text2 = new Text2D("<3", new Vector2D(43,12), 4, 8);
	this.text_ticks = 0;
}

Hexagons.prototype.update = function(){
	if (--this.tick == 0){
		this.tick = this.ticks;
		if (++this.index == this.directions.length){
			this.index = 0;
			this.start_scaling = true;
		}
		this.direction = this.directions[this.index ];
	}
	if (this.start_scaling){
		this.hexas.update(this.directions[this.index].x *0.5, this.directions[this.index].y*0.5, new Vector2D(0,0));
	}
	else {
		this.hexas.update(0,0, this.direction);
	}
}

Hexagons.prototype.draw = function(){
	clear();
	this.hexas.draw();
	if (++this.text_ticks > 10 && this.text_ticks < 230){
		clear_area( new Vector2D(28,32), new Vector2D(67,43));
		this.text0.draw();
		clear_area( new Vector2D(7,22), new Vector2D(90,32));
		this.text1.draw();
		clear_area( new Vector2D(42,11), new Vector2D(55,22));
		this.text2.draw();
	}
}


/**
* Credits jypa
**/
function Credits_jypa(){
	
	var code = new Array(
		[3,0],[7,6],[15,4],[7,2],[3,0],[3,4],[3,2],
		[7,2],[15,0],[7,6],[15,4],[10,2],
		[15,0],[3,2],[4,3],[7,4],[4,5],[3,6],[9,2],
		[7,2],[7,6],[7,0],[4,2],[1,0],[4,6],[7,0],[7,2]
	);

	var dot = new Array(
		[1,2],[1,4],[1,6]
	);

	var jypa = new Array(
		[2,4],[7,2],[15,0],[3,2],
		[4,3],[11,4],[11,0],[4,1],[3,2],
		[15,4],[7,0],[3,2],[3,1],[2,0],[3,7],[3,6],[9,2],
		[15,4],[7,0],[7,2],[8,0],[7,6],[7,2],[15,4]
	);


	this.code_jypa = new Array(
		new Tron_text(new Vector2D(12,42),code),
		new Tron_text(new Vector2D(44,34),dot),
		new Tron_text(new Vector2D(44,40),dot),
		new Tron_text(new Vector2D(4,12),jypa)
	);
	
	this.first = true;
	this.index = 0;
	
	this.ticks = 2;
	this.tick = this.ticks;
	
	
	this.cam = new Vector3D(0,0,10);
	this.piv = new Vector3D(0,0,0);
	this.rotation = new Vector3D(0,0.05,0);
	this.pyramid1 = new Object3D(this.cam,new Vector3D(0.17,0,0),this.piv,new Vector3D(26,3,0),12,1,"pyramid");
	this.pyramid2 = new Object3D(this.cam,new Vector3D(3,0,0),this.piv,new Vector3D(26,-3,0),12,1,"pyramid");
}

Credits_jypa.prototype.update = function(){
	if (this.first){
		clear();
		this.first = false;
	}
	if (--this.tick == 0){
		this.tick = this.ticks;
		if ( this.index < this.code_jypa.length){
			if ( this.code_jypa[this.index].update() == 1){
				this.index++;
			}
		}
	}
	
	this.pyramid1.rotate(this.rotation);
	this.pyramid2.rotate(this.rotation);
}

Credits_jypa.prototype.draw = function(){
	clear_area(new Vector2D(56,1), new Vector2D(95,54));
	this.pyramid1.draw();
	this.pyramid2.draw();
}


/**
* Credits zipola
**/
function Credits_zipola(){
	
	var music = new Array(
		[15,0],[3,3],[3,1],[15,4],[3,2],
		[15,0],[15,4],[6,2],[15,0],[15,4],[3,2],
		[6,2],[10,0],[6,6],[5,0],[10,2],
		[15,4],[4,2],
		[7,2],[7,6],[15,0],[7,2]
	);

	var dot = new Array(
		[1,2],[1,4],[1,6]
	);

	var zipola = new Array(
		[12,2],[2,4],[11,5],[1,6],[2,4],[12,2],[3,2],
		[15,0],[4,2],
		[15,4],[10,0],[6,2],[5,0],[6,6],[9,2],
		[6,2],[15,4],[6,6],[15,0],[9,2],
		[15,4],[7,2],
		[15,0],[6,2],[5,4],[6,6],[6,2],[10,4]
	);


	this.text = new Array(
		new Tron_text(new Vector2D(4,30),music),
		new Tron_text(new Vector2D(45,34),dot),
		new Tron_text(new Vector2D(45,41),dot),
		new Tron_text(new Vector2D(4,25),zipola)
	);
	
	this.first = true;
	this.index = 0;
	
	this.ticks = 2;
	this.tick = this.ticks;
	
	
	this.cam = new Vector3D(0,0,10);
	this.angle = new Vector3D(0.3,0,0);
	this.piv = new Vector3D(0,0,0);
	this.pos = new Vector3D(30,0,0);
	this.rotation = new Vector3D(0.02,0.05,0.03);
	
	this.cube = new Object3D(this.cam,this.angle,this.piv,this.pos,10,1,"cube");
}

Credits_zipola.prototype.update = function(){
	if (this.first){
		clear();
		this.first = false;
	}
	
	if (--this.tick == 0){
		this.tick = this.ticks;
		if ( this.index < this.text.length){
			if ( this.text[this.index].update() == 1){
				this.index++;
			}
		}
	}
	
	this.cube.rotate(this.rotation);
}

Credits_zipola.prototype.draw = function(){
	clear_area(new Vector2D(56,1), new Vector2D(95,54));
	this.cube.draw();
}


/**
* Springs
**/
function Springs(){
	this.springs = new Array(
		new Sine_spring(9,1,0),
		new Sine_spring(22,5,1),
		new Sine_spring(34,9,2),
		new Sine_spring(47,13,3),
		new Sine_spring(60,17,4),
		new Sine_spring(73,21,5),
		new Sine_spring(86,24,6)
	);
}

Springs.prototype.update = function(){
	for (i in this.springs){
		this.springs[i].update();
	}
}


Springs.prototype.draw = function(){
	clear();
	for (i in this.springs){
		this.springs[i].draw();
	}
}

/**
* Triforce
**/
function Triforce(){
	this.cam = new Vector3D(0,0,10);
	this.angle = new Vector3D(0.15,0,0);
	this.piv = new Vector3D(0,0,0);
	this.rotation = new Vector3D(0,0.06,0);
	this.pyramids = new Array(
		new Object3D(this.cam,this.angle,this.piv,new Vector3D(2,3,0),16,1,"pyramid"),
		new Object3D(this.cam,this.angle,this.piv,new Vector3D(23,-19,0),15,1,"pyramid"),
		new Object3D(this.cam,this.angle,this.piv,new Vector3D(-21,-19,0),15,1,"pyramid")
	);
	this.text1 = new Text2D("egyptian", new Vector2D(22,30), 4, 8);
	this.text2 = new Text2D("triforce", new Vector2D(22,18), 4, 8);
	this.ticks = 0;
}

Triforce.prototype.update = function(){
	for (i in this.pyramids){
		this.pyramids[i].rotate(this.rotation);
	}
}


Triforce.prototype.draw = function(){
	clear();
	for (i in this.pyramids){
		this.pyramids[i].draw();
	}
	if (++this.ticks > 30 && this.ticks <80){
		clear_area(new Vector2D(21,28),new Vector2D(81,39));
		this.text1.draw();
		clear_area(new Vector2D(21,17),new Vector2D(76,28));
		this.text2.draw();
	}
}


/**
* Hypercube
**/
function Hypercube(){
	this.cam = new Vector3D(0,0,10);
	this.angle = new Vector3D(0.3,0,0);
	this.piv = new Vector3D(0,0,0);
	this.pos = new Vector3D(35,7,0);
	this.rotation = new Vector3D(0.04,0.06,0.08);
	this.cube = new Object3D(this.cam,this.angle,this.piv,this.pos,8,1,"hyper");
	this.ticks = 80;
	this.tick = this.ticks;
	this.texts = new Array(
		new Text2D("kilpikon3", new Vector2D(4,41), 4, 10),
		new Text2D("naapurinpoika", new Vector2D(4,4), 4, 10),
		new Text2D("louk", new Vector2D(4,25), 4, 10),
		new Text2D("thanks for", new Vector2D(3,42), 4, 9),
		new Text2D("watching", new Vector2D(4,10), 4, 9)
	);
	this.text = this.texts[0];
	this.index = 0;
}

Hypercube.prototype.update = function(){
	this.cube.rotate(this.rotation);
	if (--this.tick == 0){
		this.tick = this.ticks;
		this.text = this.texts[++this.index];
	}
}


Hypercube.prototype.draw = function(){
	clear();
	this.cube.draw();
	this.text.draw();
}


// dummy:


function Dummy(){

}

Dummy.prototype.update = function(){

}


Dummy.prototype.draw = function(){
	clear();
}


// < EFFECTS />

// < SCRIPT >


//EPIC VARIABLES:


//Effects

var effects = new Array(
	new Dummy,
	new Countdown,
	new Sine,
	new Stars,
	new Cubes,
	new Hexagons,
	new Credits_jypa,
	new Credits_zipola,
	new Springs,
	new Triforce,
	new Hypercube,
	new Dummy
);

var effect_index = 0;
var effect = effects[0];

/**
* Changes the effect
**/
function next(){
	effect = effects[++effect_index];
}

/**
* Starts changing the wave
**/
function toggle_wave(){
	effects[2].toggle_wave();
}

/**
* Demo logic here
**/
function update(){
	effect.update();
}

/**
* Drawing here
**/ 
function draw(){
	effect.draw();
}

/**
* Application loop
**/ 
function application_loop(){
	update();
	draw();
}

/**
* Runs on start
* initializing and timing here
**/
window.onload = function() {
	bgColor('white');
	window.setInterval(application_loop, 10);
	window.setTimeout(next,3000);	//countdown
	window.setTimeout(next,11000); //sine
	window.setTimeout(toggle_wave,26000);
	window.setTimeout(next,63000); //stars
	window.setTimeout(next,77500); //cube 
	window.setTimeout(next,107000); //hexas
	window.setTimeout(next,136500); //c_jypa
	window.setTimeout(next,151000); //c_zipola
	window.setTimeout(next,166000);	//springs
	window.setTimeout(next,181000); //triforce
	window.setTimeout(next,196000); //hyper
	window.setTimeout(next,225000); //end
}

// < SCRIPT />