/*

 Name      :  Hardware VGM player
 Hardware  :  YM2612 & SN76489
 Software  :  VGM 1.5 Spec
 Notes     :  Music Playback using sound chips from sega genesis
 
 references:
 http://www.smspower.org/Development/VGMFormat
 http://en.wikipedia.org/wiki/Yamaha_YM2612
 http://softsolder.wordpress.com/2009/07/18/arduino-hardware-assisted-spi-synchronous-serial-data-io/
 
 */

unsigned long startTime;                // keep track of elapsedTime for pause command
unsigned long duration;

byte buffer[BUFFER_LENGTH];     // VGM instruction buffer
byte bufferCounter=0;           // how many bytes have we chewed off buffer

void init_vgm(){  
  writeFM(B00000000, B000011);  // initialize YM2612 registers, system reset
  silence();                    // turn down the volume on all FM and PSG 
}

// fill the sound buffer
void fillBuffer (){
  GHI_PutS("R 10>0000001E\r"); // read 30 bytes (0x1E) from file handle #1  
  GHI_GetResponse(4);          // handle !00 response

  // fill the buffer with data
  for (byte i = 0; i <BUFFER_LENGTH; i++)
  {
    buffer[i]=GHI_GetC();     // fill buffer one byte at time
  } 

  GHI_GetResponse(14);        // handle $00000000<CR>!00<CR> response
}

void loadVGM(char * musicfile){ 
  silence();                  // mute all channels
  GHI_PutS("C 1\r");          // close file #1 just in case
  GHI_GetResponse(4);         // handle !00 response 
  printResult("\nopen file");  
  GHI_PutS(musicfile);        // send open file instruction to module
  GHI_GetResponse(4);         // handle !00 response  
  printResult("\ndone opening file");
  bufferCounter=0;
  fillBuffer();               // preload buffer
  readVGMHeader();            // get past the 64 byte header
}

void vgm(){
  // if sound needs a delay skip VGM processing
  if((micros()-startTime) <= duration){
     return;   // instead of waiting do other work
  }

  byte command = getByte();

  switch (command) {

  case VGM_CMD_GG_STEREO:  // Game Gear PSG stereo, write dd to port 0x06        
    writePSG(getByte());
    break;  

  case VGM_CMD_PSG:        // PSG (SN76489/SN76496) write value dd
    writePSG(getByte());
    break;

  case VGM_CMD_YM2413:     // YM2413, write value dd to register aa        
    getByte(); 
    getByte(); 
    break;

  case VGM_CMD_YM2612_0:   // YM2612 port 0, write value dd to register aa        
    // null, A1, A0, WR, RD, CS, CE, WE 
    writeFM(getByte(),  B10001011); // select part 1 register 
    writeFM(getByte(),  B10101011); // write data to selected part 1 register   
    break;

  case VGM_CMD_YM2612_1:   // YM2612 port 1, write value dd to register aa
    // null, A1, A0, WR, RD, CS, CE, WE 
    // checkFMBusy(B11110111); // reads status of part2
    // checkFMBusy(B11110011); // reads status of part1
    writeFM(getByte(),  B11001011); // select part 2 register
    writeFM(getByte(),  B11101011); // write data to selected part 1 register  
    break;

  case VGM_CMD_2151:      // YM2151, write value dd to register aa
    getByte(); 
    getByte(); 
    break;

  case VGM_CMD_WAIT:      // 0x61 nn nn : Wait n samples, n can range from 0 to 65535 (approx 1.49)
    pause(read16());
    break;

  case VGM_CMD_WAIT_735:  // wait 735 samples (60th of a second), a shortcut for 0x61 0xdf 0x02
    pause(735);
    break;

  case VGM_CMD_WAIT_882:  //  wait 882 samples (50th of a second), a shortcut for 0x61 0x72 0x03
    pause(882);
    break;

  case VGM_PAUSE_BYTE:    //  wait 882 samples (50th of a second), a shortcut for 0x61 0x72 0x03
    pause(882);
    break;

  case VGM_CMD_EOF:       //  end of sound data
    //        silence();
    break;

  case VGM_CMD_DATA_BLOCK: //data block
    getByte(); // compatibility command
    getByte(); // datatype
    // long datalength = read32(); // length of data
    // dataYM2612 = getByte();     // 0x66 = compatibility command to make older players stop parsing the stream
    // dataYM2612 = getByte();     // data type 

    // loop through size of data, in bytes 
    // for (long datablock = 0; datablock <= datalength; datablock++) {
    // store this data into an SRAM Chip
    // dataYM2612 = getByte();     // eat it up
    //}       
    break;

  case 0x70: // no-data wait n+1 samples, n can range from 0 to 15
  case 0x71: 
  case 0x72:
  case 0x73: 
  case 0x74: 
  case 0x75: 
  case 0x76: 
  case 0x77: 
  case 0x78: 
  case 0x79: 
  case 0x7A: 
  case 0x7B: 
  case 0x7C: 
  case 0x7D: 
  case 0x7E: 
  case 0x7F: 
    //pause((command & 0xf)+1);
    break;

    //PCM data bank write then short wait
  case 0x80: // YM2612 port 0 address 2A write from the data bank wait n+1 samples, n can range from 0 to 15
  case 0x81:
  case 0x82: 
  case 0x83: 
  case 0x84: 
  case 0x85: 
  case 0x86: 
  case 0x87: 
  case 0x88: 
  case 0x89: 
  case 0x8A: 
  case 0x8B: 
  case 0x8C: 
  case 0x8D: 
  case 0x8E: 
  case 0x8F: 
    //      writeFM(0x2A, B10001011); // select part 1 register 
    //      writeFM(0x2A, B10101011); // write data to selected part 1 register        
    pause((command & 0xf));
    break;

  case VGM_CMD_YM2612_PCM_SEEK: // seek to offset dddddddd (Intel byte order) in PCM data bank   
    read32();
    break;

  // if nothing else matches, do the default
  default:
    break; 
  }  
}

// delayMicroseconds compatible with RTOS
void delayMicroRTOS(unsigned int ms){
  unsigned long start = micros();
  while (micros() - start <= ms); // do other tasks while you wait
}

// delayMicroseconds not compatible with RTOS
void delayMicro(unsigned int ms){
  do{
    delayMicroseconds(1000);      // halt all program execution
  }
  while (--ms);
}

// how long should we wait  
void pause(long samples){  
  duration = ((1000.0 / (44100.0/(float)samples))*1000);  
  startTime = micros();  
}

//read 64 byte header
void readVGMHeader(){
  VGMHeader.VGMIdent[0] = char(getByte()); // V
  VGMHeader.VGMIdent[1] = char(getByte()); // G
  VGMHeader.VGMIdent[2] = char(getByte()); // M
  VGMHeader.VGMIdent[3] = char(getByte());

  VGMHeader.EoFOffset = read32() + 4;

  long val = read32();
  unsigned char  buff[4];
  buff[0] = toHex( (unsigned char)val >> 16 );
  buff[1] = toHex( (unsigned char)(val >> 8));
  buff[2] = toHex( (unsigned char)(val >> 4));
  buff[3] = toHex( (unsigned char)(val));
  VGMHeader.Version =  ((int(buff[0])-48) * 1000) + ( (int(buff[1])-48) * 100) + ( (int(buff[2])-48) * 10) + (int(buff[3])-48);

  VGMHeader.PSGClock       = read32();
  VGMHeader.YM2413Clock    = read32();
  VGMHeader.GD3Offset      = read32() - 4;  // 4 = size of VGMIdent
  VGMHeader.TotalLength    = read32();
  VGMHeader.LoopOffset     = read32();
  VGMHeader.LoopLength     = read32();
  VGMHeader.RecordingRate  = read32();
  VGMHeader.SNFB           = read16();
  VGMHeader.SNW            = int(getByte());
  getByte();
  VGMHeader.YM2612Clock    = read32();
  VGMHeader.YM2151Clock    = read32();
  VGMHeader.DataOffset     = read32();
  read32();
  read32();

  // todo: seek to file VGMHeader.GD3Offset and get GD3 Data

  printResult("VGM marker           ");  
  printResult(VGMHeader.VGMIdent[0]);
  printResult(VGMHeader.VGMIdent[1]);
  printResult(VGMHeader.VGMIdent[2]);
  printResult(VGMHeader.VGMIdent[3]);

  printResult("End-of-file offset   ");  
  printResult(VGMHeader.EoFOffset); 

  printResult("VGM version          "); 
  printResult(VGMHeader.Version);

  printResult("PSG SN76489 speed    ");   
  printResult(VGMHeader.PSGClock);

  printResult("YM2413 speed         ");
  printResult(VGMHeader.YM2413Clock);

  printResult("GD3 tag offset       ");
  printResult(VGMHeader.GD3Offset);

  printResult("Total # of samples   ");
  printResult(VGMHeader.TotalLength); 

  printResult("Loop Offset          ");
  printResult(VGMHeader.LoopOffset);

  printResult("Loop # samples       ");
  printResult(VGMHeader.LoopLength);

  printResult("Rate                 ");
  printResult(VGMHeader.RecordingRate);

  printResult("SN FB                ");
  printResult(VGMHeader.SNFB);

  printResult("SNW                  ");
  printResult(VGMHeader.SNW);

  printResult("2612 clock           ");
  printResult(VGMHeader.YM2612Clock);

  printResult("2151 clock           ");
  printResult(VGMHeader.YM2151Clock);

  printResult("VGM Data Offset      ");
  printResult(VGMHeader.DataOffset);
}

unsigned int read16(){
  return getByte() + (getByte() << 8);  // read a 16-bit number from 2 bytes
}

long read32(){  
  long v0 = long(getByte()) ;
  long v1 = long(getByte());  
  long v2 = long(getByte());  
  long v3 = long(getByte());
  return v0 + (v1 << 8) + (v2 << 16) + (v3 << 24);
}

// get VGM data from buffer
byte getByte(){
  if (bufferCounter==BUFFER_LENGTH){
    fillBuffer();
    bufferCounter=0;
  }
  return buffer[bufferCounter++];
}

// FM 8-bit write to data bus
void writeFM(volatile char dataYM2612,volatile  char control){
  latchOff();
  spi_transfer_nodelay(dataYM2612);  
  spi_transfer_nodelay(control);  // null, A1, A0, WR, RD, CS, CE, WE 
  latchOn();
  delayMicroseconds(2);
}

// psg 8-bit write to data bus
void writePSG(byte dataSN76489){
  latchOff();
  spi_transfer_nodelay(dataSN76489);  
  spi_transfer_nodelay(B10000100);  // null, A1, A0, WR, RD, CS, CE, WE 
  latchOn();
  
  delayMicroseconds(5);
  
  latchOff();
  spi_transfer_nodelay(dataSN76489);  
  spi_transfer_nodelay(B10000111);  
  latchOn();
}

// silence all channels
void silence(){
  // PSG chip - set lowest volume on four psg channels
  writePSG(0x9F); // Channel 1 - 10011111
  writePSG(0xBF); // Channel 2 - 10111111
  writePSG(0xDF); // Channel 3 - 11011111
  writePSG(0xFF); // Channel 4 - 11111111

  // YM2612 chip - set lowest volume on six channels
  writeFM(0x22, B10001011); 
  writeFM(0x27, B10001011); 
  writeFM(0,    B10101011); 
  writeFM(0x28, B10001011); 
  writeFM(1,    B10101011); 
  writeFM(0x28, B10001011); 
  writeFM(2,    B10101011); 
  writeFM(0x28, B10001011); 
  writeFM(0,    B10101011); 
  writeFM(0x28, B10001011); 
  writeFM(4,    B10101011); 
  writeFM(0x28, B10001011); 
  writeFM(5,    B10101011); 
  writeFM(0x28, B10001011); 
  writeFM(6,    B10101011); 
  writeFM(0x2B, B10001011); 
  writeFM(0,    B10101011); 
}

// convert number to hexadecimal 255 = FF
static char toHex(int h){
  h &= 0xf;
  if( h < 10 ) 
    return (h + '0');
  return (h - 10 + 'A');
}

// enable this to debug
void printResult(char str){
//     Serial.print(str);
}

void printResult(char * str){
//     Serial.println(str);
}

// convert on HEX characeter into decimal number
int hex2dec(byte c){
  if (c >= '0' && c <= '9') {
    return c - '0';
  } 
  else if (c >= 'A' && c <= 'F') {
    return c - 'A' + 10;
  }
}

// play music for x milliseconds
void play(unsigned long length){
  unsigned long start = millis();
  while (millis() - start <= length){
    vgm();
  }
}

// faster than digital write
void dataOff(){
  bitClear(PORTB,MOSI_PIN_PORTB);
}

void dataOn(){
  bitSet(PORTB,MOSI_PIN_PORTB);
}

void clockOff(){
  bitClear(PORTB,SCK_PIN_PORTB);
}

void clockOn(){
  bitSet(PORTB,SCK_PIN_PORTB);
}

void latchOn(){
  bitSet(PORTB,VGM_SS_PIN_PORTB);
}

void latchOff(){
  bitClear(PORTB,VGM_SS_PIN_PORTB);
}

void slaveOn(){
  bitSet(PORTB,FAT_SS_PIN_PORTB);
}

void slaveOff(){
  bitClear(PORTB,FAT_SS_PIN_PORTB);
}

