package org.shiftone.jrat.util.log;

import org.shiftone.jrat.core.JRatException;
import org.shiftone.jrat.core.Settings;
import org.shiftone.jrat.util.Assert;
import org.shiftone.jrat.util.Command;
import org.shiftone.jrat.util.log.target.LogTarget;
import org.shiftone.jrat.util.log.target.NullLogTarget;
import org.shiftone.jrat.util.log.target.ProxyLogTarget;
import org.shiftone.jrat.util.log.target.TandemTarget;
import org.shiftone.jrat.util.log.target.ThreadLocalLogTarget;
import org.shiftone.jrat.util.log.target.WriterLogTarget;
import java.io.PrintWriter;

/**
 * There are currently 3 ways that logging can be configured...
 * <li>turned off - using NullLogTarget
 * <li>logging to a PrintWriter - System.out or file
 * <li>using thread based logging - each thread can have it's own LogTarget
 * (this is for the Desktop)
 * 
 * @author $Author: jeffdrost $
 * @version $Revision: 1.5 $
 */
public class LoggerFactory implements Constants {

	private static final NullLogTarget NULL_LOG_TARGET = NullLogTarget.INSTANCE;
	private static final WriterLogTarget SYSTEM_OUT_TARGET = new WriterLogTarget(System.out);
	private static final ProxyLogTarget PROXY_LOG_TARGET = new ProxyLogTarget(SYSTEM_OUT_TARGET);
	private static final ThreadLocalLogTarget THREAD_TARGET = new ThreadLocalLogTarget(SYSTEM_OUT_TARGET);

	public static void initialize() {

		// try {
		setLevel(getLevelFromName(Settings.getLogLevel()));
		// } catch (Exception e) {}
	}

	public static Logger getLogger(Class klass) {

		String className = klass.getName();
		String shortName = className.substring(className.lastIndexOf('.') + 1);
		return getLogger(shortName);
	}

	public static Logger getLogger(String topic) {

		return new Logger(topic, PROXY_LOG_TARGET);
	}

	public static int getLevelFromName(String levelName) {

		Assert.assertNotNull("levelName", levelName);
		Assert.assertNotNull("LEVEL_NAMES", LEVEL_NAMES);
		levelName = levelName.toUpperCase();
		for (int i = 0; i < LEVEL_NAMES.length; i++) {
			if (levelName.equals(LEVEL_NAMES[i])) {
				return i;
			}
		}
		throw new JRatException("log level '" + levelName + "' is not known");
	}

	public static void setLevel(int level) {

		PROXY_LOG_TARGET.setCurrentLevel(level);
	}

	public static int getLevel() {

		return PROXY_LOG_TARGET.getCurrentLevel();
	}

	public static void disableLogging() {

		PROXY_LOG_TARGET.setLogTarget(NULL_LOG_TARGET);
	}

	public static void enableThreadBasedLogging() {

		PROXY_LOG_TARGET.setLogTarget(THREAD_TARGET);
	}

	public static void enableSystemOutLogging() {

		PROXY_LOG_TARGET.setLogTarget(SYSTEM_OUT_TARGET);
	}

	public static void redirectLogging(PrintWriter printWriter) {

		TandemTarget tandemTarget = new TandemTarget(SYSTEM_OUT_TARGET, new WriterLogTarget(printWriter));
		PROXY_LOG_TARGET.setLogTarget(tandemTarget);
	}

	/**
	 * this will only have any effect on logging if the current mode is using
	 * the ThreadLocalLogTarget - meaning a call to enableDesktopLoggingMode was
	 * made.
	 */
	public static Object executeInThreadScope(LogTarget newTarget, Command command) {

		return THREAD_TARGET.executeInScope(newTarget, command);
	}
}
