using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Audio;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.GamerServices;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Input;
using Microsoft.Xna.Framework.Media;

using Game.Entidades;
using Game.Servicios;  

namespace Game.Graficos
{
    /// <summary>
    /// This is a game component that implements IUpdateable.
    /// </summary>
    public class JugadorProcessor : AbstractProcessor
    {
        private Matrix world_matriz = Matrix.Identity;
        private Matrix centrado_matriz = Matrix.CreateTranslation(Parametros.TRANS_CENTRADO);

        private Viewport vista; 

        private Vector3 posicion_objetivo; 

        private GameScene Main;

        private Contador contador; 

        private Matriz matriz; 
        private Jugador jugador;

        // IA
        List<Cubo> cubosExaminar = new List<Cubo>();
        List<Cubo> cubosExaminados = new List<Cubo>();
        private Boolean IA_Activada;
        private int IA_Dificultad; 

        // Velocidad caida piezas
        private float velocidadReferenciaCaida; 

        private ControlProcessor control; 
        private PlayerIndex player;

        public JugadorProcessor(DefaultScene game, PlayerIndex player, int IA_Dificultad)
            :this( game, player )
        {
            // IA
            this.IA_Activada = true;
            this.IA_Dificultad = IA_Dificultad; 
            // Velocidad caida
            velocidadReferenciaCaida = Parametros.IA_VELOCIDAD_CAIDA[IA_Dificultad];
            //velocidadReferenciaCaida -= velocidadReferenciaCaida / 4.0f; 
        }

        public JugadorProcessor(DefaultScene game, PlayerIndex player )
            : base(game)
        {
            // clase base
            this.Main = (GameScene)game;
            // Jugador
            this.player = player;
            // control
            this.control = Main.control[(int)player]; 
            // Matriz
            this.matriz = Main.matrices[(int)player]; 
            // Datos
            this.jugador = Main.jugadores[(int)player]; 
            // Vista 
            this.vista = Main.vistas[(int)player];
            // Contador de efecto de detonadores
            this.contador = new Contador(50, 4);
            // IA
            IA_Activada = false;  
            // Velocidad Caida
            velocidadReferenciaCaida = Parametros.JUGADOR_VELOCIDAD_CAIDA; 
        }

        public override void Initialize()
        {
        }

        public override void LoadContent()
        {
        }

        public override void UnloadContent()
        {
        }

        #region Logica

        public override void Update(GameTime gameTime)
        {
            if (IA_Activada)
                UpdateIA(gameTime);
            else
                UpdateHUmana(gameTime);

            // Calculo de transformaciones
            this.world_matriz = centrado_matriz * Matrix.CreateRotationY(matriz.rotacion);
            this.posicion_objetivo = Vector3.Transform(Vector3.Zero, Matrix.CreateTranslation(new Vector3(jugador.X, 7.5f, jugador.Z)) * world_matriz);
        }

        private void UpdateHUmana(GameTime gameTime)
        {            
            int dX = 0;
            int dZ = 0;
            int mX = 0;
            int mZ = 0;
            // Control de movimiento del jugador. 
            if ( control.Fondo ) mZ = -1; 
            if ( control.Frente ) mZ = +1; 
            if ( control.Izquierda ) mX = -1; 
            if ( control.Derecha ) mX = +1;

            // Control de rotacion           
            if (this.matriz.rotacion > MathHelper.ToRadians(315) || 
                this.matriz.rotacion <= MathHelper.ToRadians(45)) {
                dX = mX;
                dZ = mZ; 
            } else if (this.matriz.rotacion > MathHelper.ToRadians(45) && 
                this.matriz.rotacion <= MathHelper.ToRadians(135)) {
                dX = -mZ;
                dZ = mX; 
            } else if (this.matriz.rotacion > MathHelper.ToRadians(135) && 
                this.matriz.rotacion <= MathHelper.ToRadians(225)) {
                dX = -mX;
                dZ = -mZ; 
            } else if (this.matriz.rotacion > MathHelper.ToRadians(225) && 
                this.matriz.rotacion <= MathHelper.ToRadians(315)) {
                dX = mZ;
                dZ = -mX; 
            }
            // Limitaciones de movimiento
            if (dX != 0 || dZ != 0)
            {
                if (jugador.X + dX >= 0 && jugador.X + dX < Parametros.MAX_X &&
                    jugador.Z + dZ >= 0 && jugador.Z + dZ < Parametros.MAX_Z)
                {
                    float alturaColumna = this.matriz.ObtenerPosado(jugador.X + dX, jugador.Z + dZ);
                    if (jugador.Cubo.PosicionBase > alturaColumna) jugador.MoverHorizontal(dX, dZ);
                }
            }

            // Control de posado de cubo jugador.
            if (matriz.ComprobarPosar(jugador.Cubo) || control.Posar) {
                Main.audio.Play(Parametros.SONIDO_COLISION);
                matriz.Posar(jugador.Cubo);
                jugador.NuevoCubo(matriz.ObtenerTotalCubos());
                // Actualizacion de control de transferencia
                UpdateTransferencia();
            } else {
                //velocidadDinamicaCaida = velocidadReferenciaCaida + ((Constantes.IA_VELOCIDAD_CAIDA[IA_Dificultad]/2.0f) * (matriz.ObtenerTotalCubos() / Constantes.MAX_CUBOS_MATRIX));
                jugador.MoverVertical(velocidadReferenciaCaida * gameTime.ElapsedGameTime.Milliseconds);
            }
        }

        private void UpdateIA(GameTime gameTime)
        {
            // Control de posado de cubo jugador.
            if (matriz.ComprobarPosar(jugador.Cubo))
            {
                Main.audio.Play(Parametros.SONIDO_COLISION);
                matriz.Posar(jugador.Cubo);
                jugador.NuevoCubo(matriz.ObtenerTotalCubos());

                    int X, Z;
                    CalcularMovimiento(jugador.Cubo.Color, jugador.Cubo.EsDetonador, out X, out Z);
                    jugador.Posicionar(X, Z);

                // Actualizacion de control de transferencia
                UpdateTransferencia();
            }
            else
            {
                //velocidadDinamicaCaida = velocidadReferenciaCaida + ((Constantes.IA_VELOCIDAD_CAIDA[IA_Dificultad]/2.0f) * (matriz.ObtenerTotalCubos() / Constantes.MAX_CUBOS_MATRIX));
                jugador.MoverVertical(velocidadReferenciaCaida * gameTime.ElapsedGameTime.Milliseconds);
            }
        }

        private void UpdateTransferencia()
        {
            int colX, colZ;
            Cubo cubo;
            int color; 
            // Obtencin de transferencia del jugador contrario
            Transferencia transferenciaPropia = Main.transferencias[(int)this.player];
            Transferencia transferenciaAjena = (this.player == PlayerIndex.One) ? 
                                                Main.transferencias[(int)PlayerIndex.Two]: 
                                                Main.transferencias[(int)PlayerIndex.One];
            // Comprobacion de cubos pendientes de ser enviados. 
            if (transferenciaAjena.TotalCubosSalida > 0)
            {
                for (int cubos = 0; cubos < 10; cubos++)
                {
                    color = transferenciaAjena.Recibir();
                    // Obtiene coordenadas de columna con menor numero de cubos. 
                    matriz.ObtenerColumnaMenor(out colX, out colZ);
                    // Generacion de cubo transferi1do
                    cubo = Cubo.GenerarCuboTransferencia(color);
                    // Posicionar cubo5
                    cubo.MoverPosicionHorizontal(colX, colZ);
                    // Posar cubo
                    matriz.Posar(cubo);
                    // Sonido de caida tras el ultimo cubo.

                    if (transferenciaAjena.TotalCubosSalida == 0)
                    {
                        this.Main.audio.Play(Parametros.SONIDO_RECEPCION);
                        break; // NO QUEDAN MAS CUBOS
                    }
                }

            }
        }

        #endregion 

        #region IA

        // Calculo de movimiento.
        private void CalcularMovimiento(int color, bool detonador, out int X, out int Z)
        {
            Z = 0;
            X = 0; 
            int valoracionMovimiento = 0;
            int valoracionPosicion = 0; 
            // Bucle de valoracion de posiciones 
            for (int posX = 0; posX < Parametros.MAX_X; posX++)
            {
                for (int posZ = 0; posZ < Parametros.MAX_Z; posZ++)
                {
                    valoracionPosicion = ValorarMovimiento( posX, posZ, color );
                    valoracionPosicion += (detonador) ? matriz.ObtenerColorColumna(posX, posZ, color) : 0;
                    valoracionPosicion *= (Parametros.MAX_Y - matriz.ObtenerAltura(posX, posZ));
                    if (valoracionPosicion > valoracionMovimiento )
                    {
                        X = posX;
                        Z = posZ;
                        valoracionMovimiento = valoracionPosicion; 
                    }
                }
            }
 
            // Ningun movimiento beneficioso detectado.
            if (valoracionMovimiento == 0)
            {
                X = new Random().Next(0, Parametros.MAX_X);
                Z = new Random().Next(0, Parametros.MAX_Z);
            }
        }

        // Valoracion de contiguidad
        private int ValorarMovimiento(int X, int Z, int color)
        {
            cubosExaminar.Clear();
            cubosExaminados.Clear();
            Cubo cuboAnalisis; 
            cubosExaminar.Add(new Cubo(color, false) { MatrizPosicion = new Vector3(X, matriz.ObtenerAltura(X, Z), Z) });
            while (cubosExaminar.Count > 0)
            {
                int _X = (int)cubosExaminar[0].MatrizPosicion.X;
                int _Z = (int)cubosExaminar[0].MatrizPosicion.Z;
                int _Altura = (int)cubosExaminar[0].MatrizPosicion.Y;
                // propagacion arriba
                cuboAnalisis = matriz.ObtenerCubo(_X, _Z, _Altura+1);
                if (cuboAnalisis != null && cuboAnalisis.Color == color && !cubosExaminados.Contains(cuboAnalisis) && !cubosExaminar.Contains(cuboAnalisis)) cubosExaminar.Add(cuboAnalisis);
                // propagacion abajo
                cuboAnalisis = matriz.ObtenerCubo(_X, _Z, _Altura-1);
                if (cuboAnalisis != null && cuboAnalisis.Color == color && !cubosExaminados.Contains(cuboAnalisis) && !cubosExaminar.Contains(cuboAnalisis)) cubosExaminar.Add(cuboAnalisis);
                // propagacion izquierda
                cuboAnalisis = matriz.ObtenerCubo(_X-1, _Z, _Altura);
                if (cuboAnalisis != null && cuboAnalisis.Color == color && !cubosExaminados.Contains(cuboAnalisis) && !cubosExaminar.Contains(cuboAnalisis)) cubosExaminar.Add(cuboAnalisis);
                // propagacion derecha
                cuboAnalisis = matriz.ObtenerCubo(_X+1, _Z, _Altura + 1);
                if (cuboAnalisis != null && cuboAnalisis.Color == color && !cubosExaminados.Contains(cuboAnalisis) && !cubosExaminar.Contains(cuboAnalisis)) cubosExaminar.Add(cuboAnalisis);
                // propagacion fondo
                cuboAnalisis = matriz.ObtenerCubo(_X, _Z-1, _Altura + 1);
                if (cuboAnalisis != null && cuboAnalisis.Color == color && !cubosExaminados.Contains(cuboAnalisis) && !cubosExaminar.Contains(cuboAnalisis)) cubosExaminar.Add(cuboAnalisis);
                // propagacion frente
                cuboAnalisis = matriz.ObtenerCubo(_X, _Z+1, _Altura + 1);
                if (cuboAnalisis != null && cuboAnalisis.Color == color && !cubosExaminados.Contains(cuboAnalisis) && !cubosExaminar.Contains(cuboAnalisis)) cubosExaminar.Add(cuboAnalisis);
                // Registro de cubo examinado
                cubosExaminados.Add(cubosExaminar[0]);
                cubosExaminar.RemoveAt(0);
            }
            // Retorna numero de cubos contiguos del mismo color
            // Resultado = Cubos examinados ( - hipotetico ) * altura. 
            return (cubosExaminados.Count - 1);
        }

        #endregion 

        #region Visualizacion

        public override void Draw(GameTime gameTime)
        {
            // Actualizacion de contador de efecto de detonadores
            this.contador.Update(gameTime);
            // Seleccion de vista
            GraphicsDevice.Viewport = vista;
            DrawCubo(gameTime);
            DrawObjetivo();
        }

        private void DrawCubo(GameTime gameTime)
        {
            foreach (ModelMesh mesh in Recursos.CuboModelo.Meshes)
            {
                foreach (Effect effect in mesh.Effects)
                {
                    CubeEffect be = effect as CubeEffect;
                    be.DiffuseTextureBase = Recursos.TDifusaCubo;
                    be.DiffuseTextureOver = ( jugador.Cubo.EsDetonador ) ? Recursos.TEmisionDetonador_Armado[1] : Recursos.TEmisionCubo;
                    be.Color = ( jugador.Cubo.EsDetonador) ? Parametros.DETONADOR_COLORES[jugador.Cubo.Color, contador.Valor]  : Parametros.CUBO_COLORES[jugador.Cubo.Color]; 
                    be.Projection = Main.camara.projection;
                    be.View = Main.camara.view;
                    be.World = Matrix.CreateTranslation(jugador.Cubo.Posicion) * world_matriz;
                }
                mesh.Draw();
            }
        }

        private void DrawObjetivo()
        {
            foreach (ModelMesh mesh in Recursos.ObjetivoModelo.Meshes)
            {
                foreach (BasicEffect effect in mesh.Effects)
                {
                    effect.Texture = Recursos.ObjetivoTexturas[(int)player];
                    effect.Projection = Main.camara.projection;
                    effect.View = Main.camara.view;
                    effect.World = Matrix.CreateTranslation(posicion_objetivo);
                }
                mesh.Draw();
            }
        }

        #endregion 
    }
}
