
layout (std140) uniform GerstnerWaveModel {
	vec3  GerstnerA;
	vec3  GerstnerPhi;
	vec3  GerstnerKLength;
	vec3  GerstnerOmega;
	vec2  GerstnerK1;
	vec2  GerstnerK2;
	vec2  GerstnerK3;
	float GerstnerTime;
};


void calculateGerstnerVertex(in vec2 x0, out vec4 pos, out vec3 normal)
{
	vec3 theta;
	theta.x = dot(GerstnerK1, x0) - GerstnerOmega.x*GerstnerTime + GerstnerPhi.x;
	theta.y = dot(GerstnerK2, x0) - GerstnerOmega.y*GerstnerTime + GerstnerPhi.y;
	theta.z = dot(GerstnerK3, x0) - GerstnerOmega.z*GerstnerTime + GerstnerPhi.z;

	vec3 SinValue = sin(theta);
	vec3 CosValue = cos(theta);

	vec3 Displacement;
	Displacement.x = (GerstnerK1.x/GerstnerKLength.x)*GerstnerA.x*SinValue.x +
					 (GerstnerK2.x/GerstnerKLength.y)*GerstnerA.y*SinValue.y +
					 (GerstnerK3.x/GerstnerKLength.z)*GerstnerA.z*SinValue.z;
	Displacement.y = GerstnerA.x*CosValue.x + GerstnerA.y*CosValue.y + GerstnerA.z*CosValue.z;
	Displacement.z = (GerstnerK1.y/GerstnerKLength.x)*GerstnerA.x*SinValue.x +
					 (GerstnerK2.y/GerstnerKLength.y)*GerstnerA.y*SinValue.y +
					 (GerstnerK3.y/GerstnerKLength.z)*GerstnerA.z*SinValue.z;

	pos.x = x0.x - Displacement.x;
	pos.y =        Displacement.y;
	pos.z = x0.y - Displacement.z;
	pos.w = 1.0f;

	normal.x = -( (GerstnerK1.x/GerstnerKLength.x)*GerstnerA.x*CosValue.x +
				  (GerstnerK2.x/GerstnerKLength.y)*GerstnerA.y*CosValue.y +
				  (GerstnerK3.x/GerstnerKLength.z)*GerstnerA.z*CosValue.z );
	normal.y = 1.0f;
	normal.z = -( (GerstnerK1.y/GerstnerKLength.x)*GerstnerA.x*CosValue.x + 
				  (GerstnerK2.y/GerstnerKLength.y)*GerstnerA.y*CosValue.y +
				  (GerstnerK3.y/GerstnerKLength.z)*GerstnerA.z*CosValue.z );
	normal = normalize(normal);
}


void calculateGerstnerVertexTBN(in vec2 x0, out vec4 pos, out vec3 t, out vec3 b, out vec3 n)
{
	vec3 theta;
	theta.x = dot(GerstnerK1, x0) - GerstnerOmega.x*GerstnerTime + GerstnerPhi.x;
	theta.y = dot(GerstnerK2, x0) - GerstnerOmega.y*GerstnerTime + GerstnerPhi.y;
	theta.z = dot(GerstnerK3, x0) - GerstnerOmega.z*GerstnerTime + GerstnerPhi.z;

	vec3 SinValue = sin(theta);
	vec3 CosValue = cos(theta);

	vec3 Displacement;
	Displacement.x = (GerstnerK1.x/GerstnerKLength.x)*GerstnerA.x*SinValue.x +
					 (GerstnerK2.x/GerstnerKLength.y)*GerstnerA.y*SinValue.y +
					 (GerstnerK3.x/GerstnerKLength.z)*GerstnerA.z*SinValue.z;
	Displacement.y = GerstnerA.x*CosValue.x + GerstnerA.y*CosValue.y + GerstnerA.z*CosValue.z;
	Displacement.z = (GerstnerK1.y/GerstnerKLength.x)*GerstnerA.x*SinValue.x +
					 (GerstnerK2.y/GerstnerKLength.y)*GerstnerA.y*SinValue.y +
					 (GerstnerK3.y/GerstnerKLength.z)*GerstnerA.z*SinValue.z;

	pos.x = x0.x - Displacement.x;
	pos.y =        Displacement.y;
	pos.z = x0.y - Displacement.z;
	pos.w = 1.0f;

	// tangent space
	b.x = 1.0;
	b.y = ((GerstnerK1.x/GerstnerKLength.x)*GerstnerA.x*CosValue.x +
		  (GerstnerK2.x/GerstnerKLength.y)*GerstnerA.y*CosValue.y +
		  (GerstnerK3.x/GerstnerKLength.z)*GerstnerA.z*CosValue.z);
	b.z = 0.0;
	b = normalize(b);

	t.x = 0.0;
	t.y = ((GerstnerK1.y/GerstnerKLength.x)*GerstnerA.x*CosValue.x +
		  (GerstnerK2.y/GerstnerKLength.y)*GerstnerA.y*CosValue.y +
		  (GerstnerK3.y/GerstnerKLength.z)*GerstnerA.z*CosValue.z);
	t.z = -1.0;
	t = normalize(t);

	n = cross(b, t);
}

void calculateGerstnerWaterDepthVertex(in vec2 x0, in float waterDepth, out vec4 pos, out vec3 normal)
{
	const float GRAVITYCONSTANT = 9.81f;
	vec3 omega = GerstnerKLength * tanh(GerstnerKLength * waterDepth) * GRAVITYCONSTANT;

	vec3 theta = vec3( dot(GerstnerK1, x0) - omega.x*GerstnerTime + GerstnerPhi.x,
					   dot(GerstnerK2, x0) - omega.y*GerstnerTime + GerstnerPhi.y,
					   dot(GerstnerK3, x0) - omega.z*GerstnerTime + GerstnerPhi.z );

	vec3 SinValue = sin(theta);
	vec3 CosValue = cos(theta);

	vec3 Displacement = vec3(
		(GerstnerK1.x/GerstnerKLength.x)*GerstnerA.x*SinValue.x + 
		(GerstnerK2.x/GerstnerKLength.y)*GerstnerA.y*SinValue.y +
		(GerstnerK3.x/GerstnerKLength.z)*GerstnerA.z*SinValue.z,

		GerstnerA.x * CosValue.x + GerstnerA.y * CosValue.y + GerstnerA.z * CosValue.z,

		(GerstnerK1.y/GerstnerKLength.x)*GerstnerA.x*SinValue.x + 
		(GerstnerK2.y/GerstnerKLength.y)*GerstnerA.y*SinValue.y +
		(GerstnerK3.y/GerstnerKLength.z)*GerstnerA.z*SinValue.z );

	pos.x = x0.x - Displacement.x;
	pos.y =        Displacement.y;
	pos.z = x0.y - Displacement.z;
	pos.w = 1.0f;

	normal.x = -( (GerstnerK1.x/GerstnerKLength.x)*GerstnerA.x*CosValue.x +
				  (GerstnerK2.x/GerstnerKLength.y)*GerstnerA.y*CosValue.y +
				  (GerstnerK3.x/GerstnerKLength.z)*GerstnerA.z*CosValue.z );
	normal.y = 1.0f;
	normal.z = -( (GerstnerK1.y/GerstnerKLength.x)*GerstnerA.x*CosValue.x +
				  (GerstnerK2.y/GerstnerKLength.y)*GerstnerA.y*CosValue.y +
				  (GerstnerK3.y/GerstnerKLength.z)*GerstnerA.z*CosValue.z );

	// normal = normalize(normal);
}


void calculateGerstnerWaterDepthVertexTBN(in vec2 x0, in float waterDepth, out vec4 pos, out vec3 t, out vec3 b, out vec3 n)
{
	const float GRAVITYCONSTANT = 9.81f;
	vec3 omega = GerstnerKLength * tanh(GerstnerKLength * waterDepth) * GRAVITYCONSTANT;

	vec3 theta = vec3( dot(GerstnerK1, x0) - omega.x*GerstnerTime + GerstnerPhi.x,
					   dot(GerstnerK2, x0) - omega.y*GerstnerTime + GerstnerPhi.y,
					   dot(GerstnerK3, x0) - omega.z*GerstnerTime + GerstnerPhi.z );

	vec3 SinValue = sin(theta);
	vec3 CosValue = cos(theta);

	vec3 Displacement = vec3(
		(GerstnerK1.x/GerstnerKLength.x)*GerstnerA.x*SinValue.x + 
		(GerstnerK2.x/GerstnerKLength.y)*GerstnerA.y*SinValue.y +
		(GerstnerK3.x/GerstnerKLength.z)*GerstnerA.z*SinValue.z,

		GerstnerA.x * CosValue.x + GerstnerA.y * CosValue.y + GerstnerA.z * CosValue.z,

		(GerstnerK1.y/GerstnerKLength.x)*GerstnerA.x*SinValue.x + 
		(GerstnerK2.y/GerstnerKLength.y)*GerstnerA.y*SinValue.y +
		(GerstnerK3.y/GerstnerKLength.z)*GerstnerA.z*SinValue.z );

	pos.x = x0.x - Displacement.x;
	pos.y =        Displacement.y;
	pos.z = x0.y - Displacement.z;
	pos.w = 1.0f;
#if 0
	normal.x = -( (GerstnerK1.x/GerstnerKLength.x)*GerstnerA.x*CosValue.x +
				  (GerstnerK2.x/GerstnerKLength.y)*GerstnerA.y*CosValue.y +
				  (GerstnerK3.x/GerstnerKLength.z)*GerstnerA.z*CosValue.z );
	normal.y = 1.0f;
	normal.z = -( (GerstnerK1.y/GerstnerKLength.x)*GerstnerA.x*CosValue.x +
				  (GerstnerK2.y/GerstnerKLength.y)*GerstnerA.y*CosValue.y +
				  (GerstnerK3.y/GerstnerKLength.z)*GerstnerA.z*CosValue.z );

	// normal = normalize(normal);
#else
	// tangent space
	b.x = 1.0;
	b.y = 0.0;
	b.z = ((GerstnerK1.x/GerstnerKLength.x)*GerstnerA.x*CosValue.x +
		  (GerstnerK2.x/GerstnerKLength.y)*GerstnerA.y*CosValue.y +
		  (GerstnerK3.x/GerstnerKLength.z)*GerstnerA.z*CosValue.z);
	b = normalize(b.xzy);

	t.x = 0.0;
	t.y = -1.0;
	t.z = ((GerstnerK1.y/GerstnerKLength.x)*GerstnerA.x*CosValue.x +
		  (GerstnerK2.y/GerstnerKLength.y)*GerstnerA.y*CosValue.y +
		  (GerstnerK3.y/GerstnerKLength.z)*GerstnerA.z*CosValue.z);
	t = normalize(t.xzy);

	n = cross(b, t);
#endif
}
