#version 330 core

uniform vec2 res;
uniform vec3 camPos;
uniform vec3 camLookAt;
uniform float noiseTime; // in seconds
uniform vec3 lightPos;
uniform vec3 upVector;

#if USE_NOISE
uniform sampler2D noiseTex;
#endif

in vec2 ex_TexCoord;
out vec4 out_color;

#if USE_NOISE
#define Detail -2.2
#define AODetail -0.5
#define MaxStep 80
#define MaxDist 5.0
#else
#define Detail -2.7
#define AODetail -0.5
#define MaxStep 150
#define MaxDist 10.0
#endif

#define AOSamples 6
#define NormalBackStep 0.5
#define ClarityPower 1.0
#define Dither 0.5
#define NoiseCoef 0.8
#define lightFalloff 2.0
#define lightIntensity 16.0
#define SpecHardness 256.0

vec4 LightColor = vec4(1.0, 1.0, 1.0, lightIntensity); // w is light intensity
vec3 FogColor = vec3(0.2, 0.23, 0.2);
float MinDist = pow(10.0, Detail);
float AOEps = pow(10.0, AODetail);

// Orbit Trap parameters
vec3 BaseColor = vec3(0.42, 0.27, 0.04);
float OrbitStrength = 0.5;
vec4 X = vec4(0.23, 0.2, 0.0, 1.0);
// Closest distance to XZ-plane during orbit
vec4 Y = vec4(0.24, 0.2, 0.4, 1.0);
// Closest distance to XY-plane during orbit
vec4 Z = vec4(0.27, 0.18, 0.3, 1.0);
// Closest distance to  origin during orbit
vec4 R = vec4(0.2, 0.23, 0.18, 1.0);

// Return rotation matrix for rotating around vector v by angle
mat3  rotationMatrix3(vec3 v, float angle)
{
	float c = cos(radians(angle));
	float s = sin(radians(angle));
	
	return mat3(c + (1.0 - c) * v.x * v.x, (1.0 - c) * v.x * v.y - s * v.z, (1.0 - c) * v.x * v.z + s * v.y,
		(1.0 - c) * v.x * v.y + s * v.z, c + (1.0 - c) * v.y * v.y, (1.0 - c) * v.y * v.z - s * v.x,
		(1.0 - c) * v.x * v.z - s * v.y, (1.0 - c) * v.y * v.z + s * v.x, c + (1.0 - c) * v.z * v.z
		);
}

float rand(vec2 co)
{
	// implementation found at: lumina.sourceforge.net/Tutorials/Noise.html
	return fract(sin(dot(co.xy ,vec2(12.9898,78.233))) * 43758.5453);
}

vec4 OrbitTrap = vec4(10000.0);

float evalMandelbox(vec3 pos)
{
	// Constants
#if USE_NOISE
	int Iterations = 9;
#else
	int Iterations = 10;
#endif
	int ColorIterations = 3;
	float Scale = 2.9;
	float MinRadius2 = 0.3;
#if USE_NOISE
	vec3 RotVector = vec3(texture(noiseTex, pos.yz).x, texture(noiseTex, pos.xz).x, texture(noiseTex, pos.xy).x);
	float RotAngle = NoiseCoef * noiseTime;
#else
	vec3 RotVector = vec3(0.0);
	float RotAngle = 0.0;
#endif
	float absScalem1 = abs(Scale - 1.0);
	float AbsScaleRaisedTo1mIters = pow(abs(Scale), float(1.0 - Iterations));

	mat3 rot;
	rot = rotationMatrix3(normalize(RotVector), RotAngle);
	vec4 p = vec4(pos, 1.0);
	vec4 pOrigin = p;
	vec4 scale = vec4(Scale, Scale, Scale, abs(Scale)) / MinRadius2;
	
	for (int i = 0; i < Iterations; i++)
	{
		p.xyz *= rot;
		p.xyz = clamp(p.xyz, -1.0, 1.0) * 2.0 - p.xyz;
		float r2 = dot(p.xyz, p.xyz);
	
		if (i < ColorIterations)
			OrbitTrap = min(OrbitTrap, abs(vec4(p.xyz, r2)));
		
		p *= clamp(max(MinRadius2 / r2, MinRadius2), 0.0, 1.0);
		p = p * scale + pOrigin;
			if (r2 > 100.0)
				break;
	}
	return ((length(p.xyz) - absScalem1) / p.w - AbsScaleRaisedTo1mIters);
}

float evalDist(vec3 p)
{
	return evalMandelbox(p);
}

vec3 calcNormal(vec3 pos, float nd)
{
	nd = max(nd * 0.5, 1.0e-7);
	vec3 e = vec3(0.0, nd, 0.0);
	vec3 n = vec3(evalDist(pos + e.yxx) - evalDist(pos - e.yxx),
				evalDist(pos + e.xyx) - evalDist(pos - e.xyx),
				evalDist(pos + e.xxy) - evalDist(pos - e.xxy));
	n = normalize(n);
	return n;
}

float calcAO(vec3 p, vec3 n)
{
	float ao = 0.0;
	float de = evalDist(p);
	float sum = 0.0;
	float w = 1.0;
    float d = 1.0 - (Dither * rand(p.xy));

	for (float i = 1.0; i < AOSamples + 1.0; i++)
	{
		float dist = (evalDist(p + d * n * pow(i, 2.0) * AOEps) - de) / (d * pow(i, 2.0) * AOEps);
		w *= 0.6;
		ao += w * clamp(1.0 - dist, 0.0, 1.0);
		sum += w;
	}
	return clamp(ao / sum, 0.0, 1.0);
}

// Use the orbit trap to compute color
vec3 getColor()
{
	OrbitTrap.w = sqrt(OrbitTrap.w);

	vec3 orbitColor;

	orbitColor = X.xyz * X.w * OrbitTrap.x +
					Y.xyz * Y.w * OrbitTrap.y +
					Z.xyz * Z.w * OrbitTrap.z +
					R.xyz * R.w * OrbitTrap.w;
	
	vec3 color = mix(BaseColor, 3.0 * orbitColor,  OrbitStrength);
	return color;
}

vec3 castRay(vec3 origin, vec3 dir)
{
	float dist = 0.0;
	float totalDist = 0.0;
	int step = 0;
	vec3 color = FogColor * 0.5; // fake some kind of occlusion where we run out of steps
	float epsModified = 0.0;
	float eps = MinDist;
	
	while (step < MaxStep)
	{
		OrbitTrap = vec4(10000.0);
		vec3 p = origin + totalDist * dir;
		dist = evalDist(p);
		
		// apply dither ?
		totalDist += dist;

		epsModified = pow(totalDist,ClarityPower) * eps;
		if (dist < epsModified)
			break;

		if (totalDist > MaxDist)
		{
			totalDist = MaxDist;
			break;
		}

		step++;
	}

	// hit
	if (dist < epsModified)
	{
		vec3 P = origin + totalDist * dir;
		vec3 N = calcNormal(P - (NormalBackStep * epsModified * dir), epsModified);
		vec3 L = normalize(lightPos - P);
		vec3 V = normalize(origin - P);
		vec3 H = normalize(V + L);
		float lightDist = abs(length(lightPos - P));
		float falloff = 1.0 / pow(lightDist, lightFalloff) * LightColor.w;
		falloff = clamp(falloff, 0.0, 2.0);

		float ambient = max(0.1, dot(N, dir));
		float diffuse = max(0.0, dot(N, L));
		float specular = pow(max(0.0, dot(N, H)), SpecHardness);
		
		float ao = calcAO(P, N);
		
		vec3 objColor = getColor();
		color = (objColor * LightColor.xyz * diffuse) * falloff +
				(LightColor.xyz * specular) * falloff +
				(objColor * ambient);
				
		color *= ao;
	}
	color += (float(step) / float(MaxStep)) * vec3(0.6, 0.9, 0.9);
	color = mix(color, FogColor, smoothstep(0.0, 1.0, totalDist / MaxDist));
	return color;
}

void main()
{
	vec3 color;
	vec2 pix = -1.0 + 2.0 * ex_TexCoord;
    pix.x *= res.x / res.y;
	pix.x = -pix.x;

	// Ray setup
	vec3 rayOrigin = camPos;
    vec3 ww = normalize(camLookAt - rayOrigin);
    vec3 uu = normalize(cross(upVector, ww));
    vec3 vv = normalize(cross(ww, uu));
    vec3 rayDir = normalize(pix.x * uu + pix.y * vv + 1.5 * ww);

	color = castRay(rayOrigin, rayDir);

	out_color = vec4(color, 1.0);
}