/*

 Name      :  Plane Deformation
 Notes     :  Plane deformation effects
 
 2D plane deformations using lookup tables was a great way to get
 smooth animations of complex calculations running on slower processors.
 These effects were very common in the 90s demoscene. The idea was to
 take a texture and deform it using a pre-calculated math reference table.
 Different formulas gave different effects such as fake 3d tunnels,
 wormholes and landscapes. Based on formulas by Inigo Quilez.
 
 references:
 http://iquilezles.org/www/material/deform/deform.htm
 
 */
package demoplatform;

public class Deform extends Demoplatform implements Runnable {

  private Thread thread;
  private boolean ready = false;
  private boolean running = false;
  public int[] mLUTu;
  public int[] mLUTv;
  public int[] mLUTb;
  int baseEffect = 6;
  String texturePath = "/images/seamless/tex4_256.png";
  public int[] tile_bitmap;
  public int pixelIndexLength;
  int effectWidth;
  int effectHeight;

  public Deform(String param1, int param2, int width, int height) {
    debug("Deform():: initialize");
    texturePath = param1;
    baseEffect = param2;
    effectWidth = width;
    effectHeight = height;
    debug("Deform():: end initialize");
  }

  public boolean ready() {
    return ready;
  }

  public void start() {
    debug("Deform():: thread start");
    thread = new Thread(this);
    thread.start();
  }

  // run in a thread
  public void run() {
    debug("Deform():: load texture");
    tile_bitmap = getImageArray(texturePath);
    debug("Deform():: create LUT");
    createLUT(baseEffect, effectWidth, effectHeight);
    ready = true;
  }

  public void createLUT(int effectStyle, int width, int height) {
    mLUTu = new int[width * height];
    mLUTv = new int[width * height];
    mLUTb = new int[width * height];

    pixelIndexLength = width * height;

    // increment placeholder
    int index = 0;

    // u and v are euclidean coordinates
    double u = 0;
    double v = 0;
    double bright = 0;

    for (int j = 0; j < height; j++) {
      double y = -1.00 + 2.00 * (double) j / (double) height;
      for (int i = 0; i < width; i++) {
        double x = -1.00 + 2.00 * (double) i / (double) width;
        double d = Math.sqrt(x * x + y * y);
        double a = Math.atan2(y, x);
        double r = d;

        switch (effectStyle) {

        case 1:   // stereographic projection / anamorphosis
          u = Math.cos(a) / d;
          v = Math.sin(a) / d;
          bright = -10 * (2 / (6 * r + 3 * x));
          break;

        case 2:  // hypnotic rainbow spiral
          double preCalc1 = a + Math.cos(3 * r);
          double preCalc2 = Math.pow(r, 0.2);
          v = Math.sin(preCalc1) / preCalc2;
          u = Math.cos(preCalc1) / preCalc2;
          bright = 0;
          break;

        case 3:  // rotating tunnel of wonder
          v = 2 / (6 * r + 3 * x);
          u = a * 3 / Math.PI;
          bright = 15 * -v;
          break;

        case 4:  // wavy star-burst
          v = (-0.4 / r) + .1 * Math.sin(8 * a);
          u = .5 + .5 * a / Math.PI;
          bright = 0;
          break;
        case 5:  // hyper-space travel
          u = (0.02 * y + 0.03) * Math.cos(a * 3) / r;
          v = (0.02 * y + 0.03) * Math.sin(a * 3) / r;
          bright = 0;
          break;

        case 6:  // five point magnetic flare
          u = 1 / (r + 0.5 + 0.5 * Math.sin(5 * a));
          v = a * 3 / Math.PI;
          bright = 0;
          break;

        case 7:  // cloud like dream scroll
          u = 0.1 * x / (0.11 + r * 0.5);
          v = 0.1 * y / (0.11 + r * 0.5);
          bright = 0;
          break;

        case 8:  // floor and ceiling with fade to dark horizon
          u = x / Math.abs(y);
          v = 1 / Math.abs(y);
          bright = 8 * v * 8;
          break;

        case 9:  // hot magma liquid swirl
          u = 0.5 * (a) / Math.PI;
          v = Math.sin(2 * r);
          bright = 0;
          break;

        case 10:  // clockwise flush down the toilet
          v = Math.pow(r, 0.1);
          u = (1 * a / Math.PI) + r;
          bright = 0;
          break;

        case 11:  // 3D ball
          double preCalcA = Math.sqrt(4 - 4 * r * r);
          double preCalcB = (r * r + 1);
          v = x * (3 - preCalcA) / preCalcB;
          u = y * (3 - preCalcA) / preCalcB;
          bright = 8 * -25.7 * (x + y + r * r - (x + y - 1) * preCalcA / 3) / preCalcB;
          break;
        default:  // show texture with no deformation or lighting
          u = x;
          v = y;
          bright = 0;
          break;
        }

        // make amigaball have larger texture
        if (effectStyle == 11) {
          mLUTu[index] = (int) ((int) Math.floor(128 * u) & 127);
          mLUTv[index] = (int) ((int) Math.floor(128 * v) & 127);
        } 
        else {
          mLUTu[index] = (int) ((int) Math.floor(512 * u) & 511);
          mLUTv[index] = (int) ((int) Math.floor(512 * v) & 511);
        }

        mLUTb[index++] = (int) (bright);
      }
    }

    debug("Deform():: createLUT " + effectStyle);
  }

  public void draw(int currentTime, int[] renderBuffer) {
    int timeDisplacement = (currentTime >> 3);

    for (int pixelIndex = 0; pixelIndex < pixelIndexLength; pixelIndex++) {
      renderBuffer[pixelIndex] = tile_bitmap[((mLUTv[pixelIndex] + timeDisplacement) + ((mLUTu[pixelIndex] + timeDisplacement) << 8)) & 0xFFFF];
    }
  }
}

