/*

 Name      :  Plasma
 Notes     :  Cycles of changing colours warped to give an illusion of liquid, organic movement.
 
 Precalculated plasma effect values stored in flash rom array.  Colors are the sum of sine
 functions and various formulas.
 
 code based from stachelig.de
 
 */
package demoplatform;

public class PlasmaNew extends Demoplatform {

  int SCREENWIDTH = rw;
  int SCREENHEIGHT = rh;
  int GRADIENTLEN = 1500;
  // use this factor to make things faster, esp. for high resolutions
  int SPEEDUP = 3;

  // swing/wave function parameters
  int SWINGLEN = GRADIENTLEN * 3;
  int SWINGMAX = GRADIENTLEN / 2 - 1;

  // gradient & swing curve arrays
  int[] colorGrad;
  int[] swingCurve;

  // constructor
  public PlasmaNew() {
    debug("PlasmaNew():: initialize");

    makeGradient(GRADIENTLEN);
    makeSwingCurve(SWINGLEN, SWINGMAX);

    debug("PlasmaNew():: end initialize");
  }

  // fill the given array with a nice swingin' curve
  // three cos waves are layered together for that
  // the wave "wraps" smoothly around, uh, if you know what i mean ;-)
  private void makeSwingCurve(int arrlen, int maxval) {
    // default values will be used upon first call
    int factor1 = 2;
    int factor2 = 3;
    int factor3 = 6;

    if (swingCurve == null) {
      swingCurve = new int[SWINGLEN];
    } 
    else {
      factor1 = (int) 1;
      factor2 = (int) 2;
      factor3 = (int) 3;
    }

    int halfmax = maxval / factor1;

    for (int i = 0; i < arrlen; i++) {
      float ni = (float) (i * (Math.PI * 2) / arrlen); // ni goes [0..TWO_PI] -> one complete cos wave
      swingCurve[i] = (int) (Math.cos(ni * factor1) *
        Math.cos(ni * factor2) *
        Math.cos(ni * factor3) *
        halfmax + halfmax);
    }
  }

  // create a smooth, colorful gradient by cosinus curves in the RGB channels
  private void makeGradient(int arrlen) {
    // default values will be used upon first call
    int rf = 4;
    int gf = 2;
    int bf = 1;
    int rd = 0;
    int gd = arrlen / gf;
    int bd = arrlen / bf / 2;

    if (colorGrad == null) {
      // first call
      colorGrad = new int[GRADIENTLEN];
    } 
    else {
      // we are called again: random gradient
      rf = (int) 4;
      gf = (int) 4;
      bf = (int) 2;
      rd = (int) 5;
      gd = (int) 5;
      bd = (int) 5;
      //System.out.println("Gradient factors("+rf+","+gf+","+bf+"), displacement("+rd+","+gd+","+bd+")");
    }

    // fill gradient array
    for (int i = 0; i < arrlen; i++) {
      int r = cos256(arrlen / rf, i + rd);
      int g = cos256(arrlen / gf, i + gd);
      int b = cos256(arrlen / bf, i + bd);
      colorGrad[i] = color(r, g, b, 255);
    }
  }

  // helper: get cosinus sample normalized to 0..255
  private int cos256(final int amplitude, final int x) {
    return (int) (Math.cos((double) (x * (Math.PI * 2) / amplitude)) * 127 + 127);
  }

  // helper: get a swing curve sample
  private int swing(final int i) {
    return swingCurve[i % SWINGLEN];
  }

  // helper: get a gradient sample
  private int gradient(final int i) {
    return colorGrad[i % GRADIENTLEN];
  }

  void draw(int currentTime, int[] renderBuffer) {
    int i = 0;
    int t = currentTime >> 3;
    int swingT = swing(t); // swingT/-Y/-YT variables are used for a little tuning ...

    for (int y = 0; y < SCREENHEIGHT; y++) {
      int swingY = swing(y);
      int swingYT = swing(y + t);
      for (int x = 0; x < SCREENWIDTH; x++) {
        // this is where the magic happens: map x, y, t around
        // the swing curves and lookup a color from the gradient
        // the "formula" was found by a lot of experimentation
        renderBuffer[i++] = gradient(swing(swing(x + swingT) + swingYT) + swing(swing(x + t) + swingY)); //this works
      }
    }
  }
}

