/*

 Name      :  Voxel Landscape
 Notes     :  heightmap based texture mapped voxel
 
 references:
 nuclearbit's code on ps3 forums
 http://www.tecgraf.puc-rio.br/publications/artigo_1997_realtime_rendering.pdf
 http://www.bundysoft.com/L3DT/downloads/examples/#maps
 
 */
package demoplatform;

import java.util.Arrays;

public final class Voxel extends Demoplatform {

  static final float PI = 3.141593F;
  static final int RENDERWIDTH = 480;
  static final int RENDERHEIGHT = 270;
  static final int MAPWIDTH = 512;
  static final int MAPWIDTH_MIN_1 = MAPWIDTH - 1;
  static final int MAPSIZE = MAPWIDTH * MAPWIDTH;
  static final int MAPSHIFT = 9;
  int sinTable[];
  int cosTable[];
  int heightMap[];
  int texelMap[];
  int colorLut[];
  int playerX = 0;
  int playerY = 0;
  int playerZ = 350 << 12;        // higher number = overhead view
  int playerDistance = 277 << 11; // how tall the mountains are
  public int playerSpeed = 5;
  int playerAngle = 0;
  public int angleSpeed = 0;              // positive number turns right
  int viewportHeight = 100 << 12;  // higher number = view more sky

  public int frameC = 0;


  // constructor
  public Voxel() {
    debug("Voxel():: initialize");
    texelMap = getImageArray("/images/land.png");
    heightMap = new int[MAPSIZE];
    int abyte0[] = getImageArray("/images/height.png");

    int k = 0;
    do {
      int aByte = (0x000000FF & ((int) abyte0[k]));
      heightMap[k] = aByte;
    } 
    while (++k < MAPSIZE);

    sinTable = new int[3600];
    cosTable = new int[3600];
    for (int i = 0; i < 3600; i++) {
      sinTable[i] = toFP((float) Math.sin(2f * 3.141593f * i / 3600f));
      cosTable[i] = toFP((float) Math.cos(2f * 3.141593f * i / 3600f));
    }
    colorLut = new int[65536];
    for (int i = 0; i < 256; i++) {
      for (int j = 0; j < 256; j++) {
        int idx = (i << 8) + j;
        float c = (float) i * (float) j / 256f;
        colorLut[idx] = (int) Math.round(c);
      }
    }
    debug("Voxel():: end initialize");
  }

  void draw(int[] renderBuffer) {
    playerAngle += angleSpeed;
    if (playerAngle < 0) {
      playerAngle += 3600;
    } 
    else if (playerAngle >= 3600) {
      playerAngle -= 3600;
    }

    if (frameC > 18) {
      playerSpeed = 3;
      angleSpeed = 0;
      frameC = 0;
    }

    frameC++;

    int cos = cosTable[playerAngle];
    int sin = sinTable[playerAngle];

    //playerX += MathFP.mul(cos, playerSpeed);
    //  playerY += MathFP.mul(sin ,playerSpeed);
    playerX += cos * playerSpeed;
    playerY += sin * playerSpeed;

    //playerX += mul(cos, playerSpeed);
    //playerY += mul(sin, playerSpeed);

    render2(playerX, playerY, playerZ, playerAngle, playerDistance, renderBuffer);
  }

  public void render2(int eyeX, int eyeY, int eyeZ, int arc, int dpk, int[] renderBuffer) {

    Arrays.fill(renderBuffer, 0xFF000000);

    int colAngle = arc - (RENDERWIDTH >> 1); // view from angle-30 to angle+30
    if (colAngle < 0) {
      colAngle += 3600;
    }
    int col = 0;
    int pitch = eyeZ - viewportHeight;
    //  int invdpk = MathFP.div(1 << 12, dpk);
    //  int invdpk = (1 << 12)/dpk;
    int invdpk = div(1 << 12, dpk);

    do // COL iteration
    {
      int j3 = RENDERHEIGHT * RENDERWIDTH - RENDERWIDTH + col;
      int cosA = cosTable[colAngle];
      int sinA = sinTable[colAngle];
      int z = eyeZ;
      int x = eyeX;
      int y = eyeY;
      int dz = 0;
      //    int m = MathFP.div(pitch, dpk);
      //int m = pitch / dpk;
      int m = div(pitch, dpk);
      //int j = 0; // row
      for (int i = 0; i < RENDERWIDTH - 256; i++) // abscis iteration
      {
        y += sinA;
        x += cosA;
        z -= m;
        dz += invdpk;
        int xm = (x >> 12) & MAPWIDTH_MIN_1;  // mod 512
        int ym = (y >> 12) & MAPWIDTH_MIN_1;  // mod 512
        int idx = (ym << MAPSHIFT) + xm;
        int hi = heightMap[idx] << 12;
        while (hi > z && j3 > 0) {
          int texel = texelMap[idx];
          renderBuffer[j3] = texel;
          j3 -= RENDERWIDTH;
          m -= invdpk;
          z += dz;
          //j++; // move up a screen row
        }
        if (j3 < 0) {
          break;
        }
      }
      for (int i = 0; i < 256; i++) // abscis iteration
      {
        y += sinA;
        x += cosA;
        z -= m;
        dz += invdpk;
        int xm = (x >> 12) & MAPWIDTH_MIN_1;  // mod 512
        int ym = (y >> 12) & MAPWIDTH_MIN_1;  // mod 512
        int idx = (ym << MAPSHIFT) + xm;
        int hi = heightMap[idx] << 12;
        while (hi > z && j3 > 0) {
          int texel = blend(texelMap[idx], 0xFF000000, i);
          //              int texel = color(255,0,0);
          renderBuffer[j3] = texel;
          j3 -= RENDERWIDTH;
          m -= invdpk;
          z += dz;
          //j++; // move up a screen row
        }
        if (j3 < 0) {
          break;
        }
      }
      if (++colAngle >= 3600) {
        colAngle = 0;
      }
    } 
    while (++col < RENDERWIDTH);
  }

  private final int blend(int c1, int c2, int f) {
    int r1 = (c1 & 0x00FF0000) >> 16;
    int r2 = (c2 & 0x00FF0000) >> 16;
    int r = (colorLut[(r1 << 8) + (255 - f)] + colorLut[(r2 << 8) + f]) << 16;

    int g1 = (c1 & 0x0000FF00) >> 8;
    int g2 = (c2 & 0x0000FF00) >> 8;
    int g = (colorLut[(g1 << 8) + (255 - f)] + colorLut[(g2 << 8) + f]) << 8;

    int b1 = (c1 & 0x000000FF);
    int b2 = (c2 & 0x000000FF);
    int b = colorLut[(b1 << 8) + (255 - f)] + colorLut[(b2 << 8) + f];

    return 0xFF000000 + r + g + b;
  }

  public static int div(int i, int j) {
    boolean flag = false;
    if (j == 4096) {
      return i;
    }
    if ((j & 0xfff) == 0) {
      return i / (j >> 12);
    }
    if (i < 0) {
      i = -i;
      flag = true;
    }
    if (j < 0) {
      j = -j;
      if (flag) {
        flag = false;
      } 
      else {
        flag = true;
      }
    }
    byte byte0 = 0;
    if (i > 0x64fff) {
      byte0 = 3;
    }
    if (i > 0x3e8fff) {
      byte0 = 4;
    }
    if (i > 0x7dffff) {
      byte0 = 6;
    }
    if (i > 0x1f4ffff) {
      byte0 = 8;
    }
    if (i > 0x7dfffff) {
      byte0 = 10;
    }
    if (byte0 > 0) {
      int k = 2 << byte0 - 1;
      i += k;
      j += k;
    }
    int l = (i << 12 - byte0) / (j >> byte0);
    return flag ? -l : l;
  }

  public static int toFP(float f) {
    float absf = Math.abs(f);
    int k = Math.round(absf);
    int l = Math.round((absf - k) * 10000);
    int i1 = (k << 12) + (l << 12) / 10000;
    if (f < 0.0f) {
      i1 = -1 * i1;
    }
    return i1;
  }
}

