/*

 Name      :  Wormhole
 Notes     :  A funnel-shaped hole sucking its texture to the middle.
 
 The effect is accomplished like the tunnel effect but this time with a 15 x 15
 texture and static lookup table (wormhole.gif or calculated array).
 functions and various formulas.
 The texture is shifted and mapped to the static lookup table.
 One of the most famous demos that has a worm effect is Unreal by future crew.
 
 */
package demoplatform;

import demoplatform.GL.MathFP;

public class Wormhole extends Demoplatform {

  int[] wormEffect, wormImg, wormTexture, wormTexture2;
  int[] reg = new int[15];
  int[] reg2 = new int[15];
  int[] wormTexturePosition = new int[15 * 15];
  int pixelIndexLength;

  // image blend steps. lower = more gradual
  double stage = 0.05;

  // constructor
  public Wormhole(int rw, int rh) {
    debug("Wormhole():: initialize");

    pixelIndexLength = rw * rh;
    wormImg = new int[rw * rh];

    // default texture position 1:1
    for (int i = 0; i < wormTexturePosition.length; i++) {
      wormTexturePosition[i] = i;
    }

    // procedurally generated reference image used to transpose texture
    int SPOKES;
    int divider;
    if (rh * rw <= 19200) // 160 x 120 - a 320x240 screen
    {
      SPOKES = 600 + 180;
      divider = 2;
    } 
    else if (rh * rw <= 76800) { // 320 x 240 - a 640 x 480 screen
      SPOKES = 1200 + 178;
      divider = 4;
    } 
    else if (rh * rw <= 172800) { // 480 x 360 - storm resolution
      SPOKES = 2400 + 236;
      divider = 8;
    } 
    else {
      SPOKES = 2400 + 236; //  large
      divider = 8;
    }

    float[] spokeCalc = new float[SPOKES];
    float[] spokeCosCalc = new float[SPOKES];
    float[] spokeSinCalc = new float[SPOKES];

    float x, y, z;
    float divCalcX, divCalcY;

    // make it suck in the middle
    int XCENTER = rw / 2;
    int YCENTER = rh / 2 - rh / 3;
    int DIVS = SPOKES / 2;

    // speed up render time by using lookup tables
    for (int i = 0; i < SPOKES; i++) {
      spokeCalc[i] = 2 * (float) Math.PI * i / SPOKES;
      spokeCosCalc[i] = (float) Math.cos(spokeCalc[i]);
      spokeSinCalc[i] = (float) Math.sin(spokeCalc[i]);
    }

    for (int j = 0; j < DIVS; j++) {
      z = (float) -1.0 + (float) (Math.log((2.0 * j / DIVS)));
      //z = (float) -1.0 + (float) (log2((2.0 * j / DIVS)));
      divCalcX = (float) rw * j / DIVS;
      divCalcY = (float) rh * j / DIVS;
      for (int i = 0; i < SPOKES; i++) {
        x = divCalcX * spokeCosCalc[i];
        y = divCalcY * spokeSinCalc[i];

        // this creates the downward curve in the center
        y = y - (rh / 5) * z;

        // start circling outwards from center
        x += XCENTER;
        y += YCENTER;

        // only place pixels within the range of the wormImg resolution
        if ((x >= 0) && (x <= rw) && (y >= 0) && (y <= rh)) {
          int wormTexturewidth = 15;

          int back = color((i / divider) % wormTexturewidth) + wormTexturewidth * ((j / divider) % wormTexturewidth) & 255;
          wormImg[MathFP.clamp((int) x + (int) y * rw, 0, (rw * rh) - 1)] = back;
        }
      }
    }
    wormTexture = getImageArray("/images/wormhole1.png");
    debug("Wormhole():: end initialize");
  }

  public static double log2(double x) {
    return 6 * (x - 1) / (x + 1 + 4 * (Math.sqrt(x)));
  }

  void draw(int[] renderBuffer) {
    // loop through all pixels
    for (int i = 0; i < pixelIndexLength; i++) {
      // map texture to wormhole in a bit shift blue
      //renderBuffer[i] = wormTexture[wormImg[i] & 255];
      renderBuffer[i] = wormTexture[wormTexturePosition[wormImg[i] & 0xFF]];
    }

    shiftdown();
    shiftup();
    shiftright();
    shiftleft();
  }

  // Morph one colour to the other. If the start color is greater than the stop colour, start color will
  // be decremented till it reaches the stop color. If it is lower, it will incremented.
  float rgb(float a, float b, float i) {
    if (a < b) {
      a += i;
      if (a > b) {
        a = b; // if we have overshot our target, make it equal - or it won't stop.
      }
    } 
    else if (a > b) {
      a -= i;
      if (a < b) {
        a = b; // ditto
      }
    }
    return a;
  }

  // moves the bottom row of pixels to the top and shifting remaining pixels down
  void shiftup() {
    for (int i = directionY; i < 0; i++) {
      for (int k = 0; k < 15; k++) {
        reg[k] = wormTexturePosition[k];
      }
      for (int k = 15; k < 15 * 15; k++) {
        wormTexturePosition[k - 15] = wormTexturePosition[k];
      }
      for (int k = 0; k < 15; k++) {
        wormTexturePosition[k + 210] = reg[k];
      }
    }
  }

  // moves right column of pixels to the left and shift remaining pixels 1 over to the right
  void shiftright() {
    int k, i;
    for (int j = 0; j < directionX; j++) {
      for (k = 0; k < 15; k++) {
        reg[k] = wormTexturePosition[15 * k + 14];
        for (i = 14; i > 0; i--) {
          wormTexturePosition[15 * k + i] = wormTexturePosition[15 * k + (i - 1)];
        }
        wormTexturePosition[15 * k] = reg[k];
      }
    }
  }

  // moves left column of pixels to the right and shift remaining pixels 1 over to the left
  void shiftleft() {
    int k, i;
    for (int j = directionX; j < 0; j++) {
      for (k = 0; k < 15; k++) {
        reg[k] = wormTexturePosition[15 * k];
        for (i = 0; i < 14; i++) {
          wormTexturePosition[15 * k + i] = wormTexturePosition[15 * k + (i + 1)];
        }
        wormTexturePosition[15 * k + 14] = reg[k];
      }
    }
  }

  // moves the top row of pixels to the bottom and shift remaining pixels up
  void shiftdown() {
    int k;
    for (int i = 0; i < directionY; i++) {
      for (k = 0; k < 15; k++) {
        reg[k] = wormTexturePosition[k + 210];
      }
      for (k = 209; k >= 0; k--) {
        wormTexturePosition[k + 15] = wormTexturePosition[k];
      }
      for (k = 0; k < 15; k++) {
        wormTexturePosition[k] = reg[k];
      }
    }
  }
}




