#version 330 core

uniform sampler2D dirt;
uniform sampler2D grass;
uniform sampler2D rock;
uniform sampler2D snow;
uniform sampler2D detail;
uniform sampler2D road;
uniform sampler2D road_opacity;

// .x = min height, .y = max height, .z = min slope, .w = max slope
uniform vec4 rockParams;
uniform vec4 dirtParams;
uniform vec4 snowParams;
uniform vec4 grassParams;
// .x=grass, .y=dirt, .z=rock, .w=snow
uniform vec4 mapStrength;

uniform vec2 detailParams; // x=tex coord scale, y=detail strength
uniform vec2 roadParams; // x=tex coord scale

in vec2 pos_zw;
in vec3 terrain_normal;
in vec3 normal;
in vec3 uvw;
in vec2 uv;
in float height;

layout (location = 0) out vec4 diffuse_specular;
layout (location = 1) out vec4 normal_depth;

#include <deferred/tangentspace_include.frag>
#include <terrain/terrain_include.frag>

float luminance(in vec3 color)
{
	return dot(color, vec3(0.299, 0.587, 0.114));
}


void main()
{
	// basic weights
	vec3 n = normalize(terrain_normal);
	float slope = 1.0 - n.y;

	vec4 weights;
	weights.x = computeWeight(height, rockParams.x, rockParams.y) * computeWeight(slope, rockParams.z, rockParams.w) * mapStrength.z;
	weights.y = computeWeight(height, dirtParams.x, dirtParams.y) * computeWeight(slope, dirtParams.z, dirtParams.w) * mapStrength.y;
	weights.z = computeWeight(height, snowParams.x, snowParams.y) * computeWeight(slope, snowParams.z, snowParams.w) * mapStrength.w;
	weights.w = computeWeight(height, grassParams.x, grassParams.y) * computeWeight(slope, grassParams.z, grassParams.w) * mapStrength.x;
	weights *= 1.0 / (weights.x + weights.y + weights.z + weights.w);

	// planar slope
	vec3 tpw = abs(n);
	tpw = (tpw - vec3(0.2)) * 7.0;
	tpw = max(tpw, vec3(0.0));
	tpw *= 1.0 / (tpw.x + tpw.y + tpw.z);

	// diffuse
	vec4 final = weightedTexture(dirt, tpw, uvw) * weights.y;
	final += weightedTexture(grass, tpw, uvw) * weights.w;
	final += weightedTexture(rock, tpw, uvw) * weights.x;
	final += weightedTexture(snow, tpw, uvw) * weights.z;

	float road_vis = pow(texture(road_opacity, uv).r, 4.0);
	final = mix(final, weightedTexture(road, tpw, uvw*roadParams.x), road_vis);

	final += (weightedTexture(detail, tpw, uvw*detailParams.x)*2.0 - vec4(1.0)) * detailParams.y;

	diffuse_specular.rgb = final.rgb;
	diffuse_specular.a = smoothstep(0.25, 0.5, luminance(final.rgb));

	normal_depth.xyz = normalize(normal);
	normal_depth.w = pos_zw.x/pos_zw.y;
}
