// Adapt physics systems tm

#define _WIN32_WINNT 0x400
#include <windows.h>
#include <d3d9.h>
#include <d3dx9.h>

#include <string>
#include <vector>
#include <list>
#include <time.h>
#include <algorithm>
#include <fstream>
#include <io.h>
#include <map>

#include "process.h"

using std::list;
using std::vector;
using std::string;


#include "D3DApp.h"
#include "common_globals.h"
#include "shader.h"
#include "ContextData.h"
#include "DemoContextData.h"

#include "Physics.h"

void RotateByAngle(float angle, D3DXVECTOR3 *p) {
  D3DXMATRIXA16 matRot;
  D3DXMatrixRotationZ(&matRot, angle);

  D3DXVECTOR4 t = D3DXVECTOR4(p->x, p->y, p->z, 0.0f);
  D3DXVECTOR4 pr;
  D3DXVec4Transform(&pr, &t, &matRot);
  
  p->x = pr.x;
  p->y = pr.y;
  p->z = pr.z;
}


// calculate the mass center point for all of the particles in this collection
// (current value is saved to m_currentCenter)
D3DXVECTOR3 PhysicsParticleCollection::CalculateCenter() {
  D3DXVECTOR3 center = D3DXVECTOR3(0.0f, 0.0f, 0.0f);
  float totalMass = 0.0f;
  for (std::vector<PhysicsParticle>::iterator it=m_physicsParticles.begin(); it!=m_physicsParticles.end(); it++) {
    totalMass += it->GetMass();
    center += *(it->GetPosition())*it->GetMass();
  }
  center /= totalMass;
  m_currentCenter = center;
  return center;
}

// calculate the movement of the mass center from the base
D3DXVECTOR3 PhysicsParticleCollection::CalculateTransformFromBase() {
  D3DXVECTOR3 trans = m_currentCenter - m_currentCenterBase;
  return trans;
}

// calculate the rotated angle compared to the base
float PhysicsParticleCollection::CalculateRotationAngleFromBase() {
  float angle = 0.0f;
  float angleAbsMax = 0.0f;
  for (std::vector<PhysicsParticle>::iterator it=m_physicsParticles.begin(); it!=m_physicsParticles.end(); it++) {
    D3DXVECTOR3 pos = (*(it->GetPosition()))-m_currentCenter;
    D3DXVECTOR3 posBase = *(it->GetPositionBase())-m_currentCenterBase;
    float angleNow = AtanSafe(pos.y, pos.x);
    float angleBase = AtanSafe(posBase.y, posBase.x);
    float angleDiff = angleNow-angleBase;

    float angleAbs = fabs(angleDiff);

    if (angleAbs > angleAbsMax) {
      angle = angleDiff;
      angleAbsMax = angleAbs;
    }
  }
  return angle;
}


// calculate the rotated angle compared to the initial position
float PhysicsParticleCollection::CalculateRotationAngleFromInit() {
  float angle = 0.0f;
  float angleAbsMax = 0.0f;
  for (std::vector<PhysicsParticle>::iterator it=m_physicsParticles.begin(); it!=m_physicsParticles.end(); it++) {
    D3DXVECTOR3 pos = (*(it->GetPosition()))-m_currentCenter;
    D3DXVECTOR3 posInit = *(it->GetPositionInit())-m_centerInit;
    float angleNow = AtanSafe(pos.y, pos.x);
    float angleInit = AtanSafe(posInit.y, posInit.x);
    float angleDiff = angleNow-angleInit;

    float angleAbs = fabs(angleDiff);

    if (angleAbs > angleAbsMax) {
      angle = angleDiff;
      angleAbsMax = angleAbs;
    }
  }
  m_angle = angle;
  return angle;
}

float PhysicsParticleCollection::GetTimeFromInit() {
  DemoContextData *cd = (DemoContextData*)GetCurrentContextData();
  return cd->m_timeFloat-m_timeOfInit;
}

void PhysicsParticleCollection::InitFromTexture(Texture *t, D3DXVECTOR3 *initialPosition, float scaleX, float scaleY, float scaleZ, int particlesX, int particlesY, bool bDynamic, float totalMass, float initialAngle) {
  // for now to get testing something, just generate 10x10 particles square
  int width = particlesX;
  int height = particlesY;

  m_restingForFrames = 0;

  m_bDynamic = bDynamic;
  m_centerInit = *initialPosition;
  m_position = m_centerInit;
  m_speed = D3DXVECTOR3(0.0f, 0.0f, 0.0f);

  m_scale = D3DXVECTOR3(scaleX, -scaleY, -scaleZ);

  m_angle = initialAngle/360.0f*2.0f*D3DX_PI;

  DemoContextData *cd = (DemoContextData*)GetCurrentContextData();
  m_timeOfInit = 0.0f;
  if (cd) {
    m_timeOfInit = cd->m_timeFloat;
  }

  int numParticles = width*height;
  for (int y=0; y<height; y++) {
    for (int x=0; x<width; x++) {
      float mass = totalMass/(float)numParticles;
      float particleDiam1 = scaleX/(float)max(width-1, 1);
      float particleDiam = scaleX*1.0f/(float)max(width-1, 1);
      D3DXVECTOR3 pos;
      float sx = (float)(width)/width;
      if (sx < 0.10f) {
        sx = 0.10f;
      }
      float sy = (float)(height)/height;
      if (sy < 0.10f) {
        sy = 0.10f;
      }
      pos = D3DXVECTOR3( ((float)x/(float)max(width-1, 1)-0.5f)*(scaleX*sx), ((float)y/(float)max(height-1,1)-0.5f)*(scaleY*sy), 0.0f);
      RotateByAngle(m_angle, &pos);
      D3DXVECTOR3 norm;
      D3DXVec3Normalize(&norm, &pos);
      pos += *initialPosition;
      PhysicsParticle pp(mass, &pos, particleDiam, this);
      if (x==0 && y!=0 && y!=(height-1)) {
        norm = D3DXVECTOR3(-1.0f, 0.0f, 0.0f);
      }
      if (x==(width-1) && y!=0 && y!=(height-1)) {
        norm = D3DXVECTOR3(1.0f, 0.0f, 0.0f);
      }
      if (y==0 && x!=0 && x!=(width-1)) {
        norm = D3DXVECTOR3(0.0f, -1.0f, 0.0f);
      }
      if (y==(height-1) && x!=0 && x!=(width-1)) {
        norm = D3DXVECTOR3(0.0f, 1.0f, 0.0f);
      }
      RotateByAngle(m_angle, &norm);
      pp.SetNormal(&norm);
      if (x==0 || y==0 || x==(width-1) || y==(height-1)) {
   //   {
        m_physicsParticles.push_back(pp);
      }
    }
  }

  m_angleSpeed = 0.0f;


  CalculateCenter();
  m_currentCenterBase = m_currentCenter;

  /*
      Texture *levelTexture = g_D3DApp->getTexture(le->getImage(g_languageCode, "path1")->file.c_str());
    Texture *bordersTexture = g_D3DApp->getTexture(leB->getImage(g_languageCode)->file.c_str());
    if (levelTexture && bordersTexture) {
      D3DLOCKED_RECT lockedRect;
      HRESULT hr = levelTexture->lpTexture->LockRect(0, &lockedRect, NULL, D3DLOCK_READONLY);
      int i=0;
      for (int h=0; h<levelTexture->height; h++) {
        i=h*lockedRect.Pitch;
        for (int w=0; w<levelTexture->width; w++) {
          unsigned char cr = ((unsigned char*)lockedRect.pBits)[i];
          unsigned char cg = ((unsigned char*)lockedRect.pBits)[i+1];
          unsigned char cb = ((unsigned char*)lockedRect.pBits)[i+2];
          unsigned char ca = ((unsigned char*)lockedRect.pBits)[i+3];
          if (cr) {
            D3DXVECTOR3 trans = D3DXVECTOR3(
              (((float)w*2.0f-(float)levelTexture->width)/(float)levelTexture->width)*8.0f,
              (((float)h*2.0f-(float)levelTexture->height)/(float)levelTexture->height)*8.0f,
              0.0f);
            D3DXVECTOR3 rot = D3DXVECTOR3(0.0f, 0.0f, 6.0f*(0.5f+0.5f*cosf(sinf(m_timeFloat*1.0f+(float)i*0.1f)))+3.141592f*1.0f);
            //D3DXVECTOR3 rot = D3DXVECTOR3(0.0f, 0.0f, 0.0f);
            g_D3DApp->DrawLayoutElement(leB, 1.0f, NULL, &trans, NULL, &rot);
          }
          i+=4;
        }
      }
      levelTexture->lpTexture->UnlockRect(0);
    }
  */

}


// Add all particle pointers from this collection to the given grid
// (forces for particles will be calculated by the grid, also clearing of the forces is handled by the grid)
void PhysicsParticleCollection::AddParticlesToGrid(PhysicsGrid *pgPtr) {
  for (std::vector<PhysicsParticle>::iterator it=m_physicsParticles.begin(); it!=m_physicsParticles.end(); it++) {
    int cx, cy;
    PhysicsGridCell *pgcPtr = pgPtr->GetCellPtrAtPos(it->GetPosition(), &cx, &cy);
    pgcPtr->AddParticle(&(*it));
  }
}



void CalculateInertiaTensorInverse(D3DXMATRIXA16 *res, D3DXMATRIXA16 *it0, float angle) {
  D3DXMATRIXA16 matRot;
  D3DXMATRIXA16 matRotT;
  D3DXMatrixRotationZ(&matRot, angle);
  D3DXMatrixTranspose(&matRotT, &matRot);

  D3DXMATRIXA16 temp;
  D3DXMatrixMultiply(&temp, &matRot, it0);
  D3DXMatrixMultiply(res, &temp, &matRotT);
}


float clamp(float a) {
  if (a < 0.0f) {
    a = 0.0f;
  }
  if (a > 1.0f) {
    a = 1.0f;
  }
  return a;
}

void PhysicsGridCell::CalculateMinContactLevel() {
  m_minContactLevel = 100000000;
  for (std::vector<PhysicsParticle *>::iterator it=m_physicsParticles.begin(); it!=m_physicsParticles.end(); it++) {
    int contactLevel = (*it)->GetPtrBelongsToCollection()->m_contactDistanceNumber;
    if (contactLevel < m_minContactLevel) {
      m_minContactLevel = contactLevel;
    }
  }
}

void PhysicsGrid::ClearParticleForces(std::vector<PhysicsParticleCollection*> *pParticleCollections) {
  /*
  for (int y=0; y<m_cellsY; y++) {
    for (int x=0; x<m_cellsX; x++) {
      std::vector<PhysicsParticle*> *gridPP = m_grid[y][x].GetPhysicsParticles();
      for (std::vector<PhysicsParticle*>::iterator it=gridPP->begin(); it!=gridPP->end(); it++) {
        (*it)->ClearForce();
      }
    }
  }
  */
  for (std::vector<PhysicsParticleCollection*>::iterator itC=pParticleCollections->begin(); itC!=pParticleCollections->end(); itC++) {
    PhysicsParticleCollection *pBody = (*itC);
    if (!(pBody)->GetIsDynamic()) { // skip the map creation with static bodies
      continue;
    }
    for (std::vector<PhysicsParticle>::iterator it=pBody->m_physicsParticles.begin(); it!=pBody->m_physicsParticles.end(); it++) {
      it->ClearForce();
    }
  }
}

void PhysicsGrid::CalculateMinContactLevels() {
  for (int y=0; y<m_cellsY; y++) {
    for (int x=0; x<m_cellsX; x++) {
      m_grid[y][x].CalculateMinContactLevel();
    }
  }
}

int PhysicsGrid::CalculateContactDistances(std::vector<PhysicsParticleCollection*> *pParticleCollections) {
  // body = particlecollection
  // first mark all bodies contact distance as -1 except static ones as 0
  for (std::vector<PhysicsParticleCollection*>::iterator itC=pParticleCollections->begin(); itC!=pParticleCollections->end(); itC++) {
    (*itC)->m_contactDistanceNumber = -1;
    if (!(*itC)->GetIsDynamic()) {
      (*itC)->m_contactDistanceNumber = 0;
    }
    (*itC)->m_primaryContactWith = NULL;
    (*itC)->m_mapInContactWith.clear();
  }

  // then for each body, make a map of other bodies it is in contact with
  for (std::vector<PhysicsParticleCollection*>::iterator itC=pParticleCollections->begin(); itC!=pParticleCollections->end(); itC++) {
    PhysicsParticleCollection *pBody = (*itC);
    if (!(pBody)->GetIsDynamic()) { // skip the map creation with static bodies
      continue;
    }
    for (std::vector<PhysicsParticle>::iterator it=pBody->m_physicsParticles.begin(); it!=pBody->m_physicsParticles.end(); it++) {
      PhysicsParticle *pPart = &(*it);
      // compare to nearby particles in the same grid cells
      int x, y;
      GetCellPtrAtPos((*it).GetPosition(), &x, &y);
      for (int yn=y-1; yn<y+2; yn++) {
        for (int xn=x-1; xn<x+2; xn++) {
          std::vector<PhysicsParticle*> *gridPPNear = GetCellPtrAtCell(xn, yn)->GetPhysicsParticles();
          for (std::vector<PhysicsParticle*>::iterator itNear=gridPPNear->begin(); itNear!=gridPPNear->end(); itNear++) {
            PhysicsParticle *pPartNear = (*itNear);
            if ((pPart != pPartNear) && pPartNear->GetPtrBelongsToCollection() != pBody) { // not comparing to itself or to the same body
              D3DXVECTOR3 relPos = *(pPart->GetPosition())-*(pPartNear->GetPosition());
              D3DXVECTOR3 externalForces = D3DXVECTOR3(0.0f, 1.0f, 0.0f);
              float contactAlignWithExternalForce = D3DXVec3Dot(&relPos, &externalForces);
              float diam = 1.0f*pPart->GetDiameter();
              float dist = D3DXVec3Length(&relPos);
              if (dist < diam && (contactAlignWithExternalForce < 0.0f)) { 
                PhysicsParticleCollection *pOtherC = pPartNear->GetPtrBelongsToCollection();
             //   if (pOtherC->m_contactDistanceNumber == 0) {
             //     pBody->m_primaryContactWith = pOtherC;
             //     pBody->m_contactDistanceNumber = 1;
             //   }
                pBody->m_mapInContactWith[pOtherC] = true; 
              }
            }
          }
        }
      }
    }
  }
  int maxContactLevels=100;
  int biggestContactLevel=0;
  for (int contactLevel=0; contactLevel<maxContactLevels; contactLevel++) {
    for (std::vector<PhysicsParticleCollection*>::iterator itC=pParticleCollections->begin(); itC!=pParticleCollections->end(); itC++) {
      PhysicsParticleCollection *pBody = (*itC);
      if (pBody->m_contactDistanceNumber == -1) {
        for (std::map<PhysicsParticleCollection*, bool>::iterator itOtherC=pBody->m_mapInContactWith.begin(); itOtherC!=pBody->m_mapInContactWith.end(); itOtherC++) {
          PhysicsParticleCollection *pBodyOther = (*itOtherC).first;
          int ocn = pBodyOther->m_contactDistanceNumber;
          if (ocn == contactLevel) { // in direct contact with a body having contactDistance contactLevel
            if ((pBody->m_contactDistanceNumber==-1)) { // ||
               //((pBody->m_contactDistanceNumber!=-1) && (pBody->m_contactDistanceNumber>(ocn+1)))) { // our body's contactDistance is still unset or found contact with lower level body
              pBody->m_contactDistanceNumber = ocn+1;
              if ((contactLevel+1) > biggestContactLevel) {
                biggestContactLevel = contactLevel;
              }
            } 
          }
        }
      }
    }
  }
  biggestContactLevel++;
  for (std::vector<PhysicsParticleCollection*>::iterator itC=pParticleCollections->begin(); itC!=pParticleCollections->end(); itC++) {
    PhysicsParticleCollection *pBody = (*itC);
    if (pBody->m_contactDistanceNumber == -1) {
      pBody->m_contactDistanceNumber = biggestContactLevel;
    }
  }
  return biggestContactLevel;
}

CRITICAL_SECTION critSect;

float GetShortestDistance(const D3DXVECTOR3 *pA1, const D3DXVECTOR3 *pA2, const D3DXVECTOR3 *pB, D3DXVECTOR3 *pPD=NULL) { // get shortest distance between line A1->A2 and B
  D3DXVECTOR3 dA;
  D3DXVECTOR3 dAB;
  D3DXVECTOR3 pS;
  D3DXVECTOR3 pD;

  // check if A1->A2 is too short -> distance to A1
  float epsilon = 0.000001f;
  dA = *pA2 - *pA1;
  float len = D3DXVec3LengthSq(&dA);
  dAB = *pB-*pA1;
  float lenABSq = D3DXVec3LengthSq(&dAB);
  if (len < epsilon || lenABSq < epsilon) {
    if (pPD) {
      *pPD = -dAB;
    }
    return sqrtf(lenABSq);
  }
  float u = D3DXVec3Dot(&dA, &dAB)/lenABSq;
  pS = u*dA + (*pA1);
  pD = (*pB)-pS;

  if (pPD) {
    *pPD = -pD;
  }

  return D3DXVec3Length(&pD);
}

void PhysicsParticleCollection::CalculateParticleForces(std::vector< PhysicsForce > *pForces, PhysicsGrid *pPhysicsGrid, std::map<PhysicsParticleCollection*, bool > *pColliding, int iteration, int numIterations, int level, int numLevels, float timeStep, int updateMode) {
  m_collisionForceCount = 0;

  bool bColliding = false;
  bool bOnLastIteration = (iteration == (numIterations-1)); // 

  D3DXVECTOR3 norm;
  D3DXVECTOR3 a;
  D3DXVECTOR3 b;

  D3DXVECTOR3 totalForce;
  D3DXVECTOR3 totalForceDamps;
  D3DXVECTOR3 totalForceFrict;

  D3DXVECTOR3 relPos;
  D3DXVECTOR3 relPosAB;
  D3DXVECTOR3 relPosNorm;
  D3DXVECTOR3 currentPos;
  D3DXVECTOR3 relSpeed;
  D3DXVECTOR3 relSpeedNorm;
  D3DXVECTOR3 fis;
  D3DXVECTOR3 fid;
  D3DXVECTOR3 fiv;
  D3DXVECTOR3 fif;

  D3DXVECTOR3 relSpeedTan;

  for (std::vector<PhysicsParticle>::iterator it=m_physicsParticles.begin(); it!=m_physicsParticles.end(); it++) {
    PhysicsParticle *pPart = &(*it);

    if (iteration == 0 && updateMode==0) {
      for (std::vector<PhysicsForce>::iterator itF=pForces->begin(); itF!=pForces->end(); itF++) {
        // F = ma
        (*it).AddForce(&((*it).GetMass()*(itF->m_acceleration)));
      }  
    }
    
    // compare to nearby
    int x, y;
    pPhysicsGrid->GetCellPtrAtPos((*it).GetPosition(), &x, &y);

    for (int yn=y-2; yn<y+3; yn++) {
      for (int xn=x-2; xn<x+3; xn++) {
        PhysicsGridCell *pPGC = pPhysicsGrid->GetCellPtrAtCell(xn, yn);
        if (updateMode >= 1) {
          if (pPGC->GetMinContactLevel() > level) {
            continue;
          }
        }
        std::vector<PhysicsParticle*> *gridPPNear = pPGC->GetPhysicsParticles();
        for (std::vector<PhysicsParticle*>::iterator itNear=gridPPNear->begin(); itNear!=gridPPNear->end(); itNear++) {
          if (pPart != (*itNear) && (*itNear)->GetPtrBelongsToCollection() != this) {
            // not comparing to itself or to the same collection
            if (updateMode >= 1) {
              // during the contact phase, compare only to same or lower level contact distance bodies
              if ((*itNear)->GetPtrBelongsToCollection()->m_contactDistanceNumber > level) {
                continue;
              }
            }
  // 0 - process collision (in case of collision, velocity reflecting force based on the elasticity), update velocity (gravity & external forces in) 
  // 1 - process contact (force to zero normal facing component of the velocity), update position (& candidate position with current velocity)

            D3DXVECTOR3 a = *(pPart->GetSpeed())*timeStep;
            D3DXVECTOR3 b = *((*itNear)->GetSpeed())*timeStep;
            relPosAB = (*(pPart->GetPosition())+a)-(*((*itNear)->GetPosition())+b);
            float distAB = D3DXVec3LengthSq(&relPosAB);
            relPos = (*(pPart->GetPosition()))-(*((*itNear)->GetPosition()));
            float diamSq = 2.0f*2.0f*pPart->GetDiameterSq();
            float diam = 2.0f*pPart->GetDiameter();
            float dist = D3DXVec3LengthSq(&relPos);

            D3DXVECTOR3 pD;
            float sd = GetShortestDistance(pPart->GetPosition(), &(*(pPart->GetPosition())+a), &(*((*itNear)->GetPosition())+b), &pD);
            if (sd < diam) { // dist < diamSq ||
         //    if (distAB < diamSq) { // dist < diamSq ||
/*
              D3DXVECTOR3 aPrev = *(pPart->GetPrevSpeed())*timeStep;
              D3DXVECTOR3 relPosPrevAB = (*(pPart->GetPrevPosition())+aPrev)-(*((*itNear)->GetPosition())+b);
              float distPrevAB = D3DXVec3LengthSq(&relPosPrevAB);
              if (distPrevAB >= diamSq) {
                pPart->SetPosition(pPart->GetPrevPosition());
              }
              */

              relSpeed = (*pPart->GetSpeed())-(*(*itNear)->GetSpeed());
              D3DXVECTOR3 norm = *(*itNear)->GetPosition()-*GetPosition();
           //   D3DXVECTOR3 norm = -*(*itNear)->GetNormal();
              float penetration = D3DXVec3Dot(&relSpeed,&norm);
              if (penetration >= 0.0f){
                D3DXVec3Normalize(&norm, &norm);
                penetration = D3DXVec3Dot(&relSpeed,&norm);

             //   float relSpeedLen = D3DXVec3Length(&relSpeed);
             //   float relPosLen = D3DXVec3Length(&relPos);

                bColliding = true;
                float elasticity = 0.350f;
                if (updateMode >= 1) {
                  elasticity = (-1.0f+(float)(iteration+1)/(numIterations))*1.0f;
                }
                relSpeedTan = relSpeed-penetration*norm;
                float otherMass = (*itNear)->GetMass(); //(*itNear)->GetPtrBelongsToCollection()->GetTotalMass();
                if ((updateMode>=1) && bOnLastIteration && (*itNear)->GetPtrBelongsToCollection()->m_contactDistanceNumber<level) {
                  otherMass = 10000000.0f; // "infinite mass"
                }
                float massMul;
                if (!(*itNear)->GetPtrBelongsToCollection()->GetIsDynamic()) {
                  massMul = 1.0f/(1.0f/it->GetMass());
                } else {
                  massMul = 1.0f/(1.0f/it->GetMass()+1.0f/otherMass);
                }
                massMul /= it->GetMass();
                fis = (-1.0f*norm*(penetration)*(1.0f+elasticity));

                if (updateMode >= 1) {
                //  fis += ((sqrtf(distAB)/diam)*relPosAB); //*0.020f/timeStep;
                  //fis = ((sqrtf(distAB)/diam)*relPosAB*penetration)*1.0f/timeStep;
                }
                //float k = 50.0f;
                //float n = -1.0f;
                //fis = -k*(diam-relPosLen)*relPos/relPosLen;
                //fis += n*relSpeed;
                fis -= relSpeedTan*0.50f; // some kind of friction



                fis = (massMul/(timeStep))*fis;

                m_collisionForceCount++;
                pPart->AddCollisionForce(&fis);
             //   pPart->AddCollisionForceWithPos(&fis, &(*(*itC)->GetPosition()+norm*sqrtf(dist)/diam));
              }
            }
          }
        }
      }
    }
  }
  if (updateMode == 0) {
    EnterCriticalSection(&critSect);
    (*pColliding)[this]=bColliding;
    LeaveCriticalSection(&critSect);
  }
}
HANDLE thread1 = NULL;
HANDLE thread2 = NULL;
HANDLE thread3 = NULL;
HANDLE thread4 = NULL;

unsigned int thread1Id = 0;
unsigned int thread2Id = 0;
unsigned int thread3Id = 0;
unsigned int thread4Id = 0;

HANDLE hEventWork1 = NULL;
HANDLE hEventWork2 = NULL;
HANDLE hEventWork3 = NULL;
HANDLE hEventWork4 = NULL;



unsigned PhysicsGrid::Thread1() {
  while(1) {
    WaitForSingleObject(hEventWork1, INFINITE);
    EnterCriticalSection(&critSect);
    bool bMoreData = m_dataQueue1.size();
    LeaveCriticalSection(&critSect);
    if (bMoreData) {
      EnterCriticalSection(&critSect);
      CalculateDataPF cdpf = m_dataQueue1.front();
      LeaveCriticalSection(&critSect);
      cdpf.pPPC->CalculateParticleForces(cdpf.pForces, cdpf.pPhysicsGrid, cdpf.pColliding, cdpf.iteration, cdpf.numIterations, cdpf.level, cdpf.numLevels, cdpf.timeStep, cdpf.updateMode);
      EnterCriticalSection(&critSect);
      m_dataQueue1.pop_front();
      LeaveCriticalSection(&critSect);
    }
    Sleep(0);
  }
}
unsigned PhysicsGrid::Thread2() {
  while(1) {
    WaitForSingleObject(hEventWork2, INFINITE);
    EnterCriticalSection(&critSect);
    bool bMoreData = m_dataQueue2.size();
    LeaveCriticalSection(&critSect);
    if (bMoreData) {
      EnterCriticalSection(&critSect);
      CalculateDataPF cdpf = m_dataQueue2.front();
      LeaveCriticalSection(&critSect);
      cdpf.pPPC->CalculateParticleForces(cdpf.pForces, cdpf.pPhysicsGrid, cdpf.pColliding, cdpf.iteration, cdpf.numIterations, cdpf.level, cdpf.numLevels, cdpf.timeStep, cdpf.updateMode);
      EnterCriticalSection(&critSect);
      m_dataQueue2.pop_front();
      LeaveCriticalSection(&critSect);
    }
    Sleep(0);
  }
}

unsigned PhysicsGrid::Thread3() {
  while(1) {
    WaitForSingleObject(hEventWork3, INFINITE);
    EnterCriticalSection(&critSect);
    bool bMoreData = m_dataQueue3.size();
    LeaveCriticalSection(&critSect);
    if (bMoreData) {
      EnterCriticalSection(&critSect);
      CalculateDataPF cdpf = m_dataQueue3.front();
      LeaveCriticalSection(&critSect);
      cdpf.pPPC->CalculateParticleForces(cdpf.pForces, cdpf.pPhysicsGrid, cdpf.pColliding, cdpf.iteration, cdpf.numIterations, cdpf.level, cdpf.numLevels, cdpf.timeStep, cdpf.updateMode);
      EnterCriticalSection(&critSect);
      m_dataQueue3.pop_front();
      LeaveCriticalSection(&critSect);
    }
    Sleep(0);
  }
}

unsigned PhysicsGrid::Thread4() {
  while(1) {
    WaitForSingleObject(hEventWork4, INFINITE);
    EnterCriticalSection(&critSect);
    bool bMoreData = m_dataQueue4.size();
    LeaveCriticalSection(&critSect);
    if (bMoreData) {
      EnterCriticalSection(&critSect);
      CalculateDataPF cdpf = m_dataQueue4.front();
      LeaveCriticalSection(&critSect);
      cdpf.pPPC->CalculateParticleForces(cdpf.pForces, cdpf.pPhysicsGrid, cdpf.pColliding, cdpf.iteration, cdpf.numIterations, cdpf.level, cdpf.numLevels, cdpf.timeStep, cdpf.updateMode);
      EnterCriticalSection(&critSect);
      m_dataQueue4.pop_front();
      LeaveCriticalSection(&critSect);
    }
    Sleep(0);
  }
}



// for each grid cell, check out the distance for particles in the cell and around the cell
// if near enough, apply collision force for the particle collection which the particle belongs to
void PhysicsGrid::CalculateParticleForces(std::vector<PhysicsParticleCollection*> *pParticleCollections, std::map<PhysicsParticleCollection*, bool > *pColliding, int iteration, int numIterations, int level, int numLevels, float timeStep, int updateMode) {
  
  int ik = 0;
  for (std::vector<PhysicsParticleCollection*>::iterator itC=pParticleCollections->begin(); itC!=pParticleCollections->end(); itC++) {

    if (!(*itC)->GetIsDynamic()) {
      continue;
    }
    if ((*itC)->m_contactDistanceNumber!=level) {
      continue;
    }
/*
    if (iteration != 0 && updateMode==0) {
      std::map<PhysicsParticleCollection *, bool>::iterator itP = (*pColliding).find((*itC));
      if (itP == (*pColliding).end()) {
        continue;
      }
      if (itP->second == false) {
        continue;
      }
    }
*/

    if (!thread1) {
      thread1 = (HANDLE)_beginthreadex(0, 0, &PhysicsGrid::Thread1, this, 0, &thread1Id);
      hEventWork1 = ::CreateEvent(0, FALSE, FALSE, 0);
      InitializeCriticalSection(&critSect);
    }
    if (!thread2) {
      thread2 = (HANDLE)_beginthreadex(0, 0, &PhysicsGrid::Thread2, this, 0, &thread2Id);
      hEventWork2 = ::CreateEvent(0, FALSE, FALSE, 0);
    }
    if (!thread3) {
      thread3 = (HANDLE)_beginthreadex(0, 0, &PhysicsGrid::Thread3, this, 0, &thread3Id);
      hEventWork3 = ::CreateEvent(0, FALSE, FALSE, 0);
    }
    if (!thread4) {
      thread4 = (HANDLE)_beginthreadex(0, 0, &PhysicsGrid::Thread4, this, 0, &thread4Id);
      hEventWork4 = ::CreateEvent(0, FALSE, FALSE, 0);
    }

  //  (*itC)->CalculateParticleForces(&m_forces, this, pColliding, iteration, numIterations, level, numLevels, timeStep, updateMode);
    
    CalculateDataPF cdpf;
    cdpf.iteration = iteration;
    cdpf.level = level;
    cdpf.numIterations = numIterations;
    cdpf.numLevels = numLevels;
    cdpf.timeStep = timeStep;
    cdpf.updateMode = updateMode;
    cdpf.pForces = &m_forces;
    cdpf.pPhysicsGrid = this;
    cdpf.pPPC = (*itC);
    cdpf.pColliding = pColliding;

    EnterCriticalSection(&critSect);
   //   m_dataQueue1.push_back(cdpf);
    if ((ik&3)==0) {
      m_dataQueue1.push_back(cdpf);
    } else if ((ik&3)==1) {
      m_dataQueue2.push_back(cdpf);
    } else if ((ik&3)==2) {
      m_dataQueue3.push_back(cdpf);
    } else if ((ik&3)==3) {
      m_dataQueue4.push_back(cdpf);
    }
    LeaveCriticalSection(&critSect);

    SetEvent(hEventWork1);
    SetEvent(hEventWork2);
    SetEvent(hEventWork3);
    SetEvent(hEventWork4);

    ik++;
  }
 
  bool bthreadsWorking = true;
  while (bthreadsWorking) {
    SetEvent(hEventWork1);
    SetEvent(hEventWork2);
    SetEvent(hEventWork3);
    SetEvent(hEventWork4);
    EnterCriticalSection(&critSect);
    bthreadsWorking = (m_dataQueue1.size() || m_dataQueue2.size() || m_dataQueue3.size() || m_dataQueue4.size());
    LeaveCriticalSection(&critSect);
    Sleep(0);
  }
  
}

void PhysicsParticleCollection::AddUniformForce(const D3DXVECTOR3 *force) {
  for (std::vector<PhysicsParticle>::iterator it=m_physicsParticles.begin(); it!=m_physicsParticles.end(); it++) {
    (*it).AddForce(force);
  }
}


void PhysicsParticleCollection::AddUniformImpulse(const D3DXVECTOR3 *force, float timeStep) {
  /*
  for (std::vector<PhysicsParticle>::iterator it=m_physicsParticles.begin(); it!=m_physicsParticles.end(); it++) {
    (*it).SetSpeed(&((*(*it).GetSpeed())+(*force)*0.1f));
  }
  */
  m_speed += (*force)*100.0f*timeStep;
}

void PhysicsParticleCollection::CalculateTotalMass() {
  m_totalMass = 0.0f;
  for (std::vector<PhysicsParticle>::iterator it=m_physicsParticles.begin(); it!=m_physicsParticles.end(); it++) {
    m_totalMass += it->GetMass();
  }
}

// Apply forces to all particles in this collection
void PhysicsParticleCollection::ApplyForces(float timeStep, float maxMoveDelta, int updateMode, int iteration, int numIterations, int level, int numLevels) {
  if (m_bDynamic) {
    if (m_contactDistanceNumber!=level) {
      return;
    }

    m_force = D3DXVECTOR3(0.0f, 0.0f, 0.0f);
    D3DXVECTOR3 torque = D3DXVECTOR3(0.0f, 0.0f, 0.0f);

    bool bOnLastIteration = (iteration == (numIterations-1));


    if (!m_bInertiaTensorInited) {
      InitInertiaTensor();
    }
    for (std::vector<PhysicsParticle>::iterator it=m_physicsParticles.begin(); it!=m_physicsParticles.end(); it++) {
      //it->ApplyForce(timeStep);
      D3DXVECTOR3 forceNow = *(it->GetForce());
      if (m_collisionForceCount) {
     //   for (std::vector<ForceWithPos>::iterator itFWP=it->m_collisionForcesWithPos.begin(); itFWP!=it->m_collisionForcesWithPos.end(); itFWP++) {
     //     forceNow += (*itFWP).m_force/(float)m_collisionForceCount*m_totalMass;
     //   }
        forceNow += (*(it->GetCollisionForce()))/(float)((float)m_collisionForceCount*1.0f+0.0f)*m_totalMass;
      }
      m_force += forceNow;

      D3DXVECTOR3 posInit = *(it->GetPositionInit())-m_centerInit;
   //   D3DXVECTOR3 pos = posInit;
   //   RotateByAngle(m_angle, &posInit);

      D3DXVECTOR3 torqueFromParticle = D3DXVECTOR3(0.0f, 0.0f, 0.0f);
    //  D3DXVec3Cross(&torqueFromParticle, &pos, it->GetForce());
      D3DXVECTOR3 torqueVec = *(it->GetPosition())-m_position;
      float td = D3DXVec3Dot(&torqueVec, &torqueVec);
      if (td > 0.001f) {
        D3DXVec3Cross(&torqueFromParticle, &torqueVec, &forceNow);
      }
      torque += torqueFromParticle;
/*
      for (std::vector<ForceWithPos>::iterator itFWP=it->m_collisionForcesWithPos.begin(); itFWP!=it->m_collisionForcesWithPos.end(); itFWP++) {
        D3DXVECTOR3 torqueVec = itFWP->m_pos-m_position;
        float td = D3DXVec3Dot(&torqueVec, &torqueVec);
        if (td > 0.0002f) {
          D3DXVec3Cross(&torqueFromParticle, &torqueVec, &(itFWP->m_force));
        }
        torque += torqueFromParticle;
      }
*/
    //  torque 
    }

    D3DXMATRIXA16 inertiaTensorInverse;
    D3DXMATRIXA16 inertiaTensorInverse0;
    float itDet = D3DXMatrixDeterminant(&m_inertiaTensor0);
    D3DXMatrixInverse(&inertiaTensorInverse0, &itDet, &m_inertiaTensor0);
    CalculateInertiaTensorInverse(&inertiaTensorInverse, &inertiaTensorInverse0, m_angle);
    D3DXVECTOR4 temp(torque.x, torque.y, torque.z, 1.0f);
    D3DXVECTOR4 rotateVelocityDelta;
    D3DXVec4Transform(&rotateVelocityDelta, &temp, &inertiaTensorInverse);

   // D3DXVECTOR4 temp2(0.0f, 1.0f, 0.0f, 0.0f);
   // D3DXVECTOR4 cross1;
   // D3DXVec4Cross(&cross1, &rotateVelocityDelta, &temp2);
   // float rotateVelocityDeltaAbs = D3DXVec4Length(&rotateVelocityDelta);
    float rotateVelocityDeltaAbs = rotateVelocityDelta.z*1.0f;
    float maxRotateVel = 10000.0f;
    if (rotateVelocityDeltaAbs > maxRotateVel) {
      rotateVelocityDeltaAbs = maxRotateVel;
    } else if (rotateVelocityDeltaAbs < -maxRotateVel) {
      rotateVelocityDeltaAbs = -maxRotateVel;
    }

    m_force *= 1.0f/m_totalMass;
    m_force.z = 0.0f;



    m_speed += m_force*timeStep*0.5f;
    m_angleSpeed += rotateVelocityDeltaAbs*timeStep*0.5f;
    if ((updateMode >= 1) && bOnLastIteration) {
   // if (bOnLastIteration) {
      m_position += m_speed*timeStep;
      m_angle += (m_angleSpeed)*timeStep;

      float velocity = D3DXVec3Length(&m_speed);
      if (velocity < 0.05f) {
        m_restingForFrames++;
      } else {
        m_restingForFrames=0;
      }
    }
    m_speed += m_force*timeStep*0.5f;
    m_angleSpeed += rotateVelocityDeltaAbs*timeStep*0.5f;




    // convert position, speed, angle & angular velocity back to the current particles
 //   UpdateParticleSpeedPos(bOnLastIteration, timeStep);
    UpdateParticleSpeedPos((updateMode >= 1) && bOnLastIteration, timeStep);
   // UpdateParticleSpeedPos(true, timeStep);
  }
}

void PhysicsParticleCollection::UpdateParticleSpeedPos(bool bUpdatePos, float timeStep) {
  for (std::vector<PhysicsParticle>::iterator it=m_physicsParticles.begin(); it!=m_physicsParticles.end(); it++) {
    D3DXVECTOR3 posInit = *(it->GetPositionInit())-m_centerInit;
    D3DXVECTOR3 pos = posInit;
    RotateByAngle(m_angle, &pos);
    D3DXVECTOR3 speedAngular;
    D3DXVec3Cross(&speedAngular, &D3DXVECTOR3(0.0f, 0.0f, m_angleSpeed), &pos);
    D3DXVECTOR3 speed = m_speed + speedAngular;
    it->SetSpeed(&speed);
    if (bUpdatePos) {
      pos += m_position;
    //  pos += speed*timeStep*1.0f;
      it->SetPosition(&pos);
      D3DXVECTOR3 norm = *(it->GetNormal());
      RotateByAngle(m_angle, &norm);
      it->SetNormalRot(&norm);
    }
  }
}

void PhysicsParticleCollection::InitInertiaTensor() {
  m_inertiaTensor0._11 = 0.0f;
  m_inertiaTensor0._21 = 0.0f;
  m_inertiaTensor0._31 = 0.0f;
  m_inertiaTensor0._41 = 0.0f;

  m_inertiaTensor0._12 = 0.0f;
  m_inertiaTensor0._22 = 0.0f;
  m_inertiaTensor0._32 = 0.0f;
  m_inertiaTensor0._42 = 0.0f;

  m_inertiaTensor0._13 = 0.0f;
  m_inertiaTensor0._23 = 0.0f;
  m_inertiaTensor0._33 = 0.0f;
  m_inertiaTensor0._43 = 0.0f;

  m_inertiaTensor0._14 = 0.0f;
  m_inertiaTensor0._24 = 0.0f;
  m_inertiaTensor0._34 = 0.0f;
  m_inertiaTensor0._44 = 1.0f;

  for (std::vector<PhysicsParticle>::iterator it=m_physicsParticles.begin(); it!=m_physicsParticles.end(); it++) {
    D3DXVECTOR3 posInit = *(it->GetPositionInit())-m_centerInit;
    m_inertiaTensor0._11 += it->GetMass()*(posInit.y*posInit.y+posInit.z*posInit.z);
    m_inertiaTensor0._22 += it->GetMass()*(posInit.z*posInit.z+posInit.x*posInit.x);
    m_inertiaTensor0._33 += it->GetMass()*(posInit.x*posInit.x+posInit.y*posInit.y);

    m_inertiaTensor0._12 += -it->GetMass()*posInit.x*posInit.y;
    m_inertiaTensor0._21 += -it->GetMass()*posInit.x*posInit.y;

    m_inertiaTensor0._13 += -it->GetMass()*posInit.x*posInit.z;
    m_inertiaTensor0._31 += -it->GetMass()*posInit.x*posInit.z;

    m_inertiaTensor0._23 += -it->GetMass()*posInit.y*posInit.z;
    m_inertiaTensor0._32 += -it->GetMass()*posInit.y*posInit.z;
  }
  m_inertiaTensor0._11 /= m_totalMass;
  m_inertiaTensor0._21 /= m_totalMass;
  m_inertiaTensor0._31 /= m_totalMass;

  m_inertiaTensor0._12 /= m_totalMass;
  m_inertiaTensor0._22 /= m_totalMass;
  m_inertiaTensor0._32 /= m_totalMass;

  m_inertiaTensor0._13 /= m_totalMass;
  m_inertiaTensor0._23 /= m_totalMass;
  m_inertiaTensor0._33 /= m_totalMass;
  m_bInertiaTensorInited = true;
}

