// effect.h


class EffectLayout;
struct EffectElement;
class EffectParam;

struct RtBuffer {
  std::string m_name;
  D3DFORMAT m_fmt;
};

class Effect {
private:
  string m_uniqueID;
  bool m_enabled;
public:
  enum TimelineType {
    camera=0, // sets or affects the global camera on this frame
    light, // per frame lights added by effect of this type
    render, // meshes, particles
    preproc, // some things happening just before the mixer, like shadowmap generation or ambient occlusion buffer generation
    mixer, // mix images from different buffers to one, like deferred mixer which produces one rgb buffer 
    postproc, // some fancy post proc effects like light balls and dof
    conversion // convert image with some mapping from 16 bpc to 8 bpc
  };
protected:
  float m_startTime;
  float m_endTime;
  float m_effectTime;

  float m_absTime;
  float m_timeStep;

  vector<string> m_params;

  EffectLayout *m_layout;

  DWORD m_clearFlags;
  std::vector<RtBuffer> m_rt16bBufferNames;
//  string m_rt32bBufferName;

  string m_filename;
  string m_type;

  TimelineType m_timelineType;

  string m_layoutFile;


  const struct sync_track *m_enabledSyncTrack;

  bool  m_bDoubleBuffer;

  double m_renderStartTime;
  double m_renderEndTime;

  double m_advanceStartTime;
  double m_advanceEndTime;


  bool m_bHasGlassRender;


public:
  Effect();
  ~Effect();

  bool ParseFromFile(string fileName);

  void SetEffectLayout(EffectLayout *pEffectLayout, string fileName);

  bool ReloadLayout();

  virtual bool ReloadEffect() { return true; }

  void SetStartTime(float t) { m_startTime = t; }
  void SetEndTime(float t) { m_endTime = t; }
  void SetTimeValues(float absTime, float timeStep);

  float GetStartTime() { return m_startTime; }
  float GetEndTime() { return m_endTime; }

  void SetFileAndType(string file, string type) { m_filename = file; m_type = type; }

  string GetFilename() { return m_filename; }
  string GetType() { return m_type; }

  string GetUniqueID() { return m_uniqueID; }

  std::string GetName();

  float GetEffectTime();
  float GetEffectTimeNormalized();

  float GetTimeStep() { return m_timeStep; }

  bool IsActive();

  void SetClearFlags(DWORD clearFlags);
  DWORD GetClearFlags();

  void SetRT16bBufferNames(std::vector<RtBuffer> rt16bBufferNames) { m_rt16bBufferNames = rt16bBufferNames; }
  std::vector<RtBuffer> GetRT16bBufferNames() { return m_rt16bBufferNames; }

  void SetDoubleBuffer(bool b) { m_bDoubleBuffer = b; }
  bool GetDoubleBuffer() { return m_bDoubleBuffer; }

  void SetParams(vector<string> params);

  const EffectParam *GetPR() const;

  TimelineType GetTimelineType() { return m_timelineType; }

  bool IsTimelineTypeCamera() { return m_timelineType == camera; }
  bool IsTimelineTypeLight() { return m_timelineType == light; }
  bool IsTimelineTypeRender() { return m_timelineType == render; }
  bool IsTimelineTypePreProcess() { return m_timelineType == preproc; }
  bool IsTimelineTypeMixer() { return m_timelineType == mixer; }
  bool IsTimelineTypePostProcess() { return m_timelineType == postproc; }
  bool IsTimelineTypeConversion() { return m_timelineType == conversion; }

  void SetDisabled() { m_enabled = false; }
  void SetEnabled() { m_enabled = true; }


  // sets effect into the original state
  virtual void Init() = 0;

  // advances the effect, called once per frame
  virtual void Advance() = 0;

  // renders all stuff belonging to the effect, called after the advance
  virtual int Render() = 0;

  // reload shaders from file
  virtual void ReloadShaders() = 0;

  virtual bool CheckShadersModifiedOnDisk() = 0;

  // own input handling function for this effect,
  // return false if no input should be given for the other effects handled after this
  virtual bool HandleInput() { return true; }

  // used to mark some effects like glow and particle pixel simulations not to be rendered when rendering the lights
  virtual bool DoRenderForLights() { return true; }

  void SetEnableSyncVariable(std::string varName);
  bool GetEnabledFromSync();

  bool IsEnabled() { return m_enabled && GetEnabledFromSync(); }

  void SetRenderStartTime(double t) { m_renderStartTime = t; }
  void SetRenderEndTime(double t) { m_renderEndTime = t; }
  double GetPrevRenderTime() { return m_renderEndTime-m_renderStartTime; }

  void SetAdvanceStartTime(double t) { m_advanceStartTime = t; }
  void SetAdvanceEndTime(double t) { m_advanceEndTime = t; }
  double GetPrevAdvanceTime() { return m_advanceEndTime-m_advanceStartTime; }


  DWORD GetD3DBlendFromString(const char *s, DWORD def=D3DBLEND_ONE);


  void SetHasGlassRender(bool b) { m_bHasGlassRender = b; }
  bool GetHasGlassRender() { return m_bHasGlassRender; }
};

struct Texture; 

class Light {
public:
  D3DXVECTOR4 m_position;
  D3DXVECTOR4 m_color;
  float m_intensity;
  float m_distance;
  float m_ss_size;
  float m_lightball_size;
  float m_lightball_sharpness;
 // D3DXMATRIXA16 m_viewMatrix;
  float m_viewMatrix[16];
  bool m_bCastShadows;
  Texture *m_pRtDepth;
  Light() {
    m_bCastShadows = false;
    m_pRtDepth = NULL;
  }
};

void ResetGlobalLights();

void AddToGlobalLights(Light *light);
vector<Light>* GetGlobalLights();

