#!/usr/bin/python

import sys
import random
import socket
from socket import timeout
import json
import time
import skyport
from operator import itemgetter
from worldmap import World
from players import Players

def weapon2resource(weapon):
    if 'laser' == weapon:
        return 'R'
    elif 'droid' == weapon:
        return 'C'
    elif 'mortar' == weapon:
        return 'E'
    return None

def resource2weapon(resource):
    if 'R' == resource:
        return 'laser'
    elif 'C' == resource:
        return 'droid'
    elif 'E' == resource:
        return 'mortar'
    return None

class ZomgBot:
	name = None
	gameworld = World()
	players = Players()
	transmitter = None
	mapwindow = None

	strangeDangerFactor = 0.6
	desireToMineCoefficent = 19


	def __init__(self, name, transmitter):
		self.name = name
		self.transmitter = transmitter

	def server_handshake(self):
		print "Got server handshake";

	def game_start(self, turn, map_obj, player_list):
		print "Game has started with {0} players".format(len(player_list))
		self.gameworld = World(map_obj)

		# Create player management and set reference in gameworld
		self.players = Players(player_list, self.name)
		self.gameworld.setPlayers(self.players)

		# Start debug window and set reference in gameworld
		#self.mapwindow = MapWindow(self.gameworld, self.players)
		#self.gameworld.setMapwindow(self.mapwindow)
		#self.mapwindow.setDrawActionTiles(False)
		#self.mapwindow.setDrawValueTiles(True)

		loadout = self.gameworld.loadout_analyzer()
		self.transmitter.send_loadout(loadout[0], loadout[1])

	def game_state(self, turn, map_obj, player_list):
		# Tell gameworld and players that stuff might have changed (clear graph and nodes)
		self.players.update(player_list)
		self.gameworld.update(map_obj)

		if(self.mapwindow != None):
			self.mapwindow.clearTiles()

		if(player_list[0]['name'] != self.name):
			return

		print "Got updated game state for turn {0}".format(turn)

		# If weapons are upgradable we should increment weapon level before calculating
		# so we account for more range and more damange
		upgradable_weapons = self.players.upgradable_weapons(self.name)
		for weapon in upgradable_weapons:
			self.players.me()['weapons'][weapon] += 1

		myPosition = self.players.me()['position']
		chosenTile = self.chooseBestTile(myPosition)

		#print "Best available ActionTile is:"
		#pprint.pprint(chosenTile)

		if (chosenTile == None):
			print "Unable to decide on where to go. No good options! Moving to a random tile!"
			possibleMoves = self.gameworld.possible_moves(myPosition, [], false);
			if (len(possibleMoves) > 0):
				self.transmitter.send_move(random.choice(possibleMoves))
			return

		if(self.mapwindow != None):
			self.mapwindow.hilightTile(chosenTile['position'], 1)

		actions = self.buildActionsForActionTile(chosenTile);

		if (actions == None):
			print "ActionTile has no path! Random move, it is!"
			possibleMoves = self.gameworld.possible_moves(myPosition, [], false);
			if (len(possibleMoves) > 0):
				self.transmitter.send_move(random.choice(possibleMoves))
			return

		hasAttacked = False
		for action in actions:
			if(action['action'] in ['laser', 'droid', 'mortar']):
				hasAttacked = True

		for weapon in upgradable_weapons:
			actions.insert(0, {'action': 'upgrade', 'weapon': weapon})

		if(not hasAttacked and len(actions) < 3):
			print "Only {0} actions in most desired actions, finding another desirable tile...".format(len(actions))
			augmentedPosition = self.applyActionsToState(myPosition, actions)
			self.gameworld.update()
			nextChosenTile = self.chooseBestTile(augmentedPosition)
			#print "Next best available ActionTile is:"
			#pprint.pprint(nextChosenTile)
			if(self.mapwindow != None):
				self.mapwindow.hilightTile(nextChosenTile['position'], 1)
			# Make sure to apply any actions to state since we dont account for them when they are reported
			additionalActions = self.buildActionsForActionTile(nextChosenTile)
			self.applyActionsToState(augmentedPosition, additionalActions[:(3 - len(actions))])
			actions.extend(additionalActions)

		#print "Found path: {0}".format(actions)

		# Reset our position back to starting point
		self.players.me()['position'] = myPosition

		for currentAction in actions[:3]:
			if(currentAction['action'] == "move"):
				moveNeeded = self.gameworld.find_move_needed(myPosition, currentAction['tile'])
				print "Moving from {0} to {1} ({2})".format(myPosition, currentAction['tile'], moveNeeded)
				self.transmitter.send_move(moveNeeded)
				myPosition = currentAction['tile']				

			elif (currentAction['action'] == "mortar"):
				print "Attacking {0} with mortar".format(currentAction['targetTile'])
				mortarOffsets = [currentAction['targetTile'][0] - myPosition[0], currentAction['targetTile'][1] - myPosition[1]]
				self.transmitter.attack_mortar(mortarOffsets[0], mortarOffsets[1])

			elif (currentAction['action'] == "laser"):
				print "Attacking {0} with laser".format(currentAction['direction'])
				self.transmitter.attack_laser(currentAction['direction'])
                        
			elif (currentAction['action'] == "droid"):
				print "Sending droid with sequence: {0}".format(currentAction['sequence'])
				self.transmitter.attack_droid(currentAction['sequence'])

			elif (currentAction['action'] == "mine"):
				print "Mining current tile"
				self.transmitter.mine()

			elif (currentAction['action'] == "upgrade"):
				print "Upgrading weapon: {0}".format(currentAction['weapon'])
				self.transmitter.upgrade(currentAction['weapon'])

	def applyActionsToState(self, startPosition, actions):
		pos = tuple(startPosition)
		for action in actions[:3]:
			if (action['action'] == "mine"):
				res = self.gameworld.mapData[pos[0]][pos[1]]
				self.gameworld.resources[res][pos] -= 1
				self.players.increment_resource(self.name, res)
				if self.gameworld.resources[res][pos] <= 0:
					self.gameworld.mapData[pos[0]][pos[1]] = 'G'

			if (action['action'] == "move"):
				pos = tuple(action['tile'])

		return pos


	def positionAfterActions(self, startPosition, actions):
		currentPosition = startPosition
		for action in actions:
			if(action['action'] == "move"):
				currentPosition = action['tile']

		return currentPosition

	def buildActionsForActionTile(self, chosenTile):
		chosenPath = self.gameworld.moveActionsForPath(chosenTile['path'])

		if(chosenPath == None):
			return None

		if(chosenTile['minable']):
			chosenPath.append({'action': 'mine'})

		if('best_action' in chosenTile):
			if (chosenTile['best_action'] == "laser"):
				chosenPath.append({'action': 'laser', 'direction': chosenTile['actions']['laser']['variables']['direction'] })

			elif (chosenTile['best_action'] == "mortar"):
				chosenPath.append({'action': 'mortar', 'targetTile': max(chosenTile['actions']['mortar']['options'], key=itemgetter('damage'))['variables']['target'] })
				
			elif (chosenTile['best_action'] == "droid"):
				chosenPath.append({'action': 'droid', 'sequence': max(chosenTile['actions']['droid']['options'], key=itemgetter('damage'))['variables']['moves'] })

		return chosenPath

#	def displayActionTileValues(self):
#		# Draw values on tiles with actions
#		short_action = {'laser':'L', 'mortar': 'M', 'droid': 'D'}
#		if(self.mapwindow != None):
#			self.mapwindow.clearTiles()
#			for j in range(self.gameworld.size_j):
#				for k in range(self.gameworld.size_k):
#					#print j,k,self.gameworld.actionTiles[j][k]['hasAction']
#					if self.gameworld.actionTiles[j][k]['hasAction']:
#						self.mapwindow.tileText((j,k),short_action[self.gameworld.actionTiles[j][k]['best_action']]+str(self.gameworld.actionTiles[j][k]['damage']))

	def chooseBestTile(self, myPosition):
		taken_tiles = self.players.playerPositionsWithoutMe()

		me = self.players.me()
		me['position'] = myPosition
		#print "Updating action tiles with {0} enemies and me {1}".format(self.players.enemies(), me)
		self.gameworld.populate_action_tiles(self.players.enemies(), me)

		desirable_tiles = []
		# Pick up the tiles that has desirable actions
		for j, row in enumerate(self.gameworld.actionTiles):
			for k, tile in enumerate(row):
				if (tile['hasAction'] == 1):
					tile['position'] = [j, k]
					desirable_tiles.append(tile)

		if (len(desirable_tiles) == 0):
			print "Unable to find out what to do this round"
			return None

		reachableThisRound = []
		reachableLater = []

		# Remove any tiles that are not reachable and store the path for the rest of them
		for tile in desirable_tiles:
			path_to_tile = self.gameworld.search(myPosition, tile['position'], taken_tiles)

			if (path_to_tile != None):
				tile['path'] = path_to_tile
				if(len(path_to_tile) < 3):
					reachableThisRound.append(tile)
				else:
					reachableLater.append(tile)

		#print "Reachable this round: {0} tiles".format(len(reachableThisRound))
		#print "Reachable later: {0} tiles".format(len(reachableLater))

		# Find what list we want to inspect. Are there any tiles we can reach this round?
		if (len(reachableThisRound) > 0):
			desirable_tiles = reachableThisRound
		elif(len(reachableLater) > 0):
			desirable_tiles = reachableLater
		else:
			desirable_tiles = []

		for tile in desirable_tiles:
			if ('best_action' in tile):
				tile['tileValue'] = tile['damage'] + ((tile['damage'] * 0.2) * (2 - min(len(tile['path']), 2)))

			if (tile['minable']):
				tile['tileValue'] = max((tile['tileValue'] if ('tileValue' in tile) else 0), tile['need'] * self.desireToMineCoefficent)

		# Check which tiles are hittable from enemy positions
		self.gameworld.calculate_stranger_danger()

		for tile in desirable_tiles:
			# Modify value by how far away the tile is
			tile['tileValue'] = tile['tileValue'] - len(tile['path'])
			# STRANGER DANGER! Avoid tiles the enemy can hit us...
			p = tile['position']
			tile['tileValue'] -= (self.gameworld.strangerDanger[p[0]][p[1]] * self.strangeDangerFactor)

		for tile in desirable_tiles:
			if not 'tileValue' in tile:
				tile['tileValue'] = 0

		chosenTile = max(desirable_tiles, key=itemgetter('tileValue'))

		return chosenTile

	def error_received(self, errmsg):
		print "Error received: {0}".format(errmsg)

	def choose_actions(self, action_type, who, rest_data):
		print "{who} is choosing actions: {action_type} ({rest_data})".format(who = who, action_type = action_type, rest_data = rest_data)
		pos = tuple(self.players.players[who]['position'])
		if 'move' == action_type:
			direction = rest_data['direction']
			self.players.players[who]['position'] = self.gameworld.new_position_using_direction(pos, direction)

		elif 'mine' == action_type:
			if(self.name != who):
				res = self.gameworld.mapData[pos[0]][pos[1]]
				self.gameworld.resources[res][pos] -= 1
				self.players.increment_resource(who, res)
				if self.gameworld.resources[res][pos] <= 0:
					self.gameworld.mapData[pos[0]][pos[1]] = 'G'

		elif 'upgrade' == action_type:
			weapon = rest_data['weapon']
			level = self.players.players[who]['weapons'][weapon]

			if(self.name == who):
				level -= 1

			if 1 == level:
				self.players.decrement_resource(who, weapon2resource(weapon), 4)
			elif 2 == level:
				self.players.decrement_resource(who, weapon2resource(weapon), 5)

	def turn_ended(self):
		print "Turn has ended";


class TcpBotRunner:
	inputbuf = ""
	previous_message = ""

	def __init__(self, host, port):
		self.sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
		self.sock.connect((host, port))

		self.receiver = skyport.SkyportReceiver()
		self.transmitter = skyport.SkyportTransmitter(self.send_line)

		self.bot = ZomgBot("ZomgBot", self.transmitter)

		self.receiver.handler_handshake_successful = self.bot.server_handshake
		self.receiver.handler_error = self.bot.error_received
		self.receiver.handler_gamestate = self.bot.game_state
		self.receiver.handler_gamestart = self.bot.game_start
		self.receiver.handler_action = self.bot.choose_actions
		self.receiver.handler_endturn = self.bot.turn_ended

		self.transmitter.send_handshake(self.bot.name)

	def read_packet(self):
		try:
			self.inputbuf += self.sock.recv(1)
		except socket.timeout as e:
			return None
		except socket.error as e:
			return None
		try:
			characters_to_read = self.inputbuf.index("\n")
			line = self.inputbuf[0:characters_to_read] # removing the newline
			self.inputbuf = self.inputbuf[characters_to_read+1:len(self.inputbuf)]
			return line
		except ValueError as f:
			return None


	def send_line(self, line):
		self.previous_message = line
		#print("sending: '%s'" % line)
		if self.sock.sendall(line + "\n") != None:
			print("Error sending data!")

	def run(self):
		while True:
			line = self.read_packet()
			if line != None:
				try:
					self.receiver.parse_line(line)
				except KeyboardInterrupt:
					print "OK, I'm quitting!"
				except Exception as e:
					print "error was:"
					import pprint
					pprint.pprint(e)

if(len(sys.argv) == 3):
	botrunner = TcpBotRunner(sys.argv[1], int(sys.argv[2]))
	botrunner.run()
else:
	print "ERROR! Missing command line parameters..."
	print "Usage: {0} <server ip> <server port>".format(sys.argv[0])
