#!/usr/bin/env python
#
# Oscilloscope X-Y Mode Emulator
#
#(c)2007 Felipe Sanches
#(c)2007 Leandro Lameiro
#licensed under GNU GPL v3 or later

# A bunch of bug fixes and enhancements
# Michael Sparmann, 2009

# additional improvements and optimizations
# Luis Gonzalez, 2014

import pyaudio
import wave
import struct
import pygame
import sys
import math
import getopt
import subprocess as sp
import pygame.surfarray as surfarray

# default options 
exportVideo = False;	# export frames to video 
showGrid = False;		# background oscilloscope grid 
listenInput = True;		# listen to the wave
musix = "beams_4ch.wav" # input WAV to play
app_title = "Oscilloscope X-Y Mode Emulator"

# get command line arguments
helpText = app_title+' \n\n  scope_emu.exe -i <inputfile> -g <showGridBoolean> -l <listenInputBoolean>'
try:
	opts, args = getopt.getopt(sys.argv[1:],"h:i:g:l:e:o:",["help","ifile","grid","listen","export"])	 
except getopt.GetoptError:
  print helpText
  sys.exit(2)
for opt, arg in opts:
	if opt in ('-h','--help'): 
		print helpText
		sys.exit()
	elif opt in ('-i','--ifile'): 
		musix = arg.strip()
	elif opt in ('-g','--grid'): 
		showGrid = arg.strip().lower()=="true"
	elif opt in ('-l','--listen'): 
		listenInput = arg.strip().lower()=="true"
	elif opt in ('-e','--export'): 
		exportVideo = arg.strip().lower()=="true"

W = 720
H = 720
SIZE = (W,H)
		
FFMPEG_BIN = "ffmpeg.exe" # on Windows
command = [ FFMPEG_BIN,
		'-y', # (optional) overwrite output file if it exists
		'-f', 'rawvideo', #input format is always rawvideo
		'-vcodec','rawvideo',  #rawvideo, libx264 , mjpeg
		'-s', '720x720', # size of one frame
		'-pix_fmt', 'rgb32',
		'-r', '30', # frames per second
		'-i', '-', # The input comes from a pipe
		'-an', # Tells FFMPEG not to expect any audio
		#'-c:v', 'libx264',
		#'-crf', '23',
		'-b:v','20M',  # bit rate		
		'-q','2',  #quality
        'my_output_videofile.avi',
		#'-codec:v', 'mjpeg',
		#'-q:v', '2',
		#'-codec:v','mjpeg',  #rawvideo, libx264 , mjpeg
		#'-vb','300K',  #bitrate
		#'-c','mjpeg',  # bit rate		
		#'-b','300K',  # bit rate		
		#'-b:v','300K'  # bit rate		
		]

if exportVideo:
	pipe = sp.Popen( command, stdin = sp.PIPE, stdout = sp.PIPE)		

# green
DOT1COLOR = (63,255,191)
DOT2COLOR = (15,127,47)
DOT3COLOR = (11,95,35)
DOT4COLOR = (7,31,23)
DOT5COLOR = (3,23,11)
DOT6COLOR = (1,15,5)
DOT7COLOR = (0,7,3)
GRIDCOLOR = (0,31,63)
BGCOLOR = (0,63,91)

FPS = 30
SUBFRAMES = 1
ALPHA = 128
DOTALPHA = 23
RW = W/2
RH = H/2

try:
	wro = wave.open(musix)
except:
	print '\nPlease use a valid input wave file:\n'
	print 'http://luis.net/projects/scope/beams_4ch.wav'
	sys.exit(1)

SAMPLINGRATE = wro.getframerate()
READ_LENGTH = (SAMPLINGRATE/FPS/SUBFRAMES)
READ_LENGTH_QUAD = READ_LENGTH * 4;
NFRAMES = wro.getnframes()
nchannels = wro.getnchannels()
width = wro.getsampwidth()

print(helpText)
print("\nsampling rate = {SAMPLINGRATE} Hz \nlength = {READ_LENGTH}\nchannels = {nchannels}\nsample width = {width} bytes".format(**locals()))

# realtime audio streaming
if listenInput:
	print("loading sound: "+musix)
	p = pyaudio.PyAudio()
	stream = p.open(format=p.get_format_from_width(width),channels=2,rate=SAMPLINGRATE,output=True, frames_per_buffer=READ_LENGTH)

pygame.init()
black = 0, 0, 0,93
screen = pygame.display.set_mode(SIZE,pygame.HWSURFACE|pygame.ASYNCBLIT)
pygame.display.set_caption(app_title)

#pygame.mouse.set_visible(0)
clock = pygame.time.Clock()

dot = pygame.Surface((7,7))
dot.set_alpha(DOTALPHA)
dot.fill(BGCOLOR)
dot.fill(DOT7COLOR, pygame.Rect(0,0,7,7))
dot.fill(DOT6COLOR, pygame.Rect(1,0,5,7))
dot.fill(DOT6COLOR, pygame.Rect(0,1,7,5))
dot.fill(DOT5COLOR, pygame.Rect(1,1,5,5))
dot.fill(DOT4COLOR, pygame.Rect(2,1,3,5))
dot.fill(DOT4COLOR, pygame.Rect(1,2,5,3))
dot.fill(DOT3COLOR, pygame.Rect(2,2,3,3))
dot.fill(DOT2COLOR, pygame.Rect(3,2,1,3))
dot.fill(DOT2COLOR, pygame.Rect(2,3,3,1))
dot.fill(DOT1COLOR, pygame.Rect(3,3,1,1))
#dot=dot.convert_alpha()

grid = pygame.Surface(SIZE)
grid.set_alpha(128)
grid.fill((0,0,0))

if showGrid:
	grid.fill(BGCOLOR)
	for x in range(10):
		pygame.draw.line(grid, GRIDCOLOR, (x*W/10,0), (x*W/10,W))
		pygame.draw.line(grid, GRIDCOLOR, (0 , x*H/10), (W , x*W/10))

	#cross hairs
	pygame.draw.line(grid, GRIDCOLOR, (W/2,0), (W/2,W), 3)
	pygame.draw.line(grid, GRIDCOLOR, (0 , H/2), (W , H/2), 3)

	# hairs
	for x in range(W/4):
		pygame.draw.line(grid, GRIDCOLOR, (x*W/128,H/2-3), (x*W/128,H/2+3))
		pygame.draw.line(grid, GRIDCOLOR, (W/2 - 3, x*H/128), (W/2 + 3, x*H/128))

frames = wro.readframes(READ_LENGTH)
framelen = len(frames)

while True:
	# escape key to quit
	for event in pygame.event.get():
		if event.type == pygame.QUIT :	sys.exit()
	if event.type == pygame.KEYDOWN :
		if event.key == pygame.K_ESCAPE :	sys.exit()
	
	screen.blit(grid, (0,0))
	newdata = ""
	ii=4
	for i in range(0,READ_LENGTH_QUAD,4):
		if i == framelen: sys.exit()
		# stop reading x-y graphic data after the first two channels
		if (i & 4 ==0) :
			r = struct.unpack('hh', frames[i:i+4])
			x = (r[1]* W >> 16) + RW 
			y = (-r[0]* H >> 16) + RH
			if exportVideo:  #export option needs pixels reversed
				y = (r[1]* W >> 16) + RW 
				x = (-r[0]* H >> 16) + RH
			screen.blit(dot, (x,y), None, pygame.BLEND_ADD)
		if (i & 2 ==0 and listenInput) :
			newdata += frames[ii:ii+4]
			# skip second set of channels if a four channel demo
			ii+=nchannels<<1

	if listenInput:
		# lets hear perfectly synchronized audio
		stream.write(newdata)				
	
	# update the full display Surface to the screen
	pygame.display.flip()	

	if exportVideo:
		imageArray = surfarray.array2d(screen)
		pipe.stdin.write(imageArray.tostring())	

	frames = wro.readframes(READ_LENGTH)		
	clock.tick(FPS)

if listenInput:	
	stream.stop_stream()
	stream.close()
	p.terminate()