(function(){

	BigNumber.config({
		DECIMAL_PLACES: 3,
		ROUNDING_MODE: BigNumber.ROUND_HALF_DOWN
	});

	var Game = {}, Canvas = {};

	Game.elements = {
		score:  document.querySelector("#score"),
		target: document.querySelector("#target"),
		reset: document.querySelector("#reset"),
		cheat:  document.querySelector("#cheat"),
		units:  document.querySelectorAll(".unit"),
		unit:   function(type) {
			return document.querySelector('#units-' + type);
		}
	};

	Game.state = {
		score: new BigNumber(0),
		perk:  new BigNumber(0),
		perktype: null,
		units: {}
	};

	Game.stateLoader = {
		save: function(){
			var state = {};

			state.score = Game.state.score.toString();
			state.units = {};
			for (var i in Game.state.units) {
				state.units[i] = {
					count:    Game.state.units[i].count.toString(),
					lastCost: Game.state.units[i].lastCost.toString(),
					nextCost: Game.state.units[i].nextCost.toString()
				};
			}

			if (!!localStorage) {
				localStorage.setItem('haxclick-state', JSON.stringify(state));
			}
		},

		load: function(){
			var state = null;

			if (!!localStorage) {
				state = JSON.parse(localStorage.getItem('haxclick-state'));
			}

			if (!!state) {
				if (!!state.score) {
					Game.state.score = new BigNumber(state.score);
				}
				if (!!state.units) {
					for (var i in state.units) {
						Game.state.units[i] = {
							count:    new BigNumber(state.units[i].count),
							lastCost: new BigNumber(state.units[i].lastCost),
							nextCost: new BigNumber(state.units[i].nextCost)
						}
					}
				}
			}

			return true;
		},

		setSave: function(){
			setInterval(function(){
				Game.stateLoader.save();
			}, 60 * 1000);

			window.onbeforeunload = function(){
				Game.stateLoader.save();
			};

			return true;
		},

		clear: function(){
			localStorage.setItem('haxclick-state', null);
			window.onbeforeunload = null;
			window.location.reload();
		}
	};

	Game.classes = {
		unit: function(params){
			this.type = params.type;
			this.name = (typeof(params.name) != 'undefined') ? params.name : this.type;

			// Base cost
			this.base = (typeof(params.base) != 'undefined') ? params.base : 1;

			// Cost multiplier
			this.multiplier = (typeof(params.multiplier) != 'undefined') ? params.multiplier : 0.125;

			// Additional score per second
			this.sps = (typeof(params.sps) != 'undefined') ? params.sps : 1;

			// Additional score per click
			this.spc = (typeof(params.spc) != 'undefined') ? params.spc : 0;

			this.singular = (typeof(params.singular) != 'undefined') ? params.singular : this.name;
			this.multiple = (typeof(params.multiple) != 'undefined') ? params.multiple : this.name + "s";

			this.print = function(){
				var amount = Game.state.units[this.type].count
					.round(0, BigNumber.ROUND_HALF_UP)
					.toString(),

					nc = Game.state.units[this.type].nextCost
					.round(0, BigNumber.ROUND_HALF_UP)
					.toString();

				var amountText = amount + " " + (amount == Game.constants.one ? this.singular : this.multiple);
				var nextCostText = "cost of 1 more: " + Game.format.cost(nc);

				return {
					count: amountText,
					cost:  nextCostText
				};
			};

			this.canBuy = function(){
				var nc = Game.state.units[this.type].nextCost,
					sc = Game.state.score;

				return sc.comparedTo(nc) >= 0;
			};
		}
	};

	Game.units = {
		scanner: new Game.classes.unit({
			type: 'scanner',
			name: 'port scanner',
			base: Math.pow(8, 1),
			sps: 0,
			spc: (0.125 / 8*4)
		}),

		worm: new Game.classes.unit({
			type: 'worm',
			base: Math.pow(8, 2),
			sps: 1
		}),

		cracker: new Game.classes.unit({
			type: 'cracker',
			name: 'password cracker',
			base: Math.pow(8, 3),
			sps: Math.pow(8, 1)
		}),

		icebreaker: new Game.classes.unit({
			type: 'icebreaker',
			base: Math.pow(8, 4),
			sps: Math.pow(8, 2)
		}),

		ai: new Game.classes.unit({
			type: 'ai',
			name: 'helper AI',
			base: Math.pow(8, 5),
			sps: Math.pow(8, 3)
		}),

		pk: new Game.classes.unit({
			type: 'pk',
			name: 'private key',
			base: Math.pow(8, 6),
			sps: Math.pow(8, 4),
			spc: Math.pow(8, 2)
		})
	};

	Game.action = {
		unitNextCost: function(type, count) {
			if (Game.state.units[type].count == 0) {
				return new BigNumber(Game.units[type].base);
			}

			var currentCost = Game.state.units[type].nextCost;
			if (currentCost == 0) currentCost = new BigNumber(Game.units[type].base);

			var nextCost = currentCost;

			for (var i = 0; i < count; i++) {
				var mult = new BigNumber(Game.units[type].multiplier);
				var increase = nextCost.times(mult);
				nextCost = nextCost.plus(increase);
			}

			return nextCost.round(3, BigNumber.ROUND_HALF_DOWN);
		},

		addUnit: function(type, amount, dontDisplay){
			if (!Game.state.units[type]) {
				Game.state.units[type] = {
					count:    Game.constants.zero,
					lastCost: Game.constants.zero,
					nextCost: new BigNumber(Game.units[type].base),
				};
			}

			if (amount > 0) {
				var nc = Game.state.units[type].nextCost,
					sc = Game.state.score,
					cn = Game.state.units[type].count;

				if (nc.comparedTo(sc) <= 0) {
					Game.state.score = sc.minus(nc);
					Game.state.units[type].count = cn.plus(new BigNumber(amount)).round(3, BigNumber.ROUND_HALF_DOWN);
					Game.state.units[type].nextCost = Game.action.unitNextCost(type, amount);
				}
			}

			if (!dontDisplay) {
				Game.action.display();	
			}
		},

		addScore: function(number){
			var upScore = new BigNumber(number);

			if (!upScore.equals(Game.constants.zero)) {
				Game.state.score = Game.state.score.plus(upScore);
				Game.action.display();
			}
		},

		click: function(){
			var base = 1;

			if (!!arguments[0]) {
				base = parseInt(arguments[0], 10);
			}

			var spc = Game.utils.getSpc();

			Game.action.addScore(spc.plus(base));
			Game.ui.update();
		},

		unitClick: function(type){
			Game.action.addUnit(type, 1);
			Game.ui.update();
		},

		second: function() {
			var sps = Game.utils.getSps();
			Game.action.addScore(sps);
			Game.ui.update();
		},

		formatCost: function(score){
			return Game.format.cost(score);
		},

		previousScore: 0,

		display: function(nofx){
			var displayScore = Game.action.formatCost(Game.state.score);

			Game.elements.score.innerText = displayScore;

			Object.keys(Game.units).map(function(type){
				var unitDisplay = Game.units[type].print();
				for (var i in unitDisplay) {
					Game.elements.unit(type).querySelector("." + i).innerText = unitDisplay[i];
				}
			});

			if (!nofx) {
				Game.ui.canvas(Game.state.score.toString());
				Game.ui.body(Game.state.score.toString());
			}
		},

		resetPerk: function(){
			Game.state.perkValue = new BigNumber(0);
			Game.state.perkType = null;

			var elem = document.getElementById('perk');
			if (!!elem) elem.parentNode.removeChild(elem);

			Game.timers.perk = setTimeout(function(){
				Game.action.setupPerk();
			}, Math.random() * 300 * 1000);
		},

		setupPerk: function(){
			var perkValue = Math.random() > 0.9 ? 666 : 7;
			var perkType = Math.random() > 0.9 ? 'sps' : 'spc';

			var clicked = false;

			var link = document.createElement('a');
			link.id = 'perk';
			link.href = '#';
			link.tabindex = '0';
			link.onclick = function(e){
				e.preventDefault();

				var elem = document.getElementById('perk');
				if (!!elem) elem.parentNode.removeChild(elem);

				clicked = true;

				Game.state.perkValue = new BigNumber(perkValue);
				Game.state.perkType = perkType;

				setTimeout(function(){
					Game.action.resetPerk();
				}, 45 * 1000 + (1000 * 15 * Math.random()));
			};
			link.innerText = Math.random().toString(36).substring(2);

			document.getElementById('ui').appendChild(link);

			setTimeout(function(){
				if (!clicked) {
					Game.action.resetPerk();
				}
			}, 10 * 1000);
		}
	};

	Game.timers = {
		second: null,
		perk: null
	};

	Game.events = {
		init: function(){
			Game.events.initUnits();

			Game.timers.second = setInterval(function(){
				Game.action.second();
			}, 1000);

			Game.action.resetPerk();

			Game.elements.target.onclick = Game.events.targetClick;
			Game.elements.target.onkeyup = Game.events.keyUpEvent(Game.events.targetClick);

			Game.elements.reset.onclick = Game.events.resetClick;
			Game.elements.reset.onkeyup = Game.events.keyUpEvent(Game.events.resetClick);

			var units = Game.elements.units;
			for (var i = 0; i < units.length; i++) {
				units[i].onclick = Game.events.unitClick;
				units[i].onkeyup = Game.events.keyUpEvent(Game.events.unitClick);
			}
		},

		keyUpEvent: function(targetAction){
			return function(e){
				e.preventDefault();
				if (e.keyCode == 13) {
					Game.ui.keyupHighlight(e.currentTarget);
					targetAction(e);
				}
			};
		},

		targetClick: function(e){
			e.preventDefault();
			Game.action.click();
		},

		cheatClick: function(e){
			e.preventDefault();
			Game.action.click(512);
		},

		unitClick: function(e){
			e.preventDefault();
			var type = e.currentTarget.id.replace(/units-/, '');
			Game.action.unitClick(type);
		},

		resetClick: function(e){
			e.preventDefault();
			Game.stateLoader.clear();
		},

		initUnits: function(){
			Object.keys(Game.units).map(function(type){
				Game.action.addUnit(type, 0, true);
			});
		}
	};

	Game.format = {
		cost: function(score){
			var currentScore = new BigNumber(score),
				result 		 = Game.constants.zero,
				previous     = Game.constants.zero,
				scaleText    = "",
				found        = false;

			if (currentScore.comparedTo(Game.constants.scoreDivisor) > 0) {
				for (var i in Game.constants.scaleLevels) {
					var cv = Game.constants.scaleLevels[i].value;

					if (currentScore.comparedTo(cv) > 0) {
						previous = Game.constants.scaleLevels[i];
					}
					else {
						currentScore = currentScore.div(previous.value);
						scaleText = previous.text;
						found = true;
						break;
					}
				}

				if (!found) {
					var scaleKeys = Object.keys(Game.constants.scaleLevels);
					var index = scaleKeys[scaleKeys.length - 1];
					var last = Game.constants.scaleLevels[index];

					currentScore = currentScore.div(last.value);
					scaleText = last.text;
				}
			}

			var renderScore = currentScore.round(3, BigNumber.ROUND_HALF_DOWN).toString();

			if (scaleText == "") {
				var bits = renderScore.split(".");
				var newScore = bits[0];
				if (!!bits[1]) {
					bits[1] = Math.floor(bits[1] / 125) * 125;
					if (bits[1] > 0) {
						newScore = newScore + "." + bits[1];
					}
				}
				renderScore = newScore;
			}

			var byteText = ((currentScore.comparedTo(Game.constants.one) == 0) ? "byte" : "bytes"),
				suffix = scaleText + byteText;

			return [
				renderScore,
				suffix
			].join(" ");
		}
	};

	Game.constants = {
		scales: [
			"kilo",
			"mega",
			"giga",
			"tera",
			"peta",
			"exa",
			"zetta",
			"yotta"
		],

		zero: new BigNumber(0),
		one: new BigNumber(1),
		scoreDivisor: new BigNumber(1024)
	};

	Game.constants.scaleLevels = Game.constants.scales.map(function(scale, index){
		var power = index + 1;
		return {
			text: scale,
			value: Game.constants.scoreDivisor.toPower(power)
		};
	});

	Game.ui = {
		hasClass: function(elem, className) {
			return elem.className.indexOf(className) >= 0;
		},

		addClass: function(elem, className) {
			if (!Game.ui.hasClass(elem, className)) {
				elem.className += ' ' + className;
			}
		},

		removeClass: function(elem, className) {
			if (Game.ui.hasClass(elem, className)) {
				elem.className = elem.className.replace(className, ' ');
			}
		},

		keyupHighlight: function(elem){
			Game.ui.addClass(elem, 'keypress');

			if (Game.ui.hasClass(elem, 'keypress')) {
				setTimeout(function(){
					Game.ui.removeClass(elem, 'keypress');
				}, 50);
			}
		},

		update: function(){
			Object.keys(Game.units).map(function(type){
				var canBuy = Game.units[type].canBuy();

				var elem = Game.elements.unit(type);

				if (canBuy) {
					Game.ui.removeClass(elem, 'disabled');
				}
				else {
					Game.ui.addClass(elem, 'disabled');
				}

				var unitDisplay = Game.units[type].print();
				for (var i in unitDisplay) {
					elem.querySelector("." + i).innerText = unitDisplay[i];
				}
			});
		},

		previousScore: 0,

		fx: {},

		canvas: function(newScore){
			var newScore = parseInt(newScore, 10),
				diff     = Math.floor(newScore - Game.ui.previousScore + Math.random());

			if (diff > 0) {

				if (diff > 50) diff = 50;

				if (diff > 10 && !Game.ui.fx['bright']) {
					Canvas.action.addFilter('bright', diff / 200 * 40);
					Game.ui.fx['bright'] = 1;
				}
				else if (!!Game.ui.fx['bright']) {
					Canvas.action.removeFilter('bright');
					delete Game.ui.fx['bright'];
				}

				if (!Game.ui.fx['glitch1'] && Math.floor((Math.random() + newScore)) % 10 == 0) {

					Game.ui.fx['glitch1'] = 1;

					var glitch1Action = (4 * Math.random() + 1) * 2

					Canvas.action.addFilter('glitch1', glitch1Action);
					setTimeout(function(){
						Canvas.action.removeFilter('glitch1');
						delete Game.ui.fx['glitch1'];
					}, 500);
				}

				if (!Game.ui.fx['noClear'] && Math.random() > 0.9 && diff > 5) {
					Game.ui.fx['noClear'] = 1;
					Canvas.action.addFilter('noClear', 0);
					setTimeout(function(){
						Canvas.action.removeFilter('noClear');
						delete Game.ui.fx['noClear'];
					}, 500);
				}

				if (!Game.ui.fx['noKeep'] && Math.random() > 0.9 && diff > 5) {
					Game.ui.fx['noKeep'] = 1;
					Canvas.action.addFilter('noKeep', 0);
					setTimeout(function(){
						Canvas.action.removeFilter('noKeep');
						delete Game.ui.fx['noKeep'];
					}, 500);
				}

				if (!Game.ui.fx['glitch2'] && Math.random() > 0.9 && diff > 4) {

					Game.ui.fx['glitch2'] = 1;

					var glitch2Action = (4 * Math.random() + 1) * 2

					Canvas.action.addFilter('glitch2', glitch2Action);
					setTimeout(function(){
						Canvas.action.removeFilter('glitch2');
						delete Game.ui.fx['glitch2'];
					}, 500);
				}

				for (var i = 0; i < diff; i++) {
					var delay = 0;

					if (i > 0 && diff > 1) {
						delay = 1000 * (i / diff);
					}

					setTimeout(function(){
						Canvas.action.add('char');
					}, delay);
				}
			}

			Game.ui.previousScore = newScore;
		},

		strobeOn: 50,

		body: function(newScore){
			var newScore = parseInt(newScore, 10),
				elem     = document.getElementsByTagName('body')[0];

			if (newScore % Game.ui.strobeOn == 0) {
				Game.ui.strobeOn = 50 + Math.floor(Math.random() * 50);

				var on = function(){
					if (Game.ui.hasClass(elem, 'negative')) {
						Game.ui.removeClass(elem, 'negative');
					}
					else {
						Game.ui.addClass(elem, 'negative');
					}
				};

				var t = 0, limit = 50 + Math.floor(Math.random() * 50);

				var x = setInterval(function(){
					on();

					if (t > limit) {
						Game.ui.removeClass(elem, 'negative');
						clearInterval(x);
					}
					t++;
				}, 50);
			}
		}
	};

	Game.utils = {
		getSps: function(){
			var stateKeys = Object.keys(Game.state.units);
			if (!stateKeys) return 0;

			var unitValues = stateKeys.map(function(index){
				var value = Game.state.units[index].count;

				return value * Game.units[index].sps;
			});

			if (unitValues.length == 0) return 0;

			var sps = unitValues.reduce(function(a, b){
				return a + b
			});

			var sps = new BigNumber(sps);

			if (Game.state.perkType == "sps") {
				if (Game.state.perkValue.comparedTo(Game.constants.zero) > 0) {
					sps = sps.times(Game.state.perkValue);
				}
			}

			return sps;
		},

		getSpc: function(){
			var stateKeys = Object.keys(Game.state.units);
			if (!stateKeys) return 0;

			var unitValues = stateKeys.map(function(index){
				var value = Game.state.units[index].count;

				return value * Game.units[index].spc;
			});

			if (unitValues.length == 0) return 0;

			var spc = unitValues.reduce(function(a, b){
				return a + b
			});

			var spc = new BigNumber(spc);

			if (Game.state.perkType == "spc") {
				if (Game.state.perkValue.comparedTo(Game.constants.zero) > 0) {
					spc = spc.times(Game.state.perkValue);
				}
			}

			return spc;
		}
	};

	Canvas.classes = {
		char: function(){
			this.name = "char-" + Math.floor(Math.random() * 999999999);
			this.x = 0;
			this.y = 0;
			this.lifetime = 10;
			this.life = 0;

			this.setup = function(){
				this.x = Math.round((Math.random() * window.innerWidth) / 32) * 32;
				this.y = Math.round((Math.random() * window.innerHeight) / 32 - Math.random()) * 32;
				this.lifetime = 10 + Math.floor(Math.random() * 25);
			};

			this.char = function(){
				return Math.random().toString(36).substring(2,3);
			};

			this.katakana = function(){
				return String.fromCharCode(12449 + Math.floor((12539 - 12449) * Math.random()));
			};

			this.draw = function(){
				Canvas.state.ctx.font = "32px monospace";
				Canvas.state.ctx.fillStyle = "#00ff00";

				var newChar = ' ';
				if (Math.random() > 0.8) {
					newChar = this.katakana();
				}
				else {
					newChar = this.char();
				}

				Canvas.state.ctx.fillText(newChar, Math.round(this.x), Math.round(this.y));

				this.move();
				this.live();
			};

			this.move = function(){
				var willMove = Math.random() > 0.25;
				if (willMove) {
					this.y += Math.floor(window.innerHeight / 32);
				}
			};

			this.live = function(){
				this.life += 1;

				if (this.life >= this.lifetime) {
					delete Canvas.state.objects[this.name];
				}
			};

			this.setup();
		}
	};

	Canvas.state = {
		ctx: null,
		objects: {},
		filters: {
			fade: 0.85
		},

		flushTimer: 0,
		flushTimerCycle: 5,

		maxObjects: 100
	};

	Canvas.action = {
		draw: function(){
			for (var i in Canvas.state.objects) {
				Canvas.state.objects[i].draw();
			}
		},

		prune: function(){
			var keys = Object.keys(Canvas.state.objects);

			if (keys.length > Canvas.state.maxObjects) {
				var name = keys[Math.floor(Math.random() * keys.length)];
				delete Canvas.state.objects[name];
			}
		},

		add: function(addClass) {
			Canvas.action.prune();

			var obj = new Canvas.classes[addClass]();
			Canvas.state.objects[obj.name] = obj;
		},

		addFilter: function(name, strength){
			Canvas.state.filters[name] = strength;
		},

		removeFilter: function(name, strength){
			delete Canvas.state.filters[name];
		},

		flush: function(){
			var canv = document.querySelector('#canvas-inner');

			var imgData = Canvas.state.ctx.getImageData(0, 0, canv.width, canv.height);
			
			for (var f in Canvas.state.filters) {
				if (!!Canvas.filters[f]) {
					imgData = Canvas.filters[f](imgData, Canvas.state.filters[f]);
				}
			}

			// Store the current transformation matrix
			Canvas.state.ctx.save();

			// Use the identity matrix while clearing the canvas
			Canvas.state.ctx.setTransform(1, 0, 0, 1, 0, 0);

			if (!Canvas.filters.noClear) {
				Canvas.state.ctx.clearRect(0, 0, canv.width, canv.height);
			}

			// Restore the transform
			Canvas.state.ctx.restore();

			if (!Canvas.filters.noKeep) {
				Canvas.state.ctx.putImageData(imgData, 0, 0, 0, 0, canv.width, canv.height);
			}
		}
	};

	Canvas.events = {
		init: function(){
			var canv = document.createElement('canvas');
			canv.id = "canvas-inner";
			canv.width = window.innerWidth;
			canv.height = window.innerHeight;
			document.querySelector('#canvas').appendChild(canv);

			Canvas.state.ctx = canv.getContext('2d');

			Canvas.events.animate();

			window.onresize = Canvas.events.resize;
		},

		animate: function(){
		    var reqAnimFrame =  window.mozRequestAnimationFrame    ||
		    					window.webkitRequestAnimationFrame ||
		    					window.msRequestAnimationFrame     ||
		    					window.oRequestAnimationFrame;

		    reqAnimFrame(Canvas.events.animate);

		    var ft = Canvas.state.flushTimerCycle;

		    if (Canvas.state.flushTimer % ft == ft - 1) {
		    	Canvas.action.flush();
		    }
		    Canvas.state.flushTimer++;
		    if (Canvas.state.flushTimer >= ft) Canvas.state.flushTimer = 0;

		    Canvas.action.draw();
		},

		resize: function(){
			var canv = document.querySelector('#canvas-inner');
			canv.width = window.innerWidth;
			canv.height = window.innerHeight;
		}
	};

	Canvas.filters = {

		fade: function(imgData, adjustment){
			var d = imgData.data;

			for (var i = 3; i < d.length; i += 4) {
				var alpha = d[i];
				if (alpha > 0) {
					// alpha -= (alpha * 0.01);
					var factor = Math.random();

					if (factor > adjustment) {
						alpha = 0;
					}
					d[i] = alpha;
				}
			}

			return imgData;
		},

		bright: function(imgData, adjustment) {
			var d = imgData.data;

			for (var i = 0; i < d.length; i += 4) {
				d[i] += adjustment;
				d[i+1] += adjustment;
				d[i+2] += adjustment;
			}

			return imgData;
		},

		glitch1: function(imgData, adjustment){
			var c = document.createElement('canvas');
			c.width  = Math.ceil(imgData.width / adjustment);
			c.height = Math.ceil(imgData.height / adjustment);

			var ctx = c.getContext('2d');

			ctx.putImageData(imgData, 0, 0, 0, 0, c.width, c.height);
			ctx.scale(adjustment, adjustment);

			return ctx.getImageData(0, 0, imgData.width, imgData.height);
		},

		glitch2: function(imgData, adjustment){
			var d = imgData.data;

			for (var i = 0; i < adjustment; i++) {
				var offset = Math.floor(Math.random() * imgData.width * imgData.height * 4);
				var length = Math.floor(Math.random() * imgData.width * 50);

				for (var j = offset; j < offset + length; j += 4) {
					d[j + 1] = 255;
					d[j + 3] = 255;
				}
			}

			return imgData;
		}
	};

	Game.stateLoader.load();
	Game.stateLoader.setSave();
	Game.events.init();
	Canvas.events.init();
	Game.action.display(true);

})();
