#version 420

// 1 divided by width and height
uniform vec2 viewport_mul;

// G Buffer
uniform sampler2D normal_buffer;
uniform sampler2D position_buffer;
uniform sampler2D depth_buffer;
uniform samplerCube shadow_map;

// Light uniforms
uniform vec3 light_position;
uniform float light_radius;
uniform vec4 light_color;

uniform vec3 camera_position;
uniform vec2 cam_near_far;
uniform mat4 cam_left_vp_matrix;
uniform mat4 cam_right_vp_matrix;
uniform mat4 cam_front_vp_matrix;
uniform mat4 cam_back_vp_matrix;
uniform mat4 cam_top_vp_matrix;
uniform mat4 cam_bottom_vp_matrix;

in vec2 out_TexCoord;

layout(location = 0) out vec4 diffuse;
layout(location = 1) out vec4 specular;

const float BIAS = 0.005;

float linearize_depth_value(float depth)
{
	float linear_depth = (2 * cam_near_far.x) / (cam_near_far.y + cam_near_far.x - depth * (cam_near_far.y - cam_near_far.x));
	return linear_depth;
}

vec2 get_cube_z(vec3 incident, vec3 position)
{
	float right_dot_incident = dot(incident, vec3(1, 0, 0));
	float front_dot_incident = dot(incident, vec3(0, 0, 1));
	float back_dot_incident = dot(incident, vec3(0, 0, -1));
	float top_dot_incident = dot(incident, vec3(0, 1, 0));
	float bottom_dot_incident = dot(incident, vec3(0, -1, 0));

	// left
	mat4 cam_vp_matrix = cam_left_vp_matrix;
	float lowest_angle = dot(incident, vec3(-1,0,0));

	if (right_dot_incident > lowest_angle)
	{
		cam_vp_matrix = cam_right_vp_matrix;
		lowest_angle = right_dot_incident;
	}
	if (front_dot_incident > lowest_angle)
	{
		cam_vp_matrix = cam_front_vp_matrix;
		lowest_angle = front_dot_incident;
	}
	if (back_dot_incident > lowest_angle)
	{
		cam_vp_matrix = cam_back_vp_matrix;
		lowest_angle = back_dot_incident;
	}
	if (top_dot_incident > lowest_angle)
	{
		cam_vp_matrix = cam_top_vp_matrix;
		lowest_angle = top_dot_incident;
	}
	if (bottom_dot_incident > lowest_angle)
	{
		cam_vp_matrix = cam_bottom_vp_matrix;
		lowest_angle = bottom_dot_incident;
	}

	float z = texture(shadow_map, incident).r;
	vec4 clip = cam_vp_matrix * vec4(position, 1);

	clip /= clip.w;
	clip = (clip + 1) / 2;
	return vec2(linearize_depth_value(z),
				linearize_depth_value(clip.z));
}

void main()
{
	vec2 tex_coord = gl_FragCoord.xy * viewport_mul;

	vec4 normal_and_specular = texture2D(normal_buffer, tex_coord);

	vec3 normal = normalize(normal_and_specular.rgb);
	vec3 position = texture2D(position_buffer, tex_coord).rgb;

	float dist = length ( light_position - position );
	float atten = 1.0 - clamp ( dist / light_radius , 0.0 , 1.0);
	if( atten == 0.0)
		discard;

	vec3 incident = normalize ( light_position - position);

	vec2 z = get_cube_z(-incident, position);

	 if (z.x + BIAS < z.y)
	 	discard;


	vec3 viewDir = normalize ( camera_position - position);
	vec3 halfDir = normalize ( incident + viewDir );
	float lambert = clamp ( dot ( incident , normal ) ,0.0 ,1.0);
	float rFactor = clamp ( dot ( halfDir , normal ) ,0.0 ,1.0);
	
	float sFactor = pow ( rFactor , normal_and_specular.a ); // Shininess in worldnormal.a
	

	diffuse = vec4 ( light_color. xyz * lambert * atten , 1.0);
	specular = vec4 ( light_color. xyz * sFactor * atten *0.33 ,1.0);

}

