#!/usr/bin/python # This is server.py file

#######################################
#                                     #
#  Written by Fredrik Strupe for the  #
#  TG14 AI Programming competition.   #
#  (19.04.2014)                       #
#                                     #
#######################################

import socket
from collections import deque
import heapq
from sys import argv
import re

class Opponent:
	def __init__(self, idNum, tupleCoord):
		self.id = int(idNum)
		self.coord = (int(tupleCoord[0]), int(tupleCoord[1]))

class Bomb:
	def __init__(self, tupleCoord, detonationTime):
		self.coord = (int(tupleCoord[0]), int(tupleCoord[1]))
		self.ticksLeft = int(detonationTime)
	
class GameData:
	# Data members:
	# - opponents (list containing each opponent's id and coordinates
	#			   in the form of instances of the class Opponent)
	# - bombs (list containing the coord and ticks left for each bomb
	#		   in the form of instances of the class Bomb)
	# - ownCoord (tuple containing the coordinates of the bot)
	# - map (2d list made of different ascii tiles)
	#	IMPORTANT: The list is accessed with the coordinates [y][x] and NOT [x][y]
	# - dangerMap (2d list made of values from -1 to 5. -1 means obstacle, and
	#   0 means walkable tile. 1-5 means those tiles are in a bomb's blast radius
	# - dead (boolean that says if the player is dead or the round is over)

	def getRawData(self, sock):
		datablock = sock.recv(4096)
		return datablock.split('\n')

	def updateGameData(self, sock):
		self.dead = False
		rawData = self.getRawData(sock)
		
		# If rawData is completely empty, either a) the bot has been kicked or
		# b) something went terribly wrong. Either way, just quit.
		if rawData == ['']:
			print 'QUITTING'
			quit()

		# Continue trying to receive data if non-complete data is received.
		# This is to avoid crashing (to some degree) when playing on maps with
		# a lot of players (20+), as the server seems to have issues sending
		# too many packages at once (this may have been fixed in a newer version).
		while 'PLAYERS' not in rawData and 'ENDMAP' not in rawData:
			if 'DEAD' in rawData or 'ENDOFROUND' in rawData:
				self.dead = True
				return
			rawData = self.getRawData(sock)

		# Get the coordinates of the opponents
		self.opponents = []
		for x in xrange(rawData.index('PLAYERS')+1, rawData.index('ENDPLAYERS')):
			data = rawData[x].split(' ')
			self.opponents.append(Opponent(data[0], tuple(data[1].split(','))))
		
		# Get the coordinates and time/ticks to detonation for the bombs
		self.bombs = []
		for x in xrange(rawData.index('BOMBS')+1, rawData.index('ENDBOMBS')):
			data = rawData[x].split(' ')
			self.bombs.append(Bomb(tuple(data[0].split(',')), data[1]))
		
		# Get the bot's own coordinates
		xCoord = yCoord = 0
		for line in rawData:
			if 'X ' in line:  xCoord = line[len('X '):]
			elif 'Y ' in line: yCoord = line[len('Y '):]
		self.ownCoord = (int(xCoord), int(yCoord))
		
		# Extract the map
		self.map = []
		for x in xrange(rawData.index('MAP')+1, rawData.index('ENDMAP')):
			self.map.append(rawData[x])
		# Split each line in the map into separate characters (to get a 2d list)
		for x in xrange(len(self.map)):
			lineList = []
			for char in self.map[x]:
				lineList.append(char)
			self.map[x] = lineList
			
		self.dangerMap = self.generateDangerMap(self.map, self.bombs)

	def generateDangerMap(self, map, bombs):
		# Creates a map with dimensions equal to self.map where all tiles are set to 0
		dMap = [[0 for x in xrange(len(map[0]))] for y in xrange(len(map))]
		
		# Set all non-walkable tiles to -1
		for y in xrange(len(map)):
			for x in xrange(len(map[y])):
				if map[y][x] is '+' or map[y][x] is '#':
					dMap[y][x] = -1

		# Place the bomb(s) on the map with the value
		# 6 - amount of ticks left. This means the higher the number,
		# the higher the danger. (The bombs start with 5 ticks)
		for bomb in bombs:
			dMap[bomb.coord[1]][bomb.coord[0]] = 6 - bomb.ticksLeft
		
		# Calculate the tiles the bomb(s) will affect, and give those
		# tiles the same value as the bomb(s). The affected tiles will be the
		# horizontal and vetical tiles exactly one or two tiles away. Remember
		# that the blow doesn't go through more than one block, and therefore
		# blocks shouldn't be replaced with danger values.
		for bomb in bombs:
			x = bomb.coord[0]
			y = bomb.coord[1]

			# Limit the range if a rocks is right next to the bomb, to avoid
			# explosion paths going through rocks
			yRange = [-2, 3]
			if 0 <= y-1 < len(dMap):
				if game.map[y-1][x] == '#': yRange[0] = 0
			if 0 <= y+1 < len(dMap):
				if game.map[y+1][x] == '#': yRange[1] = 1
			
			# Add the explosion path
			for i in xrange(yRange[0], yRange[1]):
				if 0 <= y+i < len(dMap):
					if dMap[y+i][x] >= 0 and dMap[y][x] > dMap[y+i][x]:
						dMap[y+i][x] = dMap[y][x]
						
			xRange = [-2, 3]
			if 0 <= x-1 < len(dMap[y]):
				if game.map[y][x-1] == '#': xRange[0] = 0
			if 0 <= x+1 < len(dMap[y]):
				if game.map[y][x+1] == '#': xRange[1] = 1
				
			for i in xrange(xRange[0], xRange[1]):
				if 0 <= x+i < len(dMap[y]):
					if dMap[y][x+i] >= 0 and dMap[y][x] > dMap[y][x+i]:
						dMap[y][x+i] = dMap[y][x]
		return dMap

def astar(init, goal, nexts,
		  distance=lambda path: len(path),
		  heuristic=lambda pos: 0):
	queue = []
	initScore = distance([init]) + heuristic(init)
	checked = {init: initScore}
	heapq.heappush(queue, (initScore, [init]))
	while len(queue) > 0:
		score, path = heapq.heappop(queue)
		last = path[-1]
		if last == goal: 
			return path
		for pos in nexts(last):
			newpath = path + [pos]
			posScore = distance(newpath) + heuristic(pos)
			if pos in checked and checked[pos] <= posScore: 
				continue
			checked[pos] = posScore
			heapq.heappush(queue, (posScore, newpath))
	return None

def astarPath(map, ignoreRocks = False):
	for x in xrange(len(map)):
		map[x] = ''.join(map[x])

	def findCh(ch):
		for i, l in enumerate(map):
			for j, c in enumerate(l):
				if c == ch: return (i, j)
		return None

	init = findCh('S')
	goal = findCh('G')
	obstacles = []
	if ignoreRocks:
		obstacles = ['+', 'P'] 
	else:
		obstacles = ['+', '#', 'B', 'P']

	def nexts(pos):
		canYield = [True, True, True, True]
		for obstacle in obstacles:
			if map[pos[0] - 1][pos[1]] == obstacle: canYield[0] = False 
			if map[pos[0] + 1][pos[1]] == obstacle: canYield[1] = False
			if map[pos[0]][pos[1] - 1] == obstacle: canYield[2] = False
			if map[pos[0]][pos[1] + 1] == obstacle: canYield[3] = False
		if canYield[0]: yield (pos[0] - 1, pos[1])
		if canYield[1]: yield (pos[0] + 1, pos[1])
		if canYield[2]: yield (pos[0], pos[1] - 1)
		if canYield[3]: yield (pos[0], pos[1] + 1)
	def heuristic(pos):
		return ((pos[0] - goal[0]) ** 2 + (pos[1] - goal[1]) ** 2) ** 0.5
	def distance(path):
		return len(path)
	
	try:
		path = astar(init, goal, nexts, distance, heuristic)
	except TypeError:
		return []
	return path
	
def getSafeSpot(game, canEscapeCall = False):
	# Begin with finding all the safe spots in a perimeter of 4 tiles
	safeSpots = []
	dangerSpots = []
	opponentCoords = [opponent.coord for opponent in game.opponents]
	bombCoords = [bomb.coord for bomb in game.bombs]
	for y in xrange(game.ownCoord[1] - 4, game.ownCoord[1] + 4):
		for x in xrange(game.ownCoord[0] - 4, game.ownCoord[0] + 4):
			if 0 <= y < len(game.map) and 0 <= x < len(game.map[0]):
				if (x, y) not in opponentCoords and (x, y) not in bombCoords:
					if game.dangerMap[y][x] == 0:
						safeSpots.append((x, y))
					elif game.dangerMap[y][x] >= 1:
						dangerSpots.append((x, y))

	# Check the amount of moves that are required for getting to each safe spot
	leastMoves = 1000
	adjacentTiles = 0
	bestSafeSpot = (-1, -1)
	for spot in safeSpots:
		path = getPath(game, spot)
		moves = len(path)

		# Check if the bot will blow up while walking this path by already placed bombs
		if moves > 0:
			for x in xrange(moves):
				if game.dangerMap[path[x][0]][path[x][1]] > 0 and\
					game.dangerMap[path[x][0]][path[x][1]]+ x == 5:
					moves = 0 # Cancel the path, as it is dangerous

		# If two safe spots are equally close, choose the one with the most adjacent tiles (to mitigate trapping)
		if moves == leastMoves:
			currAdjTiles = 0
			if spot[1]+1 < len(game.map):
				if game.map[spot[1]+1][spot[0]] == '.': currAdjTiles += 1
			if spot[1]-1 >= 0:
				if game.map[spot[1]-1][spot[0]] == '.': currAdjTiles += 1
			if spot[0]+1 < len(game.map[0]):
				if game.map[spot[1]][spot[0]+1] == '.': currAdjTiles += 1
			if spot[0]-1 >= 0:
				if game.map[spot[1]][spot[0]-1] == '.': currAdjTiles += 1
			if currAdjTiles > adjacentTiles:
				adjacentTiles = currAdjTiles
				bestSafeSpot = spot
		if 0 < moves < leastMoves:
			leastMoves = moves
			bestSafeSpot = spot

	# If the function is called from the canEscape function, the calculations need to stop
	# here. The rest won't make sense if the bot is checking if it should drop a bomb or not.
	if canEscapeCall is True:
		return (game.ownCoord if bestSafeSpot == (-1, -1) else bestSafeSpot)
		
	# If no safe spots are found, find the closest danger spot that can be reached.
	# While standing there, always check if there are any new safe spots/less dangerous spots,
	# and move there. Take bomb ticks into consideration when chosing a danger spot. Do not
	# go to a danger spot with a higher danger value than the one the bot's currently on.
	bestDangerSpot = (-1, -1) # Best means safest
	if bestSafeSpot == (-1, -1):
		mostTicks = 0 # Ticks left when the tile has been reached. The higher the better
		for spot in dangerSpots:
			moves = len(getPath(game, spot))
			ticks = 6 - (game.dangerMap[spot[1]][spot[0]] + moves)
			if ticks < 6 and ticks > mostTicks:
				mostTicks = ticks
				bestDangerSpot = spot
		dangerOwnSpot = game.dangerMap[game.ownCoord[1]][game.ownCoord[0]]
		if dangerOwnSpot <= game.dangerMap[bestDangerSpot[1]][bestDangerSpot[0]]:
			bestDangerSpot = game.ownCoord

	bestSpot = ()
	if bestSafeSpot != (-1, -1):
		bestSpot = bestSafeSpot
	else:
		if bestDangerSpot == (-1, -1):
			bestSpot = game.ownCoord
		else:
			bestSpot = bestDangerSpot
			
	return bestSpot

def getPath(game, coordinates, ignoreRocks = False):	
	map = [[0 for x in xrange(len(game.map[0]))] for y in xrange(len(game.map))]
	for y in xrange(len(game.map)):
		for x in xrange(len(game.map[0])):
			map[y][x] = str(game.map[y][x])

	for bomb in game.bombs:
		map[bomb.coord[1]][bomb.coord[0]] = 'B'
	for opponent in game.opponents:
		map[opponent.coord[1]][opponent.coord[0]] = 'P'
	
	map[game.ownCoord[1]][game.ownCoord[0]] = 'S'
	map[int(coordinates[1])][int(coordinates[0])] = 'G'
		
	# Returns a list with tuple-coordinates (in the form (y, x))
	path = astarPath(map, ignoreRocks)
	if path == None:
		path = []

	return path

def getPathInstructions(path):
	pathInstructions = []
	for x in xrange(1, len(path)):
		if path[x][0] < path[x-1][0]:
			pathInstructions.append('UP\n')
		elif path[x][0] > path[x-1][0]:
			pathInstructions.append('DOWN\n')
		elif path[x][1] < path[x-1][1]:
			pathInstructions.append('LEFT\n')
		elif path[x][1] > path[x-1][1]:
			pathInstructions.append('RIGHT\n')
			
	return pathInstructions
	
def getRockBlowupInstructions(game):
	# This function calculates the shortest route to the closest player
	# (ignoring rocks), and then - taking rocks into consideration - find
	# out what rocks need to be blown up to open that path.
	
	# Find the opponent closest to our own bot
	shortestLen = 1000
	targetPath = []
	for opponent in game.opponents:
		path = getPath(game, opponent.coord, True)
		if 0 < len(path) < shortestLen:
			shortestLen = len(path)
			targetPath = list(path)
	
	if len(targetPath) == 0:
		return []
		
	# Find the tile closest to the opponent that can be reached
	# (without blowing up rocks)
	currPath = []
	currPath.append(targetPath[0])
	for coord in targetPath[1:]:
		# If the tile is reachable the length will be > 0
		if len(getPath(game, (coord[1], coord[0]))) > 0:
			currPath.append(coord)
		else:
			break
	
	obstacles = ['+', '#', 'B', 'P']
	for x in xrange(len(currPath)):
		if game.map[currPath[x][0]][currPath[x][1]] in obstacles:
			currPath = currPath[:x]
			break
	
	instructions = getPathInstructions(currPath)
	instructions.append('BOMB\n')
	return instructions

def getOpponentAttackInstructions(game):
	# This function finds the opponent closest to our bot that is also
	# reachable without blowing up rocks. If such an opponent exists, this function
	# will find the path to the opponent, and then make the bot drop a bomb.
	
	# First check if there are any reachable opponents
	# (if there's not, then the getRockBlowupInstructions function
	# needs to be used).
	oppCoord = ()
	smallestLen = 1000
	for opponent in game.opponents:
		length = len(getPath(game, opponent.coord))
		if 0 < length < smallestLen:
			oppCoord = opponent.coord
			smallestLen = length

	if len(oppCoord) == 0:
		return []
	
	# The follow approach is the lazy temporary approach
	instructions = getPathInstructions(getPath(game, oppCoord))
	if instructions == None:
		return []
		
	# Remove the last item, as this will be the location of the opponent itself
	instructions.pop() 
	instructions.append('BOMB\n')
	return instructions
	
def canEscape(game):
	bombs = list(game.bombs)

	# Add a hypothetical bomb at the bot's own coordinates
	bombs.append(Bomb(game.ownCoord, 5))
	
	# Generate the hypothetical danger map with the hypothetical bomb
	game.dangerMap = game.generateDangerMap(game.map, bombs)

	safeSpot = getSafeSpot(game, True)
	
	# Remember to revert back to the real danger map
	game.dangerMap = game.generateDangerMap(game.map, game.bombs)

	# If the safe spot is equal to the bot's own coordinates, it means there
	# are no available safe spot
	return safeSpot != game.ownCoord
	
sock = socket.socket() # Create a socket object
host = socket.gethostname() # Get local machine name
port = 54321 # Reserve a port for your service.

# Will be true if an IP-address is specified as a command line argument
if len(argv) > 1:
	# Check if the specified IP-address is valid.
	if re.match('^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}', argv[1]):
		hostAndPort = argv[1].split(':')
		host = hostAndPort[0]
		if ':' in argv[1]:
			port = int(hostAndPort[1])
	else:
		print 'BAD IP SPECIFIED'
		quit()

sock.connect((host, port))
#sock.connect(("INSERT IP HERE", port))
sock.send(b'NAME bob-kaare\n')

game = GameData()
game.updateGameData(sock)

instructions = deque([])

while True:
	game.updateGameData(sock)
	if game.dead == False:
		# If in a dangerous spot: move away
		if game.dangerMap[game.ownCoord[1]][game.ownCoord[0]] >= 1:
			instructions = deque(getPathInstructions(getPath(game, getSafeSpot(game))))
		else:
			oppInstr = getOpponentAttackInstructions(game)
			# If an opponent is in reach: go to it and bomb it
			if len(oppInstr) > 0:
				instructions = deque(oppInstr)
			# If no opponents is in reach: blow up rocks to get to one
			else:
				instructions = deque(getRockBlowupInstructions(game))
		
		if len(instructions) > 0:
			movement = {'UP\n' : (0, -1), 'DOWN\n' : (0, 1), 'LEFT\n' : (-1, 0),\
						'RIGHT\n' : (1, 0), 'BOMB\n' : (0, 0)}
			instruction = instructions[0]
			# If the current spot is safe, and the next spot is dangerous: do nothing
			# (Do not go from safe to dangerous spots)
			if game.dangerMap[game.ownCoord[1]][game.ownCoord[0]] == 0 and\
				game.dangerMap[game.ownCoord[1] + movement[instruction][1]]\
							 [game.ownCoord[0] + movement[instruction][0]] > 0:
				# Do nothing
				pass
			else:
				if 'BOMB' in instruction:
					# Always check if there will be an escape route before dropping a bomb
					if canEscape(game):
						sock.send(instructions.popleft())
				else:
					sock.send(instructions.popleft())
	else:
		instructions.clear()
		
sock.close # Close the socket when done