#ifndef ODINAI_TIMER_H_
#define ODINAI_TIMER_H_

#ifdef _WIN32
#include <Windows.h>
#else
#include <ctime>
#include <chrono>
#include <ratio>
using namespace std::chrono;
#endif

#include "MathUtil.h"

/**
 * Timer for time dependent code.
 */
namespace OdinAI
{
	

	class Timer
	{
	public:
		Timer(){ Reset(); }

		/**
		 * Initialize timer. NOTE: This function should only be called once!
		 */
		static bool Init()
		{
#ifdef _WIN32
			if (!QueryPerformanceFrequency(&m_ticksPerSecond))
			{
				return false;
			}

			m_realFreq = 1.0 / (double)m_ticksPerSecond.QuadPart;
#endif
			GlobalReset();

			return true;
		}

		/**
		 * Resets the global clock.
		 */
		static void GlobalReset()
		{
#ifdef _WIN32
			QueryPerformanceCounter(&m_gStartTime);
			m_gElapsedTime = m_gStartTime;
#else
			m_gStartTime = high_resolution_clock::now();
			m_gElapsedTime = m_gStartTime;
#endif
		}

		/**
		 * Elapsed time from last call to this function, or other functions which also use elapsed time.
		 * @return Real number in seconds.
		 */
		template<typename T>
		static T GetGlobalElapsedTime()
		{
#ifdef _WIN32
			LARGE_INTEGER currentTime;
			QueryPerformanceCounter(&currentTime);

			T sec = (T)(currentTime.QuadPart - m_gElapsedTime.QuadPart) * m_realFreq;
			m_gElapsedTime = currentTime;
			return sec;
#else
			TimePoint currentTime = high_resolution_clock::now();
			duration<T> time_span = duration_cast< duration<T> >(currentTime - m_gElapsedTime);
			m_gElapsedTime = currentTime;
			return time_span.count();
#endif
		}

		/**
		 * Elapsed ticks from last call to this or other functions, which also use elapsed time..
		 */
		template<typename T>
		static T GetGlobalElapsedMillisec()
		{
#ifdef _WIN32
			LARGE_INTEGER currentTime;
			QueryPerformanceCounter(&currentTime);
			double milliSec = (double)(currentTime.QuadPart - m_gElapsedTime.QuadPart) / (double)m_ticksPerSecond.QuadPart;

			m_gElapsedTime = currentTime;

			return (T)(milliSec * 1000);
#else
			TimePoint currentTime = high_resolution_clock::now();
			milliseconds time_span = duration_cast< milliseconds >(currentTime - m_gElapsedTime);
			m_gElapsedTime = currentTime;
			return (T)time_span.count();
#endif
		}

		/**
		 * Time since last reset of global clock.
		 */
		template<typename T>
		static T GetGlobalMillisec()
		{
#ifdef _WIN32
			LARGE_INTEGER currentTime;
			QueryPerformanceCounter(&currentTime);
			double milliSec = (double)(currentTime.QuadPart - m_gStartTime.QuadPart) / (double)m_ticksPerSecond.QuadPart;

			return (T)(milliSec * 1000);
#else
			TimePoint currentTime = high_resolution_clock::now();
			milliseconds time_span = duration_cast< milliseconds >(currentTime - m_gStartTime);

			return (T)time_span.count();
#endif
		}

		/**
		 * Reset this instance of timer.
		 */
		void Reset()
		{
#ifdef _WIN32
			QueryPerformanceCounter(&m_startTime);
			m_elapsedTime = m_startTime;
#else
			m_startTime = high_resolution_clock::now();
			m_elapsedTime = m_startTime;
#endif
		}

		/**
		 * Elapsed time from last call to this function, or other functions which also use elapsed time.
		 * @return Real number in seconds.
		 */
		template<typename T>
		T GetElapsedTime()
		{
#ifdef _WIN32
			LARGE_INTEGER currentTime;
			QueryPerformanceCounter(&currentTime);

			T sec = (T)(currentTime.QuadPart - m_elapsedTime.QuadPart) * m_realFreq;
			m_elapsedTime = currentTime;
			return sec;
#else
			TimePoint currentTime = high_resolution_clock::now();
			duration<T> time_span = duration_cast< duration<T> >(currentTime - m_elapsedTime);
			m_elapsedTime = currentTime;
			return time_span.count();
#endif
		}

		/**
		 * Time since last call to this or other functions, which also use elapsed time.
		 */
		template<typename T>
		T GetElapsedMillisec()
		{
#ifdef _WIN32
			LARGE_INTEGER currentTime;
			QueryPerformanceCounter(&currentTime);
			double milliSec = (double)(currentTime.QuadPart - m_elapsedTime.QuadPart) / (double)m_ticksPerSecond.QuadPart;

			m_elapsedTime = currentTime;

			return (T)(milliSec * 1000);
#else
			TimePoint currentTime = high_resolution_clock::now();
			milliseconds time_span = duration_cast<milliseconds>(currentTime - m_elapsedTime);
			m_elapsedTime = currentTime;
			return (T)time_span.count();
#endif
		}

		/**
		 * Time since last reset of this instance.
		 */
		template<typename T>
		T GetMillisec()
		{
#ifdef _WIN32
			LARGE_INTEGER currentTime;
			QueryPerformanceCounter(&currentTime);
			double milliSec = (double)(currentTime.QuadPart - m_startTime.QuadPart) / (double)m_ticksPerSecond.QuadPart;

			return (T)(milliSec * 1000);
#else
			TimePoint currentTime = high_resolution_clock::now();
			milliseconds time_span = duration_cast<milliseconds>(currentTime - m_startTime);
			return (T)time_span.count();
#endif
		}

		/**
		 * Time since last reset of this instance.
		 */
		template<typename T>
		T GetTime()
		{
#ifdef _WIN32
			LARGE_INTEGER currentTime;
			QueryPerformanceCounter(&currentTime);
			T secs = (T)(currentTime.QuadPart - m_startTime.QuadPart) / (T)m_ticksPerSecond.QuadPart;

			return secs;
#else
			TimePoint currentTime = high_resolution_clock::now();
			duration<T> time_span = duration_cast< duration<T> >(currentTime - m_startTime);
			return time_span.count();
#endif
		}

#ifdef _WIN32
		static LARGE_INTEGER m_gStartTime;//!< When the global timer was last reset.
		static LARGE_INTEGER m_ticksPerSecond;//!< Number of ticks per second, in other words the frequency.
		static LARGE_INTEGER m_gElapsedTime;//!< Time elapsed between each call.
		static double m_realFreq;//!< Floating/Double point frequency.

		LARGE_INTEGER m_startTime;//!< When this instance was last reset.
		LARGE_INTEGER m_elapsedTime;//!< Time elapsed between each call.
#else
		typedef high_resolution_clock::time_point TimePoint;
		static TimePoint m_gStartTime;
		static TimePoint m_gElapsedTime;

		TimePoint m_startTime;
		TimePoint m_elapsedTime;
#endif

	};
}
#endif
