#version 330 core

// Input data
layout(location = 0) in vec3 vertexPosition_modelspace;
layout(location = 1) in vec2 vertexUV;
layout(location = 2) in vec3 vertexNormal_modelspace;

// Output data
out vec2 texCoord;
out vec3 Position_worldspace;
out vec3 Normal;
out vec3 EyeDirection_cameraspace;
out vec3 LightDirection_cameraspace;

vec3 tangent; 
vec3 binormal; 

// Values that stay constant for the whole draw.
uniform mat4 MVP;
uniform mat4 V;
uniform mat4 M;

uniform vec3 LightPosition_worldspace;
uniform float time;

uniform float frequency;
uniform float amplitude;

vec3 computeNormal( vec3 pos, 
                    vec3 tangent, 
                    vec3 binormal, 
                    float phase, 
                    float freq )
{
	mat3 J;
	
	float dist = sqrt(pos.x*pos.x + pos.z*pos.z);
	float jac_coef = cos(freq*dist + phase) / (dist+0.00001);
	
	// A matrix is an array of column vectors so J[2] is 
	// the third column of J.
	
	J[0][0] = 1.0;
	J[0][1] = jac_coef * pos.x;
	J[0][2] = 0.0;
	
	J[1][0] = 0.0;
	J[1][1] = 1.0;
	J[1][2] = 0.0;

	J[2][0] = 0.0;
	J[2][1] = jac_coef * pos.z;
	J[2][2] = 1.0;
	
	vec3 u = J * tangent;
	vec3 v = J * binormal;
	
	vec3 n = cross(v, u);
	return normalize(n);
}

vec4 displaceVertexFunc( vec4 pos, float phase, float frequency ) 
{
	vec4 new_pos;
	
	new_pos.x = pos.x;
	new_pos.z = pos.z;
	new_pos.w = pos.w;
	
	float expY = sin(pos.x + pos.z + time) * 0.1;

	float dist = sqrt(pos.x*pos.x + pos.z*pos.z);
	new_pos.y = expY + pos.y + amplitude * sin( frequency * dist + phase );
	
	return new_pos;
}

void main(void)
{
	vec4 displacedVertex;
	vec3 displacedNormal;
	texCoord = vertexUV;

	vec4 displacedPosition = displaceVertexFunc(vec4(vertexPosition_modelspace,1), time*2.0, frequency );

	gl_Position = MVP * displacedPosition;

	Normal = vec4((transpose(inverse(V * M))) *  vec4(vertexNormal_modelspace,0)).xyz;

	vec3 c1 = cross(Normal, vec3(0.0, 0.0, 1.0)); 
	vec3 c2 = cross(Normal, vec3(0.0, 1.0, 0.0)); 

	if(length(c1)>length(c2))
	{
		tangent = c1;	
	}
	else
	{
		tangent = c2;	
	}
	
	tangent = normalize(tangent);
	
	binormal = cross(Normal, tangent); 
	binormal = normalize(binormal);
	
	displacedNormal = computeNormal( displacedPosition.xyz, 
	                                 tangent.xyz, 
	                                 binormal, 
	                                 time, 
	                                 frequency );

   	Normal = vec4((transpose(inverse(V * M))) *  vec4(vertexNormal_modelspace,0)).xyz * displacedNormal;	

	Position_worldspace = (M * displacedPosition).xyz;
	vec3 vertexPosition_cameraspace = ( V * M * displacedPosition).xyz;
	EyeDirection_cameraspace = vec3(0,0,0) - vertexPosition_cameraspace;
	vec3 LightPosition_cameraspace = ( V * vec4(LightPosition_worldspace,1)).xyz;

	LightDirection_cameraspace = LightPosition_cameraspace + EyeDirection_cameraspace;

}