/**
 * A very simple basic modern OpenGL application.
 * There is no error handling code in here. We just assume that we have
 * OpenGL and everything works.
 */

// Settings
#define WINDOW_TITLE "Barfenfelden"
#define WINDOW_WIDTH 800
#define WINDOW_HEIGHT 800
// #define FULLSCREEN

// Include files with neat things.
#include "glhelpers.h"
#include "Vector.h"
#include "input.h"

// Globally shared things. Not exactly good style. Don't do it if what you are
// writing is actually important.

// Shaders
GLuint vertexBuffer;
GLuint elementBuffer;
GLuint shaderProgram;

// These are variables passed into shaders
GLuint vertexPosition;
GLuint vertexTexcoords;
GLuint vertexNormal;

GLuint worldviewMatrix;
GLuint normalviewMatrix;
GLuint polyviewMatrix;
GLuint projectionMatrix;
GLuint cuberotMatrix;

GLuint drawDistfield;
GLuint shaderTime;

// Textures
GLuint defaultTextureData;
GLuint defaultTexture;

// Status variable
float angle;
float cubeangle;
float posx;
float posy;
float posz;
float jdir_x;
float jdir_y;
float jdir_z;
float shadertime;
float y_momentum = 0;
int grounded = 1;

Vector pv;
Vector dv;
Vector sv;
Matrix vm;

// Forward declaration, blah blah
void draw();
void update();

// Vertices, for use later
typedef struct vertex_t {
	Vector p;
	float w;
	Vector n;
	float s, t;
} vertex;

// These are vertices and normals for a cube.
// They're interleaved [array-of-struct], which isn't ideal for processing
// on the CPU, but nice to read.
vertex vertices[] = {
	{ {  1.0,  1.0,  1.0 }, 1.0, {  0.0,  0.0,  1.0 }, 0.0, 0.0 },
	{ { -1.0,  1.0,  1.0 }, 1.0, {  0.0,  0.0,  1.0 }, 1.0, 0.0 },
	{ { -1.0, -1.0,  1.0 }, 1.0, {  0.0,  0.0,  1.0 }, 1.0, 1.0 },
	{ {  1.0, -1.0,  1.0 }, 1.0, {  0.0,  0.0,  1.0 }, 0.0, 1.0 },
	{ {  1.0,  1.0,  1.0 }, 1.0, {  1.0,  0.0,  0.0 }, 0.0, 0.0 },
	{ {  1.0, -1.0,  1.0 }, 1.0, {  1.0,  0.0,  0.0 }, 1.0, 0.0 },
	{ {  1.0, -1.0, -1.0 }, 1.0, {  1.0,  0.0,  0.0 }, 1.0, 1.0 },
	{ {  1.0,  1.0, -1.0 }, 1.0, {  1.0,  0.0,  0.0 }, 0.0, 1.0 },
	{ {  1.0,  1.0,  1.0 }, 1.0, {  0.0,  1.0,  0.0 }, 0.0, 0.0 },
	{ {  1.0,  1.0, -1.0 }, 1.0, {  0.0,  1.0,  0.0 }, 1.0, 0.0 },
	{ { -1.0,  1.0, -1.0 }, 1.0, {  0.0,  1.0,  0.0 }, 1.0, 1.0 },
	{ { -1.0,  1.0,  1.0 }, 1.0, {  0.0,  1.0,  0.0 }, 0.0, 1.0 },
	{ { -1.0,  1.0,  1.0 }, 1.0, { -1.0,  0.0,  0.0 }, 0.0, 0.0 },
	{ { -1.0,  1.0, -1.0 }, 1.0, { -1.0,  0.0,  0.0 }, 1.0, 0.0 },
	{ { -1.0, -1.0, -1.0 }, 1.0, { -1.0,  0.0,  0.0 }, 1.0, 1.0 },
	{ { -1.0, -1.0,  1.0 }, 1.0, { -1.0,  0.0,  0.0 }, 0.0, 1.0 },
	{ { -1.0, -1.0, -1.0 }, 1.0, {  0.0, -1.0,  0.0 }, 0.0, 0.0 },
	{ {  1.0, -1.0, -1.0 }, 1.0, {  0.0, -1.0,  0.0 }, 1.0, 0.0 },
	{ {  1.0, -1.0,  1.0 }, 1.0, {  0.0, -1.0,  0.0 }, 1.0, 1.0 },
	{ { -1.0, -1.0,  1.0 }, 1.0, {  0.0, -1.0,  0.0 }, 0.0, 1.0 },
	{ {  1.0, -1.0, -1.0 }, 1.0, {  0.0,  0.0, -1.0 }, 0.0, 0.0 },
	{ { -1.0, -1.0, -1.0 }, 1.0, {  0.0,  0.0, -1.0 }, 1.0, 0.0 },
	{ { -1.0,  1.0, -1.0 }, 1.0, {  0.0,  0.0, -1.0 }, 1.0, 1.0 },
	{ {  1.0,  1.0, -1.0 }, 1.0, {  0.0,  0.0, -1.0 }, 0.0, 1.0 }
};


float pn(float x, float y, float z) {
	int ix = (int)(floor(x));
	int iy = (int)(floor(y));
	int iz = (int)(floor(z));

	float ax = (float)(ix) * 1.0 + (float)(iy) * 57.0 + (float)(iz) * 21.0;
	float ay = ax + 57.0;
	float az = ax + 21.0;
	float aw = ax + 78.0;

	float fx = cos((x-ix)*3.14)*(-0.5) + 0.5;
	float fy = cos((y-iy)*3.14)*(-0.5) + 0.5;
	float fz = cos((z-iz)*3.14)*(-0.5) + 0.5;

	float a1x = sin(cos(ax)*ax);
	float a1y = sin(cos(ay)*ay);
	float a1z = sin(cos(az)*az);
	float a1w = sin(cos(aw)*aw);

	float a2x = sin(cos(1.0+ax)*(1.0+ax));
	float a2y = sin(cos(1.0+ay)*(1.0+ay));
	float a2z = sin(cos(1.0+az)*(1.0+az));
	float a2w = sin(cos(1.0+aw)*(1.0+aw));

	ax = (1.0 - fx) * a1x + fx * a2x;
	ay = (1.0 - fx) * a1y + fx * a2y;
	az = (1.0 - fx) * a1z + fx * a2z;
	aw = (1.0 - fx) * a1w + fx * a2w;

	ax = ay * fy + ax * (1.0 - fy);
	ay = aw * fy + az * (1.0 - fy);

	return (1.0-fz) * ax + ay * fz;
}

float field( float x, float y, float z ) {
	float ball = sqrt(x*x+y*y+(z+200.0)*(z+200.0))-7.0;

	float alpha = z / 50.0;
	float tx = x;
	float ty = y;
	x = tx * cos(alpha) - ty * sin(alpha);
	y = tx * sin(alpha) + ty * cos(alpha);

	float hill = pow(sin(z/20.0+shadertime/2.0),8) * 2.0;
	hill = hill < 0 ? 0 : hill;
	hill = 0;
	
	float plane = -y + 30.0 - hill;
	float field = plane + pn(x/3.0, z/3.0, 0) * 0.1;

	y = -y;
	float plane2 = -y + 30.0 - hill;
	float field2 = plane2 + pn(x/3.0, z/3.0, 0) * 0.1;

	field = field2 > field ? field : field2;
	field = ball > field ? field : ball;
	
	return( field );
}

Vector fieldd1( float x, float y, float z ) {
	float delta = 0.01;
	float nx1 = field( x - delta, y, z );
	float ny1 = field( x, y - delta, z );
	float nz1 = field( x, y, z - delta );
	float nx2 = field( x + delta, y, z );
	float ny2 = field( x, y + delta, z );
	float nz2 = field( x, y, z + delta );
	Vector n = MakeVector( nx1-nx2, ny1-ny2, nz1-nz2 );
	return n;
}

// Make a window for doing OpenGL
void makeWindow( int argc, char** argv ) {
	glutInit( &argc, argv );
	glutInitDisplayMode( GLUT_RGBA | GLUT_DOUBLE | GLUT_DEPTH | GLUT_MULTISAMPLE );
	
	// Open window / full screen
	#ifdef FULLSCREEN
		char modeString[255] = "";
		sprintf( modeString, "%dx%d:24", WINDOW_WIDTH, WINDOW_HEIGHT );
		glutGameModeString( modeString );
		glutEnterGameMode();
	#else
		glutInitWindowSize( WINDOW_WIDTH, WINDOW_HEIGHT );
		glutCreateWindow( WINDOW_TITLE );
	#endif

	// Use GLEW
	#ifndef __APPLE__
	glewInit();
	#endif
	
	// Set up OpenGL features
	glEnable( GL_DEPTH_TEST );
	glClearDepth( 1.0f );
	glDepthFunc( GL_LEQUAL );

	glClearColor( 0.0f, 0.0f, 0.0f, 1.0f );

	glEnable( GL_CULL_FACE );
	glCullFace( GL_BACK );

	// Multisampling.
	glEnable(GL_MULTISAMPLE);

	int buf;
	glGetIntegerv (GL_SAMPLE_BUFFERS_ARB, &buf);
	printf ("number of sample buffers is %d\n", buf);
	glGetIntegerv (GL_SAMPLES_ARB, &buf);
	printf ("number of samples is %d\n", buf);
}

// Functions that glut calls for us
void setupGlutCallbacks() {
	glutDisplayFunc( draw );
	glutTimerFunc( 15, update, 0 );
	setupInput();
}

// Initialize buffers, textures, etc.
void initObjects() {
	// Make one vertex buffer object with vertex data.
	vertexBuffer = makeBO(
		GL_ARRAY_BUFFER,
		vertices,
		sizeof( vertices ),
		GL_STATIC_DRAW
	);

	dv = MakeVector( 0, 0, 1 );
	sv = VectorCross( dv, fieldd1( posx, posy, posz ) );
	
	// Element data - just draw all vertices in order
	GLushort elementData[] = {
		0,  1,  2,  3,
		4,  5,  6,  7,
		8,  9,  10, 11,
		12, 13, 14, 15,
		16, 17, 18, 19,
		20, 21, 22, 23
	};
	elementBuffer = makeBO(
		GL_ELEMENT_ARRAY_BUFFER,
		elementData,
		sizeof( elementData ),
		GL_STATIC_DRAW
	);

	// Load textures.
	defaultTextureData = loadTexture( "texture.tga" );
}

// Initialize shaders.
void initShaders() {
	// Load a simple Vertex/Fragment shader
	GLuint vertexShader = loadShader( GL_VERTEX_SHADER, "simple.vert" );
	GLuint fragmentShader = loadShader( GL_FRAGMENT_SHADER, "simple.frag" );
	shaderProgram = makeShaderProgram( vertexShader, fragmentShader );

	// Get locations of attributes and uniforms used inside.
	vertexPosition = glGetAttribLocation( shaderProgram, "vertex" );
	vertexNormal = glGetAttribLocation( shaderProgram, "vnormal" );
	vertexTexcoords = glGetAttribLocation( shaderProgram, "texcoords" );
	worldviewMatrix = glGetUniformLocation( shaderProgram, "worldview" );
	polyviewMatrix = glGetUniformLocation( shaderProgram, "polyview" );
	normalviewMatrix = glGetUniformLocation( shaderProgram, "normalview" );
	cuberotMatrix = glGetUniformLocation( shaderProgram, "cubeview" );
	projectionMatrix = glGetUniformLocation( shaderProgram, "projection" );
	drawDistfield = glGetUniformLocation( shaderProgram, "drawDistfield" );
	defaultTexture = glGetUniformLocation( shaderProgram, "texture" );
	shaderTime = glGetUniformLocation( shaderProgram, "time" );
}

// Update status variables.
// Called every 15ms, unless the PC is too slow.
void update() {
	// Update things here.

	float speed = 0.4;

	angle += 0.03;
	cubeangle += 0.02;
	shadertime += 0.1;

	pv = MakeVector(posx, posy, posz);
	Vector nv = VectorMul( VectorNorm( fieldd1(posx, posy, posz) ), -1 );
	Matrix rot = RotationMatrix( (rotatey/360.0) * 3.14, nv );
	dv = VectorMul( VectorNorm( VectorCross( sv, nv ) ), -1 );
	sv = VectorNorm( VectorCross( dv, nv ) );
	Vector dvd = TransformVector(rot,dv);
	Vector svd = TransformVector(rot,sv);
	Vector dvs = VectorMul(dvd,speed);
	Vector svs = VectorMul(svd,speed);

	if( key[' '] == 1 ) {
		if( grounded == 1 ) {
			grounded = 0;
			y_momentum = -1.0;
			jdir_x = -nv.x;
			jdir_y = -nv.y;
			jdir_z = -nv.z;
		}
	}
	if( key['w'] == 1 ) {
		pv = VectorSub( pv, dvs );
	}
	if( key['s'] == 1 ) {
		pv = VectorAdd( pv, dvs );
	}
	if( key['a'] == 1 ) {
		pv = VectorSub( pv, svs );
	}
	if( key['d'] == 1 ) {
		pv = VectorAdd( pv, svs );
	}
	if( key['e'] == 1 ) {
		posy += 0.4;
	}
	if( key['q'] == 1 ) {
		posy -= 0.4;
	}	

	posx = pv.x;
	posy = pv.y;
	posz = pv.z;

	vm = MakeMatrixFromVectors( VectorMul(svd,-1), VectorMul(nv,-1), VectorMul(dvd,-1), MakeVector( 0, 0, 0 ) );
	
	// Redraw screen now, please, and call again in 15ms.
	glutPostRedisplay();
	glutTimerFunc( 15, update, 0 );
}

float tracedown( float ex, float ey, float ez, float dx, float dy, float dz) {
	int iter = 0;
	float dist = 200.0;
	float disttotal = 0;
	while( iter < 150 && dist > 0.01 ) {
		dist = field(ex,ey,ez);
		ex += dx * dist;
		ey += dy * dist;
		ez += dz * dist;
		disttotal += dist;
	}
	return( disttotal );
}

// Draw the scene to the screen
void draw() {
	// Clear everything first thing.
	glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
	
	// Activate shader.
	glUseProgram( shaderProgram );

	// Set projection - you don't have to recalculate this each frame, but
	// for the sake of simplicity that's what is done here.
	MatrixAsUniform(
		projectionMatrix,
		PerspectiveMatrix(
			45.0,
			(float)WINDOW_WIDTH/(float)WINDOW_HEIGHT,
			1.0,
			1000.0
		)
	);
	
	// Vertices
	glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer );
	glVertexAttribPointer(
		vertexPosition,
		4,
		GL_FLOAT,
		GL_FALSE,
		sizeof(GLfloat) * 9,
		(void*)0
	);
	glEnableVertexAttribArray( vertexPosition );

	// Normals
	glVertexAttribPointer(
		vertexNormal,
		3,
		GL_FLOAT,
		GL_FALSE,
		sizeof(GLfloat) * 9,
		(void*)(sizeof(GLfloat) * 4)
	);
	glEnableVertexAttribArray( vertexNormal );
	
	// Texture coordinates
	glVertexAttribPointer(
		vertexTexcoords,
		2,
		GL_FLOAT,
		GL_FALSE,
		sizeof(GLfloat) * 9,
		(void*)(sizeof(GLfloat) * 7)
	);
	glEnableVertexAttribArray( vertexTexcoords );
	
	// Set worldview - the world rotation and translation relative to camera
	Matrix rot = XAxisRotationMatrix( (-rotatex/360.0) * 3.14 );
// 	rot = MatrixMul( rot, YAxisRotationMatrix( (-rotatey/360.0) * 3.14 ) );
	rot = MatrixMul( vm, rot );
	Vector normal = VectorNorm( fieldd1( posx, posy, posz ) );
	float dist = tracedown(posx,posy,posz,normal.x,normal.y,normal.z);
	dist = fabs(dist);
	if( dist > 2.0 ) {
		posx += normal.x * 0.4;
		posy += normal.y * 0.4;
		posz += normal.z * 0.4;
	}
	else {
		if( grounded == 0 ) {
			grounded = 1;
		}
	}

	float posxt = posx + jdir_x * y_momentum;
	float posyt = posy + jdir_y * y_momentum;
	float poszt = posz + jdir_z * y_momentum;

	Vector normal2 = VectorNorm( fieldd1( posxt, posyt, poszt ) );
	float dist2 = tracedown(posxt,posyt,poszt,normal2.x,normal2.y,normal2.z);

	if( dist2 > 2.0 ) {
		posx = posxt;
		posy = posyt;
		posz = poszt;
	}
	else {
		y_momentum = 0;
	}
	
	if( y_momentum < 0 ) {
		y_momentum += 0.01;
	}
	else {
		y_momentum = 0;
	}
	Matrix trans = TranslationMatrix( posx, posy, posz );
	Matrix worldview = MatrixMul( trans, rot );
	Matrix polyview = MatrixMul( FastMatrixInverse(rot), trans );
	MatrixAsUniform( worldviewMatrix, worldview );
	MatrixAsUniform( polyviewMatrix, polyview );

	// Rotate and scale the cube
	Matrix cuberot = YAxisRotationMatrix( cubeangle );
	cuberot = MatrixMul( cuberot, ZAxisRotationMatrix( cubeangle / 2.0 ) );
	cuberot = MatrixMul( cuberot, ScalingMatrix( 3.0, 3.0, 3.0 ) );
	MatrixAsUniform( cuberotMatrix, cuberot );

	// Normal view matrix - inverse transpose of worldview.
	Matrix normalview = MatrixTranspose( FastMatrixInverse( worldview ) );
	MatrixAsUniform( normalviewMatrix, normalview );

	// Textures
	glActiveTexture( GL_TEXTURE0 );
	glBindTexture( GL_TEXTURE_2D, defaultTextureData );
	glUniform1i( defaultTexture, 0 );

	// Other variables
	glUniform1f( shaderTime, shadertime );
	
	// Send element buffer to GPU.
	glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, elementBuffer );

	// Drawing, part 1: Distance field.
	glUniform1i( drawDistfield, 1 );
	glDrawElements(
		GL_QUADS,
		4,
		GL_UNSIGNED_SHORT,
		(void*)0
	);

	// Drawing, part 2: Cube
	glUniform1i( drawDistfield, 0 );
	glDrawElements(
		GL_QUADS,
		24,
		GL_UNSIGNED_SHORT,
		(void*)0
	);

	// Drawing, part 3: knot.
	float scale = 8.0;
// 	float scaled = 3.0 * angle - (int)( (3.0 * angle) / 2.0 )*2;
// 	if( scaled < 1.0 ) {
// 		scale = 8 + scaled * 3.0;
// 	}
    scale = ((sin(angle) + 1.0) / 4.0 + 1.5) * 5.0;
	for( int i = 0; i < 50; i++ ) {
		float u = (i/50.0) * 2.0 * 3.14 + shadertime/10.0;
		float x =
			41.0 * cos(u) -
			18.0 * sin(u) -
			83.0 * cos(2.0 * u) -
			83.0 * sin(2.0 * u) -
			11.0 * cos(3.0 * u) +
			27.0 * sin(3.0 * u);
			x /= 100.0;
			x *= scale;

		float y =
			36.0 * cos(u) +
			27.0 * sin(u) -
			113.0 * cos(2.0 * u) +
			30.0 * sin(2.0 * u) +
			11.0 * cos(3.0 * u) -
			27.0 * sin(3.0 * u);
			y /= 100;
			y *= scale;

		float z =
			45.0 * sin(u) -
			30.0 * cos(2.0 * u) +
			113.0 * sin(2.0 * u) -
			11.0 * cos(3.0 * u) +
			27.0 * sin(3.0 * u);
			z /= 100;
			z *= scale;

		cuberot = TranslationMatrix( x, y, z );
		MatrixAsUniform( cuberotMatrix, cuberot );
		glUniform1i( drawDistfield, 0 );
		glDrawElements(
			GL_QUADS,
			24,
			GL_UNSIGNED_SHORT,
			(void*)0
		);
	}
	
	// Clean up
	glDisableVertexAttribArray( vertexPosition );
	glDisableVertexAttribArray( vertexTexcoords );

	// Switch drawing area and displayed area.
	glutSwapBuffers();
}

// Just:
//	1) Set up OpenGL
//	2) Set up GLUT
//	3) Run GLUT
int main( int argc, char** argv ) {
	makeWindow( argc, argv );
	initObjects();
	initShaders();
	setupGlutCallbacks();
	glutMainLoop();
}