// Cam controls; based on THREE.OrbitControls.
THREE.DungeonControls = function (camera, domElement) {
	this.camera = camera;
	this.domElement = (domElement !== undefined) ? domElement : document;

	// API
	this.controlsEnabled = true;
	this.center = new THREE.Vector3();
	this.userRotateSpeed = 1.0;
	this.userZoomSpeed = 1.0;
	this.minPolarAngle = 0; 				// radians
	this.maxPolarAngle = Math.PI; 			// radians
	this.minDistance = 0;
	this.maxDistance = Infinity;

	// internals
	var scope = this;
	var EPS = 0.000001;
	var PIXELS_PER_ROUND = 1800;
	var rotateStart = new THREE.Vector2();
	var rotateEnd = new THREE.Vector2();
	var rotateDelta = new THREE.Vector2();
	var phiDelta = 0;
	var thetaDelta = 0;
	var scale = 1;
	var lastPosition = new THREE.Vector3();
	var STATE = { NONE: -1, ROTATE: 0 };
	var state = STATE.NONE;
	var changeEvent = {type: 'change'};
	
	this.rotateLeft = function (angle) {
		thetaDelta -= angle;
	};

	this.rotateRight = function (angle) {
		thetaDelta += angle;
	};

	this.rotateUp = function (angle) {
		phiDelta -= angle;
	};

	this.rotateDown = function (angle) {
		phiDelta += angle;
	};

	this.zoomIn = function (zoomScale) {
		if (zoomScale === undefined)
			zoomScale = getZoomScale();
		scale /= zoomScale;
	};

	this.zoomOut = function (zoomScale) {
		if (zoomScale === undefined)
			zoomScale = getZoomScale();
		scale *= zoomScale;
	};

	this.update = function () {
		var position = this.camera.position;
		var offset = position.clone().sub(this.center);

		var theta = Math.atan2(offset.x, offset.z);												// angle from z-axis around y-axis
		theta += thetaDelta;		
		if (this.controlsEnabled)
			theta=Math.max(-Math.PI/4+.1,Math.min(Math.PI/4-.1,theta));
		
		var phi = Math.atan2(Math.sqrt(offset.x * offset.x + offset.z * offset.z), offset.y);	// angle from y-axis		
		phi += phiDelta;
		phi = Math.max(this.minPolarAngle, Math.min(this.maxPolarAngle, phi));
		phi = Math.max(EPS, Math.min(Math.PI - EPS, phi));
		var radius = offset.length() * scale;
		radius = Math.max(this.minDistance, Math.min(this.maxDistance, radius));
		offset.x = radius * Math.sin(phi) * Math.sin(theta);
		offset.y = radius * Math.cos(phi);
		offset.z = radius * Math.sin(phi) * Math.cos(theta);

		position.copy(this.center).add(offset);
		this.camera.lookAt(this.center);

		thetaDelta = 0;
		phiDelta = 0;
		scale = 1;

		if (lastPosition.distanceTo(this.camera.position) > 0) {
			this.dispatchEvent(changeEvent);
			lastPosition.copy(this.camera.position);
		}
	};

	function getZoomScale() {
		return Math.pow(0.95, scope.userZoomSpeed);
	}

	function onMouseDown(event) {
		if (scope.controlsEnabled === false) return;
		event.preventDefault();

		if (state === STATE.NONE) {
			if (event.button === 0)
				state = STATE.ROTATE;
//			if (event.button === 2)
//				ThreeEngine.doPicking(event.clientX, event.clientY);
		}
		
		if (state === STATE.ROTATE)
			rotateStart.set(event.clientX, event.clientY);
		document.addEventListener('mousemove', onMouseMove, false);
		document.addEventListener('mouseup', onMouseUp, false);
	}

	function onMouseMove(event) {
		if (scope.controlsEnabled === false) return;
		event.preventDefault();
		if (state === STATE.ROTATE) {
			rotateEnd.set(event.clientX, event.clientY);
			rotateDelta.subVectors(rotateEnd, rotateStart);
			scope.rotateLeft(2 * Math.PI * rotateDelta.x / PIXELS_PER_ROUND * scope.userRotateSpeed);
			scope.rotateUp(2 * Math.PI * rotateDelta.y / PIXELS_PER_ROUND * scope.userRotateSpeed);
			rotateStart.copy(rotateEnd);
		}
	}

	function onMouseUp(event) {
		if (scope.controlsEnabled === false) return;
		document.removeEventListener('mousemove', onMouseMove, false);
		document.removeEventListener('mouseup', onMouseUp, false);
		state = STATE.NONE;
	}

	function onMouseWheel(event) {
		if (scope.controlsEnabled === false) return;
		var delta = 0;

		if (event.wheelDelta) // WebKit / Opera / Explorer 9
			delta = event.wheelDelta;
		else if (event.detail) // Firefox
			delta = - event.detail;

		if (delta > 0)
			scope.zoomOut();
		else
			scope.zoomIn();
	}

	this.domElement.addEventListener('contextmenu', function (event) { event.preventDefault(); }, false);
	this.domElement.addEventListener('mousedown', onMouseDown, false);
	this.domElement.addEventListener('mousewheel', onMouseWheel, false);
	this.domElement.addEventListener('DOMMouseScroll', onMouseWheel, false); // firefox
};

THREE.DungeonControls.prototype = Object.create(THREE.EventDispatcher.prototype);
