Game.Map = function(tiles) {
    this._tiles = tiles;
    this._depth = tiles.length
    this._width = tiles[0].length;
    this._height = tiles[0][0].length;
    this._fov = [];
    this.setupFov();
    this._entities = {};
    this._items = {};
    this._scheduler = new ROT.Scheduler.Speed();
    this._engine = new ROT.Engine(this._scheduler);
    this._explored = new Array(this._depth);
    this._setupExploredArray();
};

Game.Map.prototype._setupExploredArray = function() {
    for (var z = 0; z < this._depth; z++) {
        this._explored[z] = new Array(this._width);
        for (var x = 0; x < this._width; x++) {
            this._explored[z][x] = new Array(this._height);
            for (var y = 0; y < this._height; y++)
                this._explored[z][x][y] = false;
        }
    }
};

// Getters
Game.Map.prototype.getDepth = function() {
    return this._depth;
};
Game.Map.prototype.getWidth = function() {
    return this._width;
};
Game.Map.prototype.getHeight = function() {
    return this._height;
};
Game.Map.prototype.getFov = function(depth) {
    return this._fov[depth];
};
Game.Map.prototype.getEngine = function() {
    return this._engine;
};
Game.Map.prototype.getEntities = function() {
    return this._entities;
};
Game.Map.prototype.getPlayer = function() {
    return this._player;
};

// Get tile at given position
Game.Map.prototype.getTile = function(x, y, z) {
    if (x < 0 || x >= this._width || y < 0 || y >= this._height || z < 0 || z >= this._depth)
        return Game.Tile.nullTile;
    else
        return this._tiles[z][x][y] || Game.Tile.nullTile;
};

// If given tile is diggable, change to floor
Game.Map.prototype.dig = function(x, y, z) {
    if (this.getTile(x, y, z).isDiggable())
        this._tiles[z][x][y] = Game.Tile.floorTile;
};

// Check if the tile is floor and also has no entity
Game.Map.prototype.isEmptyFloor = function(x, y, z) {
    return this.getTile(x, y, z) == Game.Tile.floorTile && !this.getEntityAt(x, y, z);
};

// Mark a tile as explored
Game.Map.prototype.setExplored = function(x, y, z, state) {
    if (this.getTile(x, y, z) !== Game.Tile.nullTile)
        this._explored[z][x][y] = state;
};

// Check a tile for having been explored
Game.Map.prototype.isExplored = function(x, y, z) {
    if (this.getTile(x, y, z) !== Game.Tile.nullTile)
        return this._explored[z][x][y];
    else
        return false;
};

// Iterate through each depth level, setting up the field of vision
Game.Map.prototype.setupFov = function() {
    var map = this;
    for (var z = 0; z < this._depth; z++) {
        // We have to put the following code in it's own scope to prevent the depth variable from being hoisted out of the loop.
        (function() {
            // For each depth, we need to create a callback which figures out if light can pass through a given tile.
            var depth = z;
            map._fov.push(
				new ROT.FOV.PreciseShadowcasting(function(x, y) {
                    return !map.getTile(x, y, depth).isBlockingLight();
                }, {topology: 8}));
        })();
    }
};

// Get the entity based on position key
Game.Map.prototype.getEntityAt = function(x, y, z){ 
    return this._entities[x + ',' + y + ',' + z];
};

// Get all entities in a given radius
Game.Map.prototype.getEntitiesWithinRadius = function(centerX, centerY, centerZ, radius) {
    results = [];
    var leftX = centerX - radius;
    var rightX = centerX + radius;
    var topY = centerY - radius;
    var bottomY = centerY + radius;
    for (var key in this._entities) {
        var entity = this._entities[key];
        if (entity.getX() >= leftX && entity.getX() <= rightX && entity.getY() >= topY && entity.getY() <= bottomY && entity.getZ() == centerZ)
            results.push(entity);
    }
    return results;
};

// Get a random floor position
Game.Map.prototype.getRandomFloorPosition = function(z) {
    var x, y;
    do {
        x = Math.floor(Math.random() * this._width);
        y = Math.floor(Math.random() * this._height);
    } while(!this.isEmptyFloor(x, y, z));
    return {x: x, y: y, z: z};
};

// Add an entity on a random floor position
Game.Map.prototype.addEntityAtRandomPosition = function(entity, z) {
    var position = this.getRandomFloorPosition(z);
    entity.setX(position.x);
    entity.setY(position.y);
    entity.setZ(position.z);
    this.addEntity(entity);
};

// Add an entity
Game.Map.prototype.addEntity = function(entity) {
    entity.setMap(this);
    this.updateEntityPosition(entity);	    // Update the map with the entity's position
    if (entity.hasMixin('Actor'))			// Add to scheduler
       this._scheduler.add(entity, true);
    if (entity.hasMixin(Game.EntityMixins.PlayerActor))	// Are they the player?
        this._player = entity;
};

// Remove an entity
Game.Map.prototype.removeEntity = function(entity) {
    var key = entity.getX() + ',' + entity.getY() + ',' + entity.getZ();
    if (this._entities[key] == entity) {
        delete this._entities[key];
    }
    if (entity.hasMixin('Actor'))
        this._scheduler.remove(entity);
	if (entity.hasMixin(Game.EntityMixins.PlayerActor))
        this._player = undefined;
};

// Update an entity's position
Game.Map.prototype.updateEntityPosition = function(entity, oldX, oldY, oldZ) {
    // Delete the old key if it is the same entity and we have old positions.
    if (typeof oldX === 'number') {
        var oldKey = oldX + ',' + oldY + ',' + oldZ;
        if (this._entities[oldKey] == entity) {
            delete this._entities[oldKey];
        }
    }
    // Make sure the entity's position is within bounds
    if (entity.getX() < 0 || entity.getX() >= this._width ||
        entity.getY() < 0 || entity.getY() >= this._height ||
        entity.getZ() < 0 || entity.getZ() >= this._depth) {
        throw new Error("Entity's position is out of bounds.");
    }
    // Sanity check to make sure there is no entity at the new position.
    var key = entity.getX() + ',' + entity.getY() + ',' + entity.getZ();
    if (this._entities[key]) {
        throw new Error('Tried to add an entity at an occupied position.');
    }
    // Add the entity to the table of entities
    this._entities[key] = entity;
};

// Get items for a given position
Game.Map.prototype.getItemsAt = function(x, y, z) {
    return this._items[x + ',' + y + ',' + z];
};

// Set items for a given position
Game.Map.prototype.setItemsAt = function(x, y, z, items) {
    // If our items array is empty, then delete the key from the table.
    var key = x + ',' + y + ',' + z;
    if (items.length === 0) {
        if (this._items[key])
            delete this._items[key];
    } else // Simply update the items at that key
        this._items[key] = items;
};

// Add an item at a given position
Game.Map.prototype.addItem = function(x, y, z, item) {
    var key = x + ',' + y + ',' + z;
    if (this._items[key])
        this._items[key].push(item);
    else
        this._items[key] = [item];
};

// Add an item at a random floor position
Game.Map.prototype.addItemAtRandomPosition = function(item, z) {
    var position = this.getRandomFloorPosition(z);
    this.addItem(position.x, position.y, position.z, item);
};
