#pragma once

#include <QAbstractSocket>

#include "Constants.h"

class Action
{
	public:
		enum Type
		{
			None,
			Accelerate,
			TurnLeft,
			TurnRight,
			FireNormalMissile,
			FireSeekingMissile,
			DropMine,
			NumberOfAvailableActions
		};

		Action()
			: m_type(None)
		{}

		Action(Type type)
			: m_type(type)
		{}

		Type type() const
		{
			return m_type;
		}

		int cost() const
		{
			switch (m_type)
			{
				case Accelerate:
					return C::Ship::ACCELERATION_COST;
				case TurnLeft:
					return C::Ship::ROTATE_COST;
				case TurnRight:
					return C::Ship::ROTATE_COST;
				case FireNormalMissile:
					return C::Missile::NORMAL_COST;
				case FireSeekingMissile:
					return C::Missile::SEEKING_COST;
				case DropMine:
					return C::Missile::MINE_COST;
				default:
					return 0;
			}
		}

	private:
		QString toString() const
		{
			switch (m_type)
			{
				case None:
					return "NONE"; // NONE appears to be safe to send for "no action", more sane than sending nothing
				case Accelerate:
					return "ACCELERATE";
				case TurnLeft:
					return "LEFT";
				case TurnRight:
					return "RIGHT";
				case FireNormalMissile:
					return "MISSILE";
				case FireSeekingMissile:
					return "SEEKING";
				case DropMine:
					return "MINE";
				default:
					qWarning() << "Trying to convert an unknown action to a string";
					return "";
			}
		}

		Type m_type;

	friend QAbstractSocket& operator<<(QAbstractSocket& socket, const Action& action);
};

inline QAbstractSocket& operator<<(QAbstractSocket& socket, const Action& action)
{
	socket.write(QString("%1\n").arg(action.toString()).toLocal8Bit());
	return socket;
}
