#pragma once

#include <QtCore>

#include "Angle.h"
#include "Constants.h"
#include "Vec2D.h"
#include "Velocity.h"

class Position : public Vec2D<double>
{
	public:
		enum class DistanceMode
		{
			Direct,
			Wrapping
		};

		constexpr Position()
			: Vec2D(0, 0)
		{}


		constexpr Position(double x, double y)
			: Vec2D(x, y)
		{}

		Angle angleTo(const Position& other) const
		{
			const auto& dx = other.x() - x();
			const auto& dy = other.y() - y();
			auto rotation = qAtan2(dy, dx);
			if (rotation < 0)
			{
				rotation += M_PI * 2.;
			}
			else if (rotation > M_PI * 2.)
			{
				rotation -= M_PI * 2.;
			}

			return Angle::fromRadians(rotation);
		}

		double distanceTo(const Position& other, DistanceMode mode = DistanceMode::Direct) const
		{
			if (mode == DistanceMode::Wrapping)
			{
				const auto& dx = qAbs(x() - other.x());
				const auto& dy = qAbs(y() - other.y());

				return qSqrt(
					qPow(qMin(dx, C::World::WIDTH - dx), 2)
					+ qPow(qMin(dy, C::World::HEIGHT - dy), 2)
				);
			}
			else
			{
				const auto& dx = x() - other.x();
				const auto& dy = y() - other.y();

				return qSqrt(dx * dx + dy * dy);
			}
		}

		constexpr void operator+=(const Velocity& velocity)
		{
			auto nx = x() + velocity.x();
			auto ny = y() + velocity.y();

			if (nx > 1.0)
			{
				nx = -1.0;
			}
			else if (nx < -1.0)
			{
				nx = 1.0;
			}

			if (ny > 1.0)
			{
				ny = -1.0;
			}
			else if (ny < -1.0)
			{
				ny = 1.0;
			}

			setX(nx);
			setY(ny);
		}

		static constexpr Position sun()
		{
			return Position(C::World::SUN_X, C::World::SUN_Y);
		}
};
