#include <QDebug>
#include <QJsonArray>

#include "Event.h"
#include "State.h"

void State::parse(const QJsonObject &jsonObject)
{
	clear();

	const auto& missiles = jsonObject.find("missiles").value();
	if (missiles.type() == QJsonValue::Array)
	{
		const auto& missilesArray = missiles.toArray();
		if (missilesArray.count() > 0)
		{
			m_missiles.reserve(missilesArray.count());
			for (const auto& missileData : missilesArray)
			{
				const auto& missileObject = missileData.toObject();
				m_missiles.append(Missile(missileObject));
			}
		}
	}
	else
	{
		qCritical() << "State update did not contain \"missiles\" array";
	}

	const auto& otherShips = jsonObject.find("others").value();
	if (otherShips.type() == QJsonValue::Array)
	{
		const auto& shipArray = otherShips.toArray();
		m_ships.reserve(shipArray.count() + 1); // all enemies plus ourself
		for (const auto& shipData : shipArray)
		{
			const auto& shipObject = shipData.toObject();
			m_ships.append(Ship(shipObject));
		}
	}
	else
	{
		qCritical() << "State update did not contain \"others\" array";
	}

	const auto& myShip = jsonObject.find("you").value();
	if (myShip.type() == QJsonValue::Object)
	{
		const auto& shipObject = myShip.toObject();
		m_ships.append(Ship(shipObject));
	}
	else
	{
		qCritical() << "State update did not contain \"you\" object";
	}
}

void State::step()
{
	for (auto& missile : m_missiles)
	{
		missile.step();

		const Ship* closestEnemyShip = nullptr;
		for (const auto& ship : m_ships)
		{
			if (!ship.isAlive() || missile.ownerId() == ship.id())
			{
				continue;
			}

			if (missile.type() == Missile::Type::Seeking)
			{
				if (!closestEnemyShip || missile.position().distanceTo(ship.position()) < missile.position().distanceTo(closestEnemyShip->position()))
				{
					closestEnemyShip = &ship;
				}
			}
		}

		if (closestEnemyShip)
		{
			missile.rotation() = missile.position().angleTo(closestEnemyShip->position());
		}
	}

	for (auto& ship : m_ships)
	{
		ship.step();
		ship.energy().drain(1);
	}
}

int State::simulateMissileLaunch(Missile::Type type, int maxTicks) const
{
	State simulationState(*this);
	simulationState.step(); // Step since the missile will spawn at our position next tick
	simulationState.step();
	simulationState.missiles().append(
		Missile(myShip().id(), type, simulationState.myShip().rotation(), simulationState.myShip().position())
	);
//	simulationState = *this;
//	simulationState.step();

	for (int i = 0; i < maxTicks; i++)
	{
		if (simulationState.missiles().last().position().distanceTo(Position::sun()) < 0.1)
		{
			return -1;
		}

		for (const auto& ship : simulationState.ships())
		{
			if (ship.id() == simulationState.myShip().id())
			{
				continue;
			}

			if (simulationState.missiles().last().position().distanceTo(ship.position()) < 0.1)
			{
				return i;
			}
		}

		simulationState.step();
	}

	return -1;
}

int State::simulateIfHitByMissile(int maxTicks) const
{
	State simulationState(*this);
	for (int i = 0; i < maxTicks; i++)
	{
		for (const auto& missile : missiles())
		{
			if (missile.position().distanceTo(simulationState.myShip().position()) < 0.1)
			{
				return i;
			}
		}

		simulationState.step();
	}

	return -1;
}

void State::printHumanReadableDifference(const State &other) const
{
	if (m_missiles != other.m_missiles)
	{
		qDebug() << "Missiles differ:";
		if (m_missiles.count() != other.m_missiles.count())
		{
			qDebug() << "Number of missiles differ, no individual comparison will be done\n";
		}
		else
		{
			for (int i = 0; i < m_missiles.count(); i++)
			{
				if (m_missiles[i].id() != other.m_missiles[i].id())
				{
					qDebug() << "id differs," << m_missiles[i].id() << "!=" << other.m_missiles[i].id();
				}
				if (m_missiles[i].energy().value() != other.m_missiles[i].energy().value())
				{
					qDebug() << "energy differs," << m_missiles[i].energy().value() << "!=" << other.m_missiles[i].energy().value();
				}
				if (m_missiles[i].ownerId() != other.m_missiles[i].ownerId())
				{
					qDebug() << "ownerId differs," << m_missiles[i].ownerId() << "!=" << other.m_missiles[i].ownerId();
				}
				if (m_missiles[i].rotation() != other.m_missiles[i].rotation())
				{
					qDebug() << "rotation differs," << m_missiles[i].rotation().asDegrees() << "!=" << other.m_missiles[i].rotation().asDegrees();
				}
				if (m_missiles[i].type() != other.m_missiles[i].type())
				{
					qDebug() << "type differs," << (int)m_missiles[i].type() << "!=" << (int)other.m_missiles[i].type();
				}
				if (m_missiles[i].velocity() != other.m_missiles[i].velocity())
				{
					qDebug() << "velocity differs," << m_missiles[i].velocity() << "!=" << other.m_missiles[i].velocity();
				}
				if (m_missiles[i].position() != other.m_missiles[i].position())
				{
					qDebug() << "position differs," << m_missiles[i].position() << "!=" << other.m_missiles[i].position();
				}
			}
		}
	}
}

void State::clear()
{
	// Don't .clear() QVector before Qt5.6, won't preserve capacity - https://codereview.qt-project.org/#/c/127376/
	m_missiles.erase(m_missiles.begin(), m_missiles.end());
	m_ships.erase(m_ships.begin(), m_ships.end());
}
