#pragma once

#include <QJsonObject>
#include <QVector>

#include "Action.h"
#include "Missile.h"
#include "Ship.h"

class IEvent;

class State
{
	public:
		State() = default;

		void parse(const QJsonObject& jsonObject);
		void step();

		int simulateMissileLaunch(Missile::Type type, int maxTicks) const;
		int simulateIfHitByMissile(int maxTicks) const;

		QVector<Missile>& missiles()
		{
			return m_missiles;
		}

		const QVector<Missile>& missiles() const
		{
			return m_missiles;
		}

		QVector<Ship>& ships()
		{
			return m_ships;
		}

		const QVector<Ship>& ships() const
		{
			return m_ships;
		}

		Ship& myShip()
		{
			return m_ships.last();
		}

		const Ship& myShip() const
		{
			// By the way that the deserialization works we can guarantee that
			// our own ship will always be the last one in the vector
			return m_ships.last();
		}

		const Ship& closestEnemyShip() const
		{
			if (m_ships.count() <= 2)
			{
				return m_ships[0];
			}

			const Ship* closestShip = nullptr;
			for (const auto& ship : ships())
			{
				if (!closestShip ||
					(ship.id() != myShip().id()
					&&	(myShip().position().distanceTo(ship.position(), Position::DistanceMode::Wrapping)
						< myShip().position().distanceTo(closestShip->position(), Position::DistanceMode::Wrapping))))
				{
					closestShip = &ship;
				}
			}

			return *closestShip;
		}

		void printHumanReadableDifference(const State& other) const;

		bool operator==(const State& other) const
		{
			return m_missiles == other.m_missiles && m_ships == other.m_ships;
		}

		bool operator!=(const State& other) const
		{
			return !operator==(other);
		}

	private:
		void clear();

		QVector<Missile> m_missiles;
		QVector<Ship> m_ships;
};
