#define _USE_MATH_DEFINES
#include <cmath>
#include "missile.h"
#include "parameters.h"

#define MISSILE_SPEED_MINE    0.005
#define MISSILE_SPEED_NORMAL  0.05
#define MISSILE_SPEED_SEEKING 0.03

Missile::Missile(Type type, QPointF start_position, int start_rotation, int owner)
  : m_type(type)
  , m_position(start_position)
  , m_energy(1000)
  , m_alive(true)
  , m_owner(owner)
  , m_age(0)
{
  if (Mine == type)
  {
    setRotation(atan2(start_position.y(), start_position.x()));
    m_velocityX = cos(m_rotation) * MISSILE_SPEED_MINE;
    m_velocityY = sin(m_rotation) * MISSILE_SPEED_MINE;
    m_energy = 5000;
    return;
  }

  setRotation((start_rotation * M_PI * 2) / 360.0);

  qreal speed_constant = 0.0;
  
  if (Normal == type)
  {
    speed_constant = MISSILE_SPEED_NORMAL;
  }
  else if (Seeking == type)
  {
    speed_constant = MISSILE_SPEED_SEEKING;
  }
  
  m_velocityX = cos(m_rotation) * speed_constant;
  m_velocityY = sin(m_rotation) * speed_constant;
}

Missile::Missile(Type type, QPointF position, qreal rotation, qreal velocityX, qreal velocityY, int energy, int owner)
  : m_type(type)
  , m_position(position)
  , m_rotation(rotation)
  , m_velocityX(velocityX)
  , m_velocityY(velocityY)
  , m_energy(energy)
  , m_alive(true)
  , m_owner(owner)
{}

void Missile::setRotation(qreal rotation)
{
  if (rotation < 0)
  {
    rotation += M_PI * 2.0;
  }

  if (rotation > M_PI * 2.0)
  {
    rotation -= M_PI * 2.0;
  }

  m_rotation = rotation;
}

void Missile::doMove()
{
  m_age++;
  qreal x = m_position.x();
  qreal y = m_position.y();
  const qreal distance = hypot(x, y);

  if (distance < 0.1) {
    m_alive = false;
    return;
  }

  qreal velocityMagnitude = hypot(m_velocityX, m_velocityY);
  if (velocityMagnitude > MISSILE_MAX_SPEED) {
    qreal velocityAngle = atan2(m_velocityY, m_velocityX);
    m_velocityX = cos(velocityAngle) * MISSILE_MAX_SPEED;
    m_velocityY = sin(velocityAngle) * MISSILE_MAX_SPEED;
  }

  const qreal force = distance / 1000;
  const qreal angle = atan2(y, x);
  m_velocityX -= cos(angle) * force;
  m_velocityY -= sin(angle) * force;

  x += m_velocityX;
  y += m_velocityY;

  if (x > 1.0) { x = -1.0; }
  if (y > 1.0) { y = -1.0; }
  if (x < -1.0) { x = 1.0; }
  if (y < -1.0) { y = 1.0; }

  m_position.setX(x);
  m_position.setY(y);

  // Always point in the right direction
  if (m_type == Normal) {
    setRotation(atan2(m_velocityY, m_velocityX));
  }

  // Just fall into the sun
  if (m_energy < 10) {
    m_velocityX /= 1.01;
    m_velocityY /= 1.01;
    return;
  }

  if (m_type == Mine) {
    m_velocityX += cos(m_rotation) * 0.0005;
    m_velocityY += sin(m_rotation) * 0.0005;
    m_energy-= 1;
    return;
  }

  m_energy-= 50;

  if (m_type == Normal) {
    m_velocityX += cos(m_rotation) * (m_energy / 1000000.0);
    m_velocityY += sin(m_rotation) * (m_energy / 1000000.0);
  } else if (m_type == Seeking) {
    m_velocityX += cos(m_rotation) * (m_energy / 100000.0);
    m_velocityY += sin(m_rotation) * (m_energy / 100000.0);
  }
}
