#define _USE_MATH_DEFINES
#include <cmath>
#include "player.h"
#include "parameters.h"

Player::Player()
  : m_id(0)
  , m_alive(true)
  , m_energy(START_ENERGY)
{}

Player::Player(int id, int energy, int rotation, QPointF position, qreal velocityX, qreal velocityY)
  : m_id(id)
  , m_alive(true)
  , m_energy(energy)
  , m_position(position)
  , m_velocityX(velocityX)
  , m_velocityY(velocityY)
  , m_score(0)
{
  setRotation(rotation);
}

int Player::id()
{
  return m_id;
}

void Player::setId(int id)
{
  m_id = id;
}

QString Player::lastCommand()
{
  return m_lastCommand;
}

QString Player::command()
{
  QString command = m_command;
  m_command.clear();

  if (m_lastCommand != command) {
    m_lastCommand = command;
  }

  return command;
}

void Player::setCommand(QString command)
{
  if (!isAlive()) {
    return;
  }

  m_command = command;
}

void Player::setAlive(bool alive)
{
  if (alive == m_alive) {
    return;
  }

  m_alive = alive;
}

bool Player::isAlive()
{
  return m_alive;
}

void Player::setRotation(int rotation)
{
  if (rotation < 0) {
    rotation += 360;
  }
  if (rotation > 360) {
    rotation -= 360;
  }

  m_rotation = rotation;
}

void Player::setPosition(QPointF position)
{
  qreal velocityAngle = atan2(position.y(), position.x()) + M_PI_2;
  m_velocityX = cos(velocityAngle) / 35.0;
  m_velocityY = sin(velocityAngle) / 35.0;

  m_position = position;

  setRotation(velocityAngle * 360 / (M_PI * 2.0));
}

void Player::decreaseEnergy(int amount)
{
  if (m_energy <= 0) {
    return;
  }

  m_energy -= amount;
  if (m_energy < 0) m_energy = 0;

  if (m_energy == 0 && m_alive) {
    setAlive(false);
  }
}

void Player::increaseEnergy(int amount)
{
  if (m_energy <= 0)
    return;

  m_energy += amount;
}

void Player::rotate(int amount)
{
  if (m_energy <= 0)
    return;

  decreaseEnergy(ROTATE_COST);
  setRotation(m_rotation + amount);
}

void Player::setEnergy(int energy)
{
  if (energy == m_energy) {
    return;
  }

  m_energy = energy;
}

void Player::accelerate()
{
  if (m_energy <= 0)
    return;

  qreal angle = m_rotation * M_PI * 2.0 / 360.0;
  m_velocityX += cos(angle) * ACCELERATION_FORCE;
  m_velocityY += sin(angle) * ACCELERATION_FORCE;
  decreaseEnergy(ACCELERATION_COST);
}

void Player::doMove()
{
  if (m_energy <= 0) {
    return;
  }

  qreal x = m_position.x();
  qreal y = m_position.y();

  const qreal angle = atan2(y, x);
  const qreal distance = hypot(x, y);

  if (distance < 0.1) {
    m_alive = false;
    return;
  }

  const qreal force = distance / m_energy;
  m_velocityX -= cos(angle) * force;
  m_velocityY -= sin(angle) * force;

  if (m_velocityX > 0.05) m_velocityX = 0.05;
  if (m_velocityY > 0.05) m_velocityY = 0.05;
  if (m_velocityX < -0.05) m_velocityX = -0.05;
  if (m_velocityY < -0.05) m_velocityY = -0.05;

  x += m_velocityX;
  y += m_velocityY;

  if (x > 1.0) { x = -1.0; }
  if (y > 1.0) { y = -1.0; }
  if (x < -1.0) { x = 1.0; }
  if (y < -1.0) { y = 1.0; }

  m_position.setX(x);
  m_position.setY(y);
}

void Player::turnLeft(int steps)
{
  int rotation = m_rotation - steps * ROTATE_AMOUNT;
  setRotation(rotation);
}

void Player::turnRight(int steps)
{
  int rotation = m_rotation + steps * ROTATE_AMOUNT;
  setRotation(rotation);
}
