#ifdef _MSC_VER
#define _CRT_SECURE_NO_WARNINGS
#endif
#define _USE_MATH_DEFINES
#include <stdexcept>
#include <cstdio>
#include <sstream>
#include <map>
#include <vector>
#include "simulator.h"
#include "player.h"
#include "missile.h"
#include "parameters.h"
#include "log.h"

//#define DEBUG 1

Simulator::Simulator(const char *name)
  : m_simulation_name(name)
  , m_log_fp(NULL)
{
#ifdef DEBUG
  std::string filename("gamelogs/gamelog-");
  filename += m_simulation_name;
  filename += "s.log";

  m_log_fp = fopen(filename.c_str(), "a+");
  if (!m_log_fp)
  {
    perror("fopen");
    throw std::runtime_error("Unable to open log file");
  }
#endif
}

Simulator::~Simulator()
{
  if (m_log_fp)
    fclose(m_log_fp);
}

void Simulator::update(std::vector<Player *> &players, std::vector<Missile *> &missiles, long start_tick)
{
  for (Player *p : m_org_players)
  {
    delete p;
  }
  m_org_players.clear();
  m_org_players = players;

  for (Missile *m : m_org_missiles)
  {
    delete m;
  }
  m_org_missiles.clear();
  m_org_missiles = missiles;

  m_start_tick = start_tick;
}

void Simulator::copyPlayers(std::vector<Player *> &new_players)
{
  for (Player *p : m_org_players)
  {
    new_players.push_back(new Player(*p));
  }
}

void Simulator::copyMissiles(std::vector<Missile *> &new_missiles)
{
  for (Missile *m : m_org_missiles)
  {
    new_missiles.push_back(new Missile(*m));
  }
}

void Simulator::gameTick(std::vector<Player *> &m_players, std::vector<Missile *> &m_missiles)
{
  for (Missile *missile : m_missiles)
  {
    missile->doMove();

    if (!missile->isAlive()) {
      continue;
    }

    Player *closest = nullptr;
    qreal closestDX = 0, closestDY = 0;
    for (size_t i=0; i<m_players.size(); i++)
    {
      if (!m_players[i]->isAlive()) {
        continue;
      }

      if (missile->owner() == m_players[i]->id()) {
        continue;
      }

      const qreal dx = m_players[i]->position().x() - missile->position().x();
      const qreal dy = m_players[i]->position().y() - missile->position().y();
      if (hypot(dx, dy) < 0.1) {
        m_players[i]->decreaseEnergy(MISSILE_DAMAGE);
        if (missile->owner() < static_cast<int>(m_players.size()))
        {
          m_players[missile->owner()]->increaseEnergy(MISSILE_DAMAGE);
          m_players[missile->owner()]->addScore(1);
        }
        missile->deleteLater();
        break;
      }

      // For seeking missiles
      if (!closest || hypot(dx, dy) < hypot(closestDX, closestDY))
      {
        closest = m_players[i];
        closestDX = dx;
        closestDY = dy;
        continue;
      }
    }

    if (missile->type() == Missile::Seeking && closest) {
      missile->setRotation(atan2(closestDY, closestDX));
    }
  }

  // Do not randomize players...

  int dead = 0;
  for (Player *player : m_players) {
    if (!player->isAlive()) {
      dead++;
      continue;
    }

    player->doMove();
    player->decreaseEnergy(1);

    QString command = player->command();
    if (command.empty()) {
      continue;
    }

    if (command == "ACCELERATE") {
      player->accelerate();
    } else if (command == "LEFT") {
      player->rotate(-ROTATE_AMOUNT);
    } else if (command == "RIGHT") {
      player->rotate(ROTATE_AMOUNT);
    } else if (command == "MISSILE") {
      player->decreaseEnergy(MISSILE_COST);
      Missile *missile = createMissileBasedOnPlayer(player, Missile::Normal);
      m_missiles.push_back(missile);
    } else if (command == "SEEKING") {
      player->decreaseEnergy(SEEKING_MISSILE_COST);
      Missile *missile = createMissileBasedOnPlayer(player, Missile::Seeking);
      m_missiles.push_back(missile);
    } else if (command == "MINE") {
      player->decreaseEnergy(MINE_COST);
      Missile *missile = createMissileBasedOnPlayer(player, Missile::Mine);
      m_missiles.push_back(missile);
    }
  }

  if (dead > 0 && m_players.size() - dead < 2)
  {
    // Winner winner chicken dinner
    return;
  }

  // We do not need to send status updates...
}

void Simulator::startSimulation(std::string name, std::vector<Player *> &players, std::vector<Missile *> &missiles)
{
  log(m_log_fp, "{\"type\":\"simulation_start\",\"name\":\"%s\"}\n", name.c_str());
  log(m_log_fp, "{\"tickno\":%lu,\"type\":\"tick\"}\n", m_start_tick);
  copyPlayers(players);
  copyMissiles(missiles);
}

void Simulator::endSimulation(std::string name, std::vector<Player *> &players, std::vector<Missile *> &missiles, std::string result)
{
  log(m_log_fp, "{\"type\":\"simulation_end\",\"name\":\"%s\",\"result\":%s}\n", name.c_str(), result.c_str());
  for (Player *p : players)
    delete p;
  players.clear();

  for (Missile *m : missiles)
    delete m;
  missiles.clear();
}

void Simulator::setEnemyCommand(std::string command, int my_id, std::vector<Player *> &players)
{
  for (Player *p : players)
  {
    if (p->id() == my_id)
      continue;
    p->setCommand(command);
  }
}

bool Simulator::fakeSimulateMeShooting(int my_id, Missile::Type type, int num_ticks)
{
  std::vector<Player *> players;
  std::vector<Missile *> missiles;
  startSimulation(__FUNCTION__, players, missiles);

  Player *me = getPlayerWithId(my_id, players);
  Missile *my_missile = nullptr;
  bool hit = false;

  if (!me)
    goto fakeSimulateMeShooting_end;

  logPlayers(my_id, players);
  logMissiles(missiles);

  for (int i = 0; i < num_ticks; ++i)
  {
    gameTick(players, missiles);
    logPlayers(my_id, players);
    logMissiles(missiles);

    if (!my_missile)
      my_missile = createMissileBasedOnPlayer(me, type);
    my_missile->doMove();

    // Check if any enemies are hit by this missile
    for (Player *p : players)
    {
      if (p->id() == my_id || !p->isAlive())
        continue;

      const qreal dx = p->position().x() - my_missile->position().x();
      const qreal dy = p->position().y() - my_missile->position().y();

      if (hypot(dx, dy) < 0.11) // 0.1 is a hit, this will give false hits.
      {
        log(m_log_fp, "{\"y\":%.18f,\"x\":%.18f,\"type\":\"virtualMissile\",\"rot\":%d,\"status\":\"HIT\"}\n", my_missile->position().y(), my_missile->position().x(), my_missile->rotation());
        hit = true;
        goto fakeSimulateMeShooting_end;
      }
    }
    log(m_log_fp, "{\"y\":%.18f,\"x\":%.18f,\"type\":\"virtualMissile\",\"rot\":%d}\n", my_missile->position().y(), my_missile->position().x(), my_missile->rotation());
  }

fakeSimulateMeShooting_end:
  delete my_missile;
  std::string result = "{\"hit\":";
  result += (hit ? "true" : "false");
  result += "}";

  endSimulation(__FUNCTION__, players, missiles, result);

  return hit;
}

bool Simulator::simulateMeShootingAndTheyBoost(int my_id, Missile::Type type, int num_ticks)
{
  std::vector<Player *> players;
  std::vector<Missile *> missiles;
  startSimulation(__FUNCTION__, players, missiles);

  Player *me = getPlayerWithId(my_id, players);
  Missile *my_missile = nullptr;
  bool hit = false;

  if (!me)
    goto simulateMeShootingAndTheyBoost_end;

  logPlayers(my_id, players);
  logMissiles(missiles);

  for (int i = 0; i < num_ticks; ++i)
  {
    setEnemyCommand("ACCELERATE", my_id, players);
    gameTick(players, missiles);
    logPlayers(my_id, players);
    logMissiles(missiles);

    if (!my_missile)
    {
      my_missile = createMissileBasedOnPlayer(me, type);
    }

    my_missile->doMove();

    // Check if any enemies are hit by this missile
    for (Player *p : players)
    {
      if (p->id() == my_id || !p->isAlive())
        continue;

      const qreal dx = p->position().x() - my_missile->position().x();
      const qreal dy = p->position().y() - my_missile->position().y();

      if (hypot(dx, dy) < 0.1)
      {
        log(m_log_fp, "{\"y\":%.18f,\"x\":%.18f,\"type\":\"virtualMissile\",\"rot\":%d,\"status\":\"HIT\"}\n", my_missile->position().y(), my_missile->position().x(), my_missile->rotation());
        hit = true;
        goto simulateMeShootingAndTheyBoost_end;
      }
    }
    log(m_log_fp, "{\"y\":%.18f,\"x\":%.18f,\"type\":\"virtualMissile\",\"rot\":%d}\n", my_missile->position().y(), my_missile->position().x(), my_missile->rotation());
  }

simulateMeShootingAndTheyBoost_end:
  std::string result = "{\"hit\":";
  result += (hit ? "true" : "false");
  result += "}";

  endSimulation(__FUNCTION__, players, missiles, result);

  return hit;
}

std::map<int, int> Simulator::simulateAllPlayersWithMoves(int my_id, int num_ticks, const std::map<int, std::vector<std::string> > &moves)
{
  std::vector<Player *> players;
  std::vector<Missile *> missiles;
  startSimulation(__FUNCTION__, players, missiles);

  logPlayers(my_id, players);
  logMissiles(missiles);

  for (int i = 0; i < num_ticks; ++i)
  {
    log(m_log_fp, "{\"tickno\":%lu, \"type\":\"virtualtick\"}\n", m_start_tick+i+1);

    for (auto pair : moves)
    {
      int player_id = pair.first;
      std::string move = (pair.second.size() < static_cast<unsigned>(i)) ? "NONE" : pair.second[i];
      if (move != "NONE")
        players[player_id]->setCommand(move);
    }
    gameTick(players, missiles);
    logPlayers(my_id, players);
    logMissiles(missiles);
  }

  std::map<int, int> result_map;
  std::ostringstream result_stream;
  bool first(true);
  result_stream << "{[";
  for (Player *p : players)
  {
    if (first)
      first = false;
    else
      result_stream << ",";

    result_map.emplace(p->id(), p->energy());
    result_stream << "{\"id\":" << p->id() << ", \"energy\":" << p->energy() << "}";
  }
  result_stream << "]}";

  endSimulation(__FUNCTION__, players, missiles, result_stream.str());

  return result_map;
}

int Simulator::simulatePlayerMoves(int player_id, const std::vector<std::string> &moves)
{
  std::vector<Player *> players;
  std::vector<Missile *> missiles;
  startSimulation(__FUNCTION__, players, missiles);

  Player *player = getPlayerWithId(player_id, players);

  if (!player)
    return 0;

  logPlayers(-1, players);
  logMissiles(missiles);

  for (size_t i = 0; i < moves.size(); ++i)
  {
    log(m_log_fp, "{\"tickno\":%lu, \"type\":\"virtualtick\"}\n", m_start_tick+i+1);
    if (moves[i] != "NONE")
      player->setCommand(moves[i]);
    gameTick(players, missiles);
    logPlayers(-1, players);
    logMissiles(missiles);
    if(!player->isAlive()){
      endSimulation(__FUNCTION__, players, missiles, "{\"dies_in\": \"number_here\"}");
      return -1000+i;
    }
  }

  int energy = player->energy();

  std::string result = "{\"energy\":";
  result += energy;
  result += "}";

  endSimulation(__FUNCTION__, players, missiles, result);

  return energy;
}

bool Simulator::simulateMeShooting(int my_id, Missile::Type type, int num_ticks, bool *hit, int max_missile_age)
{
  std::vector<Player *> players;
  std::vector<Missile *> missiles;
  startSimulation(__FUNCTION__, players, missiles);

  Player *me = getPlayerWithId(my_id, players);
  Missile *my_missiles[2][19] = {};

  if (!me)
    return false;

  logPlayers(my_id, players);
  logMissiles(missiles);

  for (int i = 0; i < num_ticks; ++i)
  {
    log(m_log_fp, "{\"tickno\":%lu, \"type\":\"virtualtick\"}\n", m_start_tick+i+1);
    gameTick(players, missiles);
    logPlayers(my_id, players);
    logMissiles(missiles);

    if(i == 0) {
      my_missiles[0][0] = Simulator::createMissileBasedOnPlayer(me, type);
    } else if(i < 19) {
      me->turnLeft(i);
      my_missiles[LEFT][i] = Simulator::createMissileBasedOnPlayer(me, type);
      me->turnRight(i*2);
      my_missiles[RIGHT][i] = Simulator::createMissileBasedOnPlayer(me, type);
      me->turnLeft(i);
    }
    for(int direction = 0; direction < 2; direction++){
      for(int a = 0; a < 19; a++){
        if(my_missiles[direction][a] != NULL) {
          logVirtualMissile(my_missiles[direction][a]);
          my_missiles[direction][a]->doMove();
          if(!(my_missiles[direction][a]->isAlive()) ||
              (my_missiles[direction][a]->getAge() > max_missile_age)){
            delete my_missiles[direction][a];
            my_missiles[direction][a] = NULL;
          }
        }
      }
    }

    // Check if any enemies are hit by these missiles
    for (Player *p : players)
    {
      if (p->id() == my_id || !p->isAlive())
        continue;
      for(int direction = 0; direction < 2; direction++){
        for(int a = 0; a < 19; a++){
          if(my_missiles[direction][a] != NULL) {
            const qreal dx = p->position().x() - my_missiles[direction][a]->position().x();
            const qreal dy = p->position().y() - my_missiles[direction][a]->position().y();
            if(hypot(dx, dy) < 0.1){
              hit[direction*19 + a] = true;
            }
          }
        }
      }
    }
  }
  
  for(int direction = 0; direction < 2; direction++){
    for(int a = 0; a < 19; a++){
      if(my_missiles[direction][a] != NULL) {
        delete my_missiles[direction][a];
      }
    }
  }
  

  endSimulation(__FUNCTION__, players, missiles, "");

  return 0;
}

void Simulator::logPlayers(int my_id, std::vector<Player *> &players)
{
  for (Player *p : players)
  {
    if (-1 == my_id)
    {
      log(m_log_fp, "{\"y\":%.18f,\"x\":%.18f,\"type\":\"player\",\"rot\":%d}\n", p->position().y(), p->position().x(), p->rotation());
    }
    else if (p->id() == my_id)
    {
      log(m_log_fp, "{\"y\":%.18f,\"x\":%.18f,\"type\":\"me\",\"rot\":%d}\n", p->position().y(), p->position().x(), p->rotation());
    }
    else
    {
      log(m_log_fp, "{\"y\":%.18f,\"x\":%.18f,\"type\":\"enemy\",\"rot\":%d}\n", p->position().y(), p->position().x(), p->rotation());
    }
  }
}

void Simulator::logMissiles(std::vector<Missile *> &missiles)
{
  for (Missile *m : missiles)
  {
    log(m_log_fp, "{\"y\":%.18f,\"x\":%.18f,\"type\":\"missile\",\"rot\":%d}\n", m->position().y(), m->position().x(), m->rotation());
  }
}

void Simulator::logVirtualMissile(Missile *m)
{
  log(m_log_fp, "{\"y\":%.18f,\"x\":%.18f,\"type\":\"virtualMissile\",\"rot\":%d}\n", m->position().y(), m->position().x(), m->rotation());
}

Player * Simulator::getPlayerWithId(int id, std::vector<Player *> &players)
{
  for (Player *p : players)
  {
    if (p->id() == id)
      return p;
  }
  return nullptr;
}

Missile * Simulator::createMissileBasedOnPlayer(Player *player, Missile::Type type)
{
  return new Missile(type, player->position(), player->rotation(), player->id());
}

