#version 430

in vec2 fragCoord;
out vec4 fragColor;

uniform float iGlobalTime;
uniform vec2 iResolution;
uniform vec3 iMouse;
uniform sampler2D iChannel0;

#define compactKeys  10
uniform float midi[10 * compactKeys];

vec4 texCube( sampler2D sam, in vec3 p, in vec3 n, in float k )
{
	vec4 x = texture2D( sam, p.yz );
	vec4 y = texture2D( sam, p.zx );
	vec4 z = texture2D( sam, p.xy );
    vec3 w = pow( abs(n), vec3(k) );
	return (x*w.x + y*w.y + z*w.z) / (w.x+w.y+w.z);
}

float smin( float a, float b)
{
	float k = 0.1;
    float h = clamp( 0.5+0.5*(b-a)/k, 0.0, 1.0 );
    return mix( b, a, h ) - k*h*(1.0-h);
}

float sdBox( vec3 p, vec3 b )
{
  vec3 d = abs(p) - b;
  return min(max(d.x,max(d.y,d.z)),0.0) + length(max(d,0.0));
}

float sdCylinder( vec3 p, vec3 c )
{
  return length(p.xz-c.xy)-c.z;
}

float sdCappedCylinder( vec3 p, vec2 h )
{
  vec2 d = abs(vec2(length(p.xz),p.y)) - h;
  return min(max(d.x,d.y),0.0) + length(max(d,0.0));
}

float udRoundBox( vec3 p, vec3 b, float r )
{
  return length(max(abs(p)-b,0.0))-r;
}

#define TRANSITION_TIME 30.0

vec2 map(vec3 p)
{	
	vec3 q = mod(p, 6.0) - 3.0;
	q.y = p.y;
	float robotsMinDis = udRoundBox(q - vec3(0.0, 1.0, 0.0), vec3(1.0, 1.5, 0.5), 0.3); //body
	robotsMinDis = min(robotsMinDis, udRoundBox(q - vec3(0.0, 3.5, 0.0), vec3(0.4, 0.4, 0.4), 0.3)); //head
	robotsMinDis = min(robotsMinDis, udRoundBox(q - vec3(0.0, 1.0, 0.0), vec3(1.5, 1.3, 0.2), 0.2)); // arms
	robotsMinDis = min(robotsMinDis, udRoundBox(q - vec3(-0.6, -1.0, 0.0), vec3(0.3, 1.2, 0.3), 0.2)); // leg
	robotsMinDis = min(robotsMinDis, udRoundBox(q - vec3(0.6, -1.0, 0.0), vec3(0.3, 1.2, 0.3), 0.2)); // leg
	
	vec3 q2 = mod(p, 80.0) - 40.0;
	vec3 part = floor(p / 80.0);
	
	float boxHeight = 3.0;
	if (iGlobalTime > TRANSITION_TIME) {
	
		int indexChannel = int(mod(part.x, 10.0));
		int indexKey = int(mod(part.z, 10.0));
		boxHeight += 30.0 * midi[indexChannel * compactKeys + indexKey % compactKeys];
	}
	float boxMinDis = udRoundBox(vec3(q2.x, p.y, q2.z) - vec3(0.0, boxHeight * 0.5, 0.0), vec3(15.0, boxHeight, 15.0), 10.0);
	float minDis = mix(robotsMinDis, boxMinDis, smoothstep(TRANSITION_TIME - 2.5, TRANSITION_TIME + 0.5, iGlobalTime));

	
	
	q2.y = p.y;
	minDis = max(minDis, sdBox(q2 - vec3(1.0, 0.0, 0.0), vec3(34.0, 9999999.0, 34.0)));
	return vec2(minDis, part * 2.0);
}

vec3 getNormal(vec3 p)
{
	vec3 normal;
    vec3 ep = vec3(0.01, 0, 0);
    normal.x = map(p + ep.xyz).x - map(p - ep.xyz).x;
    normal.y = map(p + ep.yxz).x - map(p - ep.yxz).x;
    normal.z = map(p + ep.yzx).x - map(p - ep.yzx).x;
    return normalize(normal);

}

float raymarch(vec3 ro, vec3 rd, inout vec3 finalPos) 
{
	float t = 0.0;
    const int maxIter = 300;
    
    float part = -1.0;
    
    vec3 p = vec3(-1.0, -1.0, -1.0);
    for (int i = 0; i < maxIter; i++) {
        p = ro + rd * t;

    	vec2 dm = map(p);
        if (dm.x < 0.01) {
        	part = dm.y;
        	break;
        }
        t += dm.x;
    }
    finalPos = p;
    return part;
}

float specular(vec3 normal, vec3 light, vec3 viewdir, float s)
{
	float nrm = (s + 8.0) / (3.1415 * 8.0);
	float k = max(0.0, dot(viewdir, reflect(light, normal)));
    return  pow(k, s);
}

void mainImage(out vec4 fragColor, in vec2 fragCoord)
{
	float u = (fragCoord.x / iResolution.x) * 2.0 - 1.0;
    float v = ((fragCoord.y / iResolution.y) * 2.0 - 1.0) * (iResolution.y/iResolution.x);
    
    float time = min(iGlobalTime, TRANSITION_TIME);
    
 	vec3 eyeClose = vec3(15.0, 3.5, 7.9 - time * 0.3);//vec3(cos(iGlobalTime * 0.5) * 7.0, sin(iGlobalTime * 0.5) * 7.0, -20.0);
    vec3 tarClose = eyeClose + vec3(0.0, 0.0, 1.0);
    
    vec3 eyeFar = vec3(15.0 + time * 10.0, 3.5 + time * 15.0, 7.7 + time * 30.0);
    vec3 tarFar = eyeFar + vec3(cos(time * 0.3), -1.0, sin(time * 0.3));
    
    vec3 eye = mix(eyeClose, eyeFar, max(0.0, time * 0.1 - 1.0));
    vec3 tar = mix(tarClose, tarFar, max(0.0, time * 0.1 - 1.0));
    vec3 dir = normalize(tar - eye);
	vec3 right = normalize(cross(vec3(0, 1, 0), dir)); 
 	vec3 up = cross(dir, right);
    
	vec3 ro = eye;	
    vec3 rd = normalize(dir + right*u + up*v);
    
    vec3 light = vec3(0.0, 20.0, -20.0);
    
    vec3 finalPos = vec3(-1.0, -1.0, -1.0);
    float part = raymarch(ro, rd, finalPos);
   	vec3 color = vec3(0.5, 0.0, 0.5);
    if (part != -1.0) {
		vec3 normal = getNormal(finalPos);
		vec3 invLight = normalize(light - finalPos);
	  	float diffuse = max(0.0, dot(invLight, normal));
    	color = vec3(0.3 + 0.2 * sin(part), 0.3 + 0.2 * sin(part * 0.5 + 1.0), 0.3 + 0.2 * sin(part * 0.25 + 2.0));
    	//color = texCube(iChannel0, finalPos, normal, 1.0).xyz;
    	//color = vec3(0.4);
    	color = 0.7 * color * (1.0 + diffuse);
	    color += specular(normal, -invLight, normalize(eye - finalPos), 70.0);
    } else if (finalPos.y < 0){
    	color = vec3(0.8, 0.6, 0.3);
    } else {
    	color = vec3(0.5, 0.7, 0.9);
    }
 	
 	color = mix(color, vec3(0.3), 1.0 - exp(- length(eye - finalPos)*0.0005));
 
    fragColor = vec4(color, 1.0);
}

void main()
{
	mainImage(fragColor, fragCoord);
}
