#!/usr/bin/python
from exceptions import ValueError

from math import pi, sin, cos, tanh, sqrt, exp
from operator import add
from random import random

## basic oscilators, phase goes from 0 to 1
def sawtooth(phase):
	if phase < 0 or phase > 1:
		raise ValueError('phase out of range')
	if phase < 0.5:
		return phase * 2
	else:
		return phase * 2 - 2

class Pulse:

	def __init__(self):
		self.previousPhase = 0
		self.previousValue = 1.0

	def step (self, phase, pulseWidth):

		if phase < 0 or phase > 1:
			raise ValueError('phase out of range')
		if pulseWidth < 0 or pulseWidth > 1:
			raise ValueError('pulse width out of range')		
		if phase < pulseWidth:
			if self.previousValue == 1.0:
				self.previousPhase = phase
				return 1.0
			else:			
				result = 2 * (1.0 - self.previousPhase) / (1.0 + phase - self.previousPhase) - 1.0
				self.previousPhase = phase
				self.previousValue = 1.0
				return result
		else:
			if self.previousValue == -1.0:
				
				self.previousPhase = phase
				return -1.0
			else:			
				result = 1.0 - 2 * (phase - pulseWidth) / (phase - self.previousPhase)
				self.previousPhase = phase
				self.previousValue = -1.0
				return result

def floatTo8Bit (sampleData):

	result = []

	for sample in sampleData:
		sample = min(127,max(int(sample*128),-128)) # clip the output
		sample = (sample if sample >= 0 else sample+256) # 2's complement
		result.append (sample)

	return result

def normalize (sampleData):

	maximum = 0
	result = []

	for sample in sampleData:
		if abs(sample) > maximum:
			maximum = abs(sample)

	for sample in sampleData:
		result.append (sample / maximum)
  
	return result

def fixDc (sampleData):

	sum = 0.0
	result = []

	for sample in sampleData:
		sum += sample

	average = sum / len(sampleData)

	for sample in sampleData:
		result.append (sample - average)

	return result

def tanhDrive (sampleData, drive):

	result = []

	for sample in sampleData:
		result.append(tanh(sample*drive))

	return result

## simple SVF filter
class SVF:

	def __init__(self):

		self.fs = 44100.0
		self.oversample = 1

		self.low = 0.0
		self.band = 0.0
		self.high = 0.0

		self.q = 1.0

	# set cutoff in Hz
	def setCutoffFrequency(self, cutoff):
		self.f = 2.0 * sin (pi * cutoff / (self.fs * self.oversample))

	# set cutoff as ratio of fs
	def setCutoffRatio(self, cutoff):
		self.f = 2.0 * sin (pi * cutoff / self.oversample)

	def step(self, input):

		for i in range (self.oversample):		
			self.high = input - self.band * self.q - self.low
			self.band = self.band + self.f * self.high
			self.low = self.low + self.f * self.band

## biquad resonator (2 pole filter)
## used to model the bridged-T circuits in the TR-808
class Resonator:

	def __init__(self):
		self.y0 = 0.0
		self.y1 = 0.0
		self.y2 = 0.0

		self.Q = 1.0

	def setQ(self, Q):
		self.Q = Q
		self.calcCoeffs()

	def setFreqRatio(self, freqRatio):
		self.freqRatio = freqRatio
		self.calcCoeffs()

	def calcCoeffs(self):
		R = 1.0 - pi * self.freqRatio / self.Q

		self.cy1 = 4 * R * R * cos (2.0*pi*self.freqRatio) / (1.0 + R*R)
		self.cy2 = - R * R

		self.cx0 = (1.0 - R * R) * 0.5

	def step(self, input):

		self.y0 = input + self.cy1 * self.y1 + self.cy2 * self.y2

		self.y2 = self.y1
		self.y1 = self.y0

		return (self.y0 - self.cy2) * self.cx0

## exponential decay
class expDecay:

	def __init__(self):
		self.a = 1.0
		self.oneMinusGamma = 0.9
		self.fs = 44100.0
		self.amp = 1.0

	def setTime(self, tau):
		self.oneMinusGamma = exp(-1.0/(tau*self.fs))

	def step(self):
		self.a *= self.oneMinusGamma
		return self.a * self.amp

	def reset(self):
		self.a = 1.0

## 1: pluck
foPluck = open('pluck.raw', 'wb')

pluckData = [0,0]

osc1Period = 32
osc2Period = osc1Period / 2.0
pw = .25

osc1Cycles = 16

filter = SVF()
filter.oversample = 4
filter.q = .25

env = expDecay()

env.fs = 16000
env.setTime(.015)
env.amp = .4

osc = Pulse()

for i in range(osc1Period * osc1Cycles):

	mix = float(i)/ (osc1Period * osc1Cycles)

	filter.setCutoffRatio(0.085 + env.step())
	filter.q = .25 + (.8 * sqrt(mix))

	osc1Phase = float(i % osc1Period) / osc1Period
	osc2Phase = float(i % osc2Period) / osc2Period

	mixed = (mix)* osc.step (osc1Phase, pw) + (1-mix)* osc.step (osc2Phase, pw)

	filter.step(mixed)

	pluckData.append(filter.low)

pluckData = normalize(pluckData)
pluckData = floatTo8Bit(pluckData)

foPluck.write(bytearray(pluckData))

numFilterSteps = 6

## 2: fifths
for step in range(numFilterSteps):

	foFifth = open('fifth' + str(step) + '.raw', 'wb')

	fifthData = [0,0]

	osc1Period = 64
	osc2Period = (2.0 / 3.0) * osc1Period # perfect fifth

	osc3Period = osc1Period / 2
	osc4Period = osc2Period / 2

	#osc1Cycles = 4
	osc1Cycles = 40

	filter = SVF()
	filter.oversample = 4
	filter.q = .9

	#baseCutoffRatio = .1
	baseCutoffRatio = .03

	env = expDecay()

	env.fs = 16000
	# env.setTime(.04)
	env.setTime(.0175 * (.5*step+1))
	env.amp = .45

	octaveAmp = 0
	#octaveAmp = .75

	for i in range(osc1Period * osc1Cycles):

		osc1Phase = float(i % osc1Period) / osc1Period
		osc2Phase = float(i % osc2Period) / osc2Period

		osc3Phase = float(i % osc3Period) / osc3Period
		osc4Phase = float(i % osc4Period) / osc4Period

		mixed = (sawtooth(osc1Phase) + sawtooth(osc2Phase) + (sawtooth(osc4Phase) + sawtooth(osc3Phase))*octaveAmp) / 2

		filter.setCutoffRatio(baseCutoffRatio + env.step())
		filter.step(mixed)

		fifthData.append(filter.low)

	fifthData = normalize(fifthData)
	fifthData = floatTo8Bit(fifthData)

	foFifth.write(bytearray(fifthData))

## 3: fourths
for step in range(numFilterSteps):
	foFourth = open('fourth' + str(step) + '.raw', 'wb')

	fourthData = [0,0]

	osc1Period = 64
	osc2Period = (3.0 / 4.0) * osc1Period # perfect fourth

	osc3Period = osc1Period / 2
	osc4Period = osc2Period / 2

	osc1Cycles = 60

	env = expDecay()

	env.fs = 16000
	# env.setTime(.04)
	env.setTime(.0175 * (.5*step+1))
	env.amp = .45

	for i in range(osc1Period * osc1Cycles):

		osc1Phase = float(i % osc1Period) / osc1Period
		osc2Phase = float(i % osc2Period) / osc2Period

		osc3Phase = float(i % osc3Period) / osc3Period
		osc4Phase = float(i % osc4Period) / osc4Period

		mixed = (sawtooth(osc1Phase) + sawtooth(osc2Phase) + (sawtooth(osc4Phase) + sawtooth(osc3Phase))*octaveAmp) / 2

		filter.setCutoffRatio(baseCutoffRatio + env.step())
		filter.step(mixed)

		fourthData.append(filter.low)

	fourthData = normalize(fourthData)
	fourthData = floatTo8Bit(fourthData)

	foFourth.write(bytearray(fourthData))

## 3.5: tritone
#foTritone = open('tritone.raw', 'wb')
#
#tritoneData = [0,0]
#
#osc1Period = 64
#osc2Period = (5.0 / 7.0) * osc1Period # perfect fourth
#
#osc3Period = osc1Period / 2
#osc4Period = osc2Period / 2
#
#osc1Cycles = 40
#
#env.reset()
#
#for i in range(osc1Period * osc1Cycles):
#
#	osc1Phase = float(i % osc1Period) / osc1Period
#	osc2Phase = float(i % osc2Period) / osc2Period
#
#	osc3Phase = float(i % osc3Period) / osc3Period
#	osc4Phase = float(i % osc4Period) / osc4Period
#
#	mixed = (sawtooth(osc1Phase) + sawtooth(osc2Phase) + (sawtooth(osc4Phase) + sawtooth(osc3Phase))*octaveAmp) / 2
#
#	filter.setCutoffRatio(baseCutoffRatio + env.step())
#	filter.step(mixed)
#
#	tritoneData.append(filter.low)

#tritoneData = normalize(tritoneData)
#tritoneData = floatTo8Bit(tritoneData)

#foTritone.write(bytearray(tritoneData))

## 3.75: fifths below
# foFifth = open('fifth.raw', 'wb')
foFourthBelow = open('fourthBelow.raw','wb')

fourthBelowData = [0,0]

osc1Period = 64
osc2Period = (4.0 / 3.0) * osc1Period # perfect fourth down

osc3Period = osc1Period / 2
osc4Period = osc2Period / 2

#osc1Cycles = 4
osc1Cycles = 40

filter = SVF()
filter.oversample = 4
filter.q = .9

#baseCutoffRatio = .015
baseCutoffRatio = .1

env = expDecay()

env.fs = 16000
env.setTime(.04)
env.amp = .45

octaveAmp = 0
#octaveAmp = .75

for i in range(osc1Period * osc1Cycles):

	osc1Phase = float(i % osc1Period) / osc1Period
	osc2Phase = float(i % osc2Period) / osc2Period

	osc3Phase = float(i % osc3Period) / osc3Period
	osc4Phase = float(i % osc4Period) / osc4Period

	mixed = (sawtooth(osc1Phase) + sawtooth(osc2Phase) + (sawtooth(osc4Phase) + sawtooth(osc3Phase))*octaveAmp) * .75 # 1.25 ?

	filter.setCutoffRatio(baseCutoffRatio + env.step())
	filter.step(mixed)

	fourthBelowData.append(filter.low)

# fourthBelowData = normalize(fourthBelowData)
fourthBelowOutputData = floatTo8Bit(fourthBelowData) # keep the floats for mixing with the chords later on ...

foFourthBelow.write(bytearray(fourthBelowOutputData))

## 4: the lead
foLead = open('lead.raw', 'wb')

leadData = [0,0]

oscPeriod = 32
attackPeriod = oscPeriod / 2

stretch = 16

osc = Pulse()

# the initial attack part
for i in range (16*attackPeriod):

	phase = float(i % attackPeriod) / attackPeriod
	leadData.append(osc.step(phase, 0.25))

# ascending PWM
for cycle in range(oscPeriod*stretch/4,(oscPeriod-2)*stretch):

	for i in range(oscPeriod):

		phase = float(i % oscPeriod) / oscPeriod
		leadData.append (osc.step(phase,float(cycle)/(oscPeriod*stretch)))

# go back to beginning for clean loop
for cycle in reversed(range(oscPeriod*stretch/4,(oscPeriod-2)*stretch)):

	for i in range(oscPeriod):

		phase = float(i % oscPeriod) / oscPeriod
		leadData.append (osc.step(phase,float(cycle)/(oscPeriod*stretch)))

leadData = normalize(leadData)
leadData = floatTo8Bit(leadData)

foLead.write(bytearray(leadData))

## 4.5: min7 chord
foMin7 = open('min7.raw', 'wb')

min7Data = [0,0]

basePeriod = 64.0

osc1Period = basePeriod
osc2Period = basePeriod / pow (2.0, 3.0 / 12)
osc3Period = basePeriod / pow (2.0, 7.0 / 12)
osc4Period = basePeriod / pow (2.0, 10.0 / 12)

attack1Period = osc1Period / 2
attack2Period = osc2Period / 2
attack3Period = osc3Period / 2
attack4Period = osc4Period / 2

stretch = 8

osc1 = Pulse()
osc2 = Pulse()
osc3 = Pulse()
osc4 = Pulse()

# the initial attack part
for i in range (stretch*int(attack1Period)):

	phase1 = float(i % attack1Period) / attack1Period
	phase2 = float(i % attack2Period) / attack2Period
	phase3 = float(i % attack3Period) / attack3Period
	phase4 = float(i % attack4Period) / attack4Period
	
	min7Data.append(
		(
			osc1.step(phase1, 0.25) +
 			osc2.step(phase2, 0.25) +
 			osc3.step(phase3, 0.25) +
 			osc4.step(phase4, 0.25)
 		) /4       
	)

# ascending PWM

for i in range(3*int(osc1Period*osc1Period)*stretch/4):

	phase1 = float(i % osc1Period) / osc1Period
	phase2 = float(i % osc2Period) / osc2Period
	phase3 = float(i % osc3Period) / osc3Period
	phase4 = float(i % osc4Period) / osc4Period

	min7Data.append (
		(
		osc1.step(phase1,float((i-osc1Period*stretch)/osc1Period)/(osc1Period*stretch) + .25) +
		osc2.step(phase2,float((i-osc1Period*stretch)/osc1Period)/(osc1Period*stretch) + .25) +
		osc3.step(phase3,float((i-osc1Period*stretch)/osc1Period)/(osc1Period*stretch) + .25) +
		osc4.step(phase4,float((i-osc1Period*stretch)/osc1Period)/(osc1Period*stretch) + .25) 
		) / 4
	)

# go back to beginning for clean loop
for i in range(3*int(osc1Period*osc1Period)*stretch/4):

	phase1 = float(i % osc1Period) / osc1Period
	phase2 = float(i % osc2Period) / osc2Period
	phase3 = float(i % osc3Period) / osc3Period
	phase4 = float(i % osc4Period) / osc4Period

	min7Data.append (
		(
		osc1.step(phase1,1.0 - float((i+osc1Period*stretch)/osc1Period)/(osc1Period*stretch)) +
		osc2.step(phase2,1.0 - float((i+osc1Period*stretch)/osc1Period)/(osc1Period*stretch)) +
		osc3.step(phase3,1.0 - float((i+osc1Period*stretch)/osc1Period)/(osc1Period*stretch)) +
		osc4.step(phase4,1.0 - float((i+osc1Period*stretch)/osc1Period)/(osc1Period*stretch))  
		) / 4

	)

# pad the fourth sample to the length of the chord if needed
while len(min7Data) > len(fourthBelowData):
	fourthBelowData = fourthBelowData + fourthBelowData[-256:]

length = min (len(min7Data),len(fourthBelowData))

min7Data = normalize(min7Data)
min7Data = map(add,min7Data[:length],fourthBelowData[:length])
min7Data = normalize(min7Data)

drive = 1.5

min7Data = tanhDrive(min7Data, drive)

min7Data = normalize(min7Data)

min7Data = floatTo8Bit(min7Data)

foMin7.write(bytearray(min7Data))

## 4.6: maj7 chord
foMaj7 = open('maj7.raw', 'wb')

maj7Data = [0,0]

basePeriod = 64.0

osc1Period = basePeriod
osc2Period = basePeriod / pow (2.0, 4.0 / 12)
osc3Period = basePeriod / pow (2.0, 7.0 / 12)
osc4Period = basePeriod / pow (2.0, 11.0 / 12)

attack1Period = osc1Period / 2
attack2Period = osc2Period / 2
attack3Period = osc3Period / 2
attack4Period = osc4Period / 2

stretch = 8

osc1 = Pulse()
osc2 = Pulse()
osc3 = Pulse()
osc4 = Pulse()

# the initial attack part
for i in range (stretch*int(attack1Period)):

	phase1 = float(i % attack1Period) / attack1Period
	phase2 = float(i % attack2Period) / attack2Period
	phase3 = float(i % attack3Period) / attack3Period
	phase4 = float(i % attack4Period) / attack4Period
	
	maj7Data.append(
		(
			osc1.step(phase1, 0.25) +
 			osc2.step(phase2, 0.25) +
 			osc3.step(phase3, 0.25) +
 			osc4.step(phase4, 0.25)
 		) /4       
	)

# ascending PWM

for i in range(3*int(osc1Period*osc1Period)*stretch/4):

	phase1 = float(i % osc1Period) / osc1Period
	phase2 = float(i % osc2Period) / osc2Period
	phase3 = float(i % osc3Period) / osc3Period
	phase4 = float(i % osc4Period) / osc4Period

	maj7Data.append (
		(
		osc1.step(phase1,float((i-osc1Period*stretch)/osc1Period)/(osc1Period*stretch) + .25) +
		osc2.step(phase2,float((i-osc1Period*stretch)/osc1Period)/(osc1Period*stretch) + .25) +
		osc3.step(phase3,float((i-osc1Period*stretch)/osc1Period)/(osc1Period*stretch) + .25) +
		osc4.step(phase4,float((i-osc1Period*stretch)/osc1Period)/(osc1Period*stretch) + .25) 
		) / 4
	)

# go back to beginning for clean loop

for i in range(3*int(osc1Period*osc1Period)*stretch/4):

	phase1 = float(i % osc1Period) / osc1Period
	phase2 = float(i % osc2Period) / osc2Period
	phase3 = float(i % osc3Period) / osc3Period
	phase4 = float(i % osc4Period) / osc4Period

	maj7Data.append (
		(
		osc1.step(phase1,1.0 - float((i+osc1Period*stretch)/osc1Period)/(osc1Period*stretch)) +
		osc2.step(phase2,1.0 - float((i+osc1Period*stretch)/osc1Period)/(osc1Period*stretch)) +
		osc3.step(phase3,1.0 - float((i+osc1Period*stretch)/osc1Period)/(osc1Period*stretch)) +
		osc4.step(phase4,1.0 - float((i+osc1Period*stretch)/osc1Period)/(osc1Period*stretch))  
		) / 4

	)
maj7Data = normalize(maj7Data)

while len(min7Data) > len(fourthBelowData):
	fourthBelowData = fourthBelowData + fourthBelowData[-256:]

length = min(len(maj7Data),len(fourthBelowData))
maj7Data = map(add,maj7Data[:length],fourthBelowData[:length])
maj7Data = normalize(maj7Data)

maj7Data = tanhDrive(maj7Data, drive)

maj7Data = normalize(maj7Data)

maj7Data = floatTo8Bit(maj7Data)

foMaj7.write(bytearray(maj7Data))

## 4.7: dominant7 chord
foDominant7 = open('dominant7.raw', 'wb')

dominant7Data = [0,0]

basePeriod = 64.0

osc1Period = basePeriod
osc2Period = basePeriod / pow (2.0, 4.0 / 12)
osc3Period = basePeriod / pow (2.0, 7.0 / 12)
osc4Period = basePeriod / pow (2.0, 10.0 / 12)

attack1Period = osc1Period / 2
attack2Period = osc2Period / 2
attack3Period = osc3Period / 2
attack4Period = osc4Period / 2

stretch = 8

osc1 = Pulse()
osc2 = Pulse()
osc3 = Pulse()
osc4 = Pulse()

# the initial attack part
for i in range (stretch*int(attack1Period)):

	phase1 = float(i % attack1Period) / attack1Period
	phase2 = float(i % attack2Period) / attack2Period
	phase3 = float(i % attack3Period) / attack3Period
	phase4 = float(i % attack4Period) / attack4Period
	
	dominant7Data.append(
		(
			osc1.step(phase1, 0.25) +
 			osc2.step(phase2, 0.25) +
 			osc3.step(phase3, 0.25) +
 			osc4.step(phase4, 0.25)
 		) /4       
	)

# ascending PWM

for i in range(3*int(osc1Period*osc1Period)*stretch/4):

	phase1 = float(i % osc1Period) / osc1Period
	phase2 = float(i % osc2Period) / osc2Period
	phase3 = float(i % osc3Period) / osc3Period
	phase4 = float(i % osc4Period) / osc4Period

	dominant7Data.append (
		(
		osc1.step(phase1,float((i-osc1Period*stretch)/osc1Period)/(osc1Period*stretch) + .25) +
		osc2.step(phase2,float((i-osc1Period*stretch)/osc1Period)/(osc1Period*stretch) + .25) +
		osc3.step(phase3,float((i-osc1Period*stretch)/osc1Period)/(osc1Period*stretch) + .25) +
		osc4.step(phase4,float((i-osc1Period*stretch)/osc1Period)/(osc1Period*stretch) + .25) 
		) / 4
	)

# go back to beginning for clean loop

for i in range(3*int(osc1Period*osc1Period)*stretch/4):

	phase1 = float(i % osc1Period) / osc1Period
	phase2 = float(i % osc2Period) / osc2Period
	phase3 = float(i % osc3Period) / osc3Period
	phase4 = float(i % osc4Period) / osc4Period

	dominant7Data.append (
		(
		osc1.step(phase1,1.0 - float((i+osc1Period*stretch)/osc1Period)/(osc1Period*stretch)) +
		osc2.step(phase2,1.0 - float((i+osc1Period*stretch)/osc1Period)/(osc1Period*stretch)) +
		osc3.step(phase3,1.0 - float((i+osc1Period*stretch)/osc1Period)/(osc1Period*stretch)) +
		osc4.step(phase4,1.0 - float((i+osc1Period*stretch)/osc1Period)/(osc1Period*stretch))  
		) / 4

	)
dominant7Data = normalize(dominant7Data)

while len(dominant7Data) > len(fourthBelowData):
	fourthBelowData = fourthBelowData + fourthBelowData[-256:]

length = min(len(dominant7Data),len(fourthBelowData))
dominant7Data = map(add,dominant7Data[:length],fourthBelowData[:length])
dominant7Data = normalize(dominant7Data)

dominant7Data = tanhDrive(dominant7Data, drive)

dominant7Data = normalize(dominant7Data)

dominant7Data = floatTo8Bit(dominant7Data)

foDominant7.write(bytearray(dominant7Data))


## 5. TR-808 closed hihat
foHat = open('hat.raw','wb')

hatData = [0,0]

basePeriod = 256.0

# six square wave oscillators
osc1Period = basePeriod / 2
osc2Period = basePeriod / 3
osc3Period = basePeriod / 4.16
osc4Period = basePeriod / 5.43
osc5Period = basePeriod / 6.79
osc6Period = basePeriod / 8.21

filter1 = SVF()
filter1.oversample = 4
filter1.q = 1.5
filter1.setCutoffRatio(.5)

filter2 = SVF()
filter2.oversample = 4
filter2.q = 1.0
filter2.setCutoffRatio(.4)

env = expDecay()

env.fs = 20000
env.setTime(.01667)
env.amp = 1.0

osc1 = Pulse()
osc2 = Pulse()
osc3 = Pulse()
osc4 = Pulse()
osc5 = Pulse()
osc6 = Pulse()

for i in range(2048):

	osc1Phase = float(i % osc1Period) / osc1Period
	osc2Phase = float(i % osc2Period) / osc2Period
	osc3Phase = float(i % osc3Period) / osc3Period
	osc4Phase = float(i % osc4Period) / osc4Period
	osc5Phase = float(i % osc5Period) / osc5Period
	osc6Phase = float(i % osc6Period) / osc6Period

	mixed = (osc1.step(osc1Phase, .5) *
	         osc2.step(osc2Phase, .5) *
	         osc3.step(osc3Phase, .5) *
	         osc4.step(osc4Phase, .5) *
	         osc5.step(osc5Phase, .5) *
	         osc6.step(osc6Phase, .5) )

	filter1.step(mixed)
	filter2.step(filter1.band)

	hatData.append(env.step() * filter2.high)

hatData = normalize(hatData)
hatData = floatTo8Bit(hatData)

while hatData and hatData[-1] is 0:
	hatData.pop()

foHat.write(bytearray(hatData))


## 6. TR-808 KICK
foKick = open('kick.raw','wb')

kickData = [0,0]

kickPeriod = 512.0

bridgedT = Resonator()

bridgedT.setFreqRatio (1.0/kickPeriod)
bridgedT.setQ(20000.0)

env = expDecay()

env.fs = 20000
env.setTime(.02)
env.amp = 4.3

#eight = eightOhEight()

for i in range(10240):

	trigger = 16500.0 if i < 5 else 0.0

	#kickData.append(eight.step(trigger))

	bridgedT.setFreqRatio((1.0 + env.step())/kickPeriod)

	kickData.append(tanh(bridgedT.step(trigger)))

kickData = normalize(kickData)
kickData = floatTo8Bit(kickData)

foKick.write(bytearray(kickData))


## 7. SNARE
foSnare = open('snare.raw', 'wb')

snareData = [0,0]

# the snare is a fifth up
snarePeriod = 128.0 * (2.0/3)

bridgedT1 = Resonator()
bridgedT2 = Resonator()

bridgedT1.setFreqRatio (1.0/snarePeriod)
bridgedT1.setQ(20.0)

bridgedT2.setFreqRatio (1.0/(snarePeriod*(2.0/3)))
bridgedT2.setQ(8.0)

env = expDecay()

env.fs = 20000
env.setTime(.04)
env.amp = 7.5

env2 = expDecay()

env2.fs = 20000
env2.setTime(10.0)
env2.amp = .4

filter1 = SVF()
filter1.oversample = 4
filter1.q = 1.5
filter1.setCutoffRatio(.2)

lowPassCuttoff = 0.155

filter2 = SVF()
filter2.oversample = 4
filter2.q = 2.0
filter2.setCutoffRatio(lowPassCuttoff)

filter1Env = expDecay()

filter1Env.fs = 20000
filter1Env.setTime(0.01)
filter1Env.amp = 4.0

filter2Env = expDecay()

filter2Env.fs = 20000
filter2Env.setTime(0.025)
filter2Env.amp = (0.5 - lowPassCuttoff) 

crackleEnv = expDecay()	

crackleEnv.fs = 20000
crackleEnv.setTime(0.05)
crackleEnv.amp = (2.0) 

filter3 = SVF()
filter3.oversample = 4
filter3.q = .5
filter3.setCutoffRatio(.06)

for i in range(8000):

	trigger = 10.0 if i < 5 else 0.0

	filter1.q = 1.5 + filter1Env.step()

	A = env.step()

	filter1.step(A * (random()-0.5))

	filter2.setCutoffRatio(lowPassCuttoff + filter2Env.step())
	filter2.step(filter1.band)
	if random() < 1/100:
		crackleEnv.reset()

	filter3.step((env2.step() - .75*A*env2.amp/env.amp) * (random()-0.5))		

	crackle = crackleEnv.step()

	snareData.append(bridgedT1.step(trigger) + bridgedT2.step(trigger) + 
		crackle * filter2.low +
		crackle * 2.0 * filter3.band
	)

snareData = normalize(snareData)

drive = 30.0

snareData = tanhDrive(snareData, drive)
snareData = fixDc(snareData)
snareData = normalize(snareData)
snareData = floatTo8Bit(snareData)

foSnare.write(bytearray(snareData))


## 7.5 high SNARE
foHiSnare = open('snare-hi.raw', 'wb')

hiSnareData = [0,0]

snarePeriod = 64.0

bridgedT1 = Resonator()
bridgedT2 = Resonator()

bridgedT1.setFreqRatio (1.0/snarePeriod)
bridgedT1.setQ(10.0)

bridgedT2.setFreqRatio (1.0/(snarePeriod*(2.0/3)))
bridgedT2.setQ(4.0)

env = expDecay()

env.fs = 20000
env.setTime(.04)
env.amp = 7.5

env2 = expDecay()

env2.fs = 20000
env2.setTime(2.0)
env2.amp = .4

filter1 = SVF()
filter1.oversample = 4
filter1.q = 1.5
filter1.setCutoffRatio(.2)

lowPassCuttoff = 0.155

filter2 = SVF()
filter2.oversample = 4
filter2.q = 2.0
filter2.setCutoffRatio(lowPassCuttoff)

filter1Env = expDecay()

filter1Env.fs = 20000
filter1Env.setTime(0.01)
filter1Env.amp = 4.0

filter2Env = expDecay()

filter2Env.fs = 20000
filter2Env.setTime(0.025)
filter2Env.amp = (0.5 - lowPassCuttoff) 

crackleEnv = expDecay()	

crackleEnv.fs = 20000
crackleEnv.setTime(0.05)
crackleEnv.amp = (2.0) 

filter3 = SVF()
filter3.oversample = 4
filter3.q = .5
filter3.setCutoffRatio(.06)

for i in range(8000):

	trigger = 10.0 if i < 5 else 0.0

	filter1.q = 1.5 + filter1Env.step()

	A = env.step()

	filter1.step(A * (random()-0.5))

	filter2.setCutoffRatio(lowPassCuttoff + filter2Env.step())
	filter2.step(filter1.band)
	if random() < 1/100:
		crackleEnv.reset()

	filter3.step((env2.step() - .75*A*env2.amp/env.amp) * (random()-0.5))		

	crackle = crackleEnv.step()

	hiSnareData.append(bridgedT1.step(trigger) + bridgedT2.step(trigger) + 
		crackle * filter2.low +
		crackle * 2.0 * filter3.band
	)

hiSnareData = normalize(hiSnareData)

drive = 30.0

hiSnareData = tanhDrive(hiSnareData, drive)
hiSnareData = fixDc(hiSnareData)
hiSnareData = normalize(hiSnareData)
hiSnareData = floatTo8Bit(hiSnareData)

foHiSnare.write(bytearray(hiSnareData))