-- livecoding.lua
--  Reminder: don't forget to add help for each new function using sethelp()

-- livecoding.lua: Test of the OSG Lua interface, and livecoding!

-- Copyright (c) 2017 cxw/Incline.  CC-BY-SA 3.0.  In any derivative work, 
-- mention or link to https://bitbucket.org/inclinescene/public and 
-- http://devwrench.com.

require 'checks'
require 'print_r'
require 'Help'

-- Reminder: this is run as a chunk of its own, so no `local` variables will
-- be exported.

print("Hello, world! 42")
--local inp = io.read()     -- this works fine - blocks the calling thread
--print("Input was: ", inp)

--=========================================================================--
-- Load helpers

-- Load the basic stuff
function reload()

    -- Unload if we are being called for the second time
    package.loaded.dbg = nil
    package.loaded.Content = nil
    package.loaded.Geometry = nil
    package.loaded.Util = nil
        -- thanks to http://lua-users.org/lists/lua-l/2011-05/msg00365.html

    -- Load it
    Util = require 'Util'     -- load ./Util.lua into the global namespace.
        -- TODO? add OSG_FILE_PATH to the Lua search path?
    Geometry = require 'Geometry'
    Content = require 'Content'
    dbg = require 'debugger'  -- https://github.com/slembcke/debugger.lua

    UTIL = nil  -- drop the old one, hopefully
    UTIL = new('osg::ScriptUtils')

end --reload()
sethelp(reload, 'Reload packages used by livecoding.lua')

reload()    -- Do the initial load

-- Hack in some help for UTIL, since it doesn't have help of its own.
-- For some reason, UTIL.viewLookAt is different from the command line than
-- it is here.  TODO find out why.
sethelp('UTIL.viewLookAt',[[
Matrix viewLookAt(Vec3 eye, Vec3 center, Vec3 up): make a view matrix
a la gluLookAt().
]])

--=========================================================================--
-- Create geometry

local grp = new("osg::Group");
--Util.dump(grp)

xform = new("osg::MatrixTransform")
--grp:addChild(xform)
xform.DataVariance='DYNAMIC'
xform.Matrix={1,0,0,0,
              0,1,0,0,
              0,0,1,0,
              0,0,0,1}
--Util.dump(xform.Matrix)   -- Members are 0..15, NOT 1..16

local drw = new("osg::ShapeDrawable");
xform:addChild(drw)   -- methods use : ; property accesses use .
drw.Color = {0.75,1,0.5,1}
drw.DataVariance = 'DYNAMIC'    -- since we will change the color

-- Need to rescale the normals since the cone will be growing and shrinking.
-- This is the equivalent of the C++
-- drw->getOrCreateStateSet()->setMode(GL_NORMALIZE, osg::StateAttribute::ON)
local ss = new("osg::StateSet")
drw.StateSet = ss
ss:set('GL_NORMALIZE', 'ON')

local shape = new("osg::Cone")
-- Now set up the shape BEFORE the assignment to `drw.Shape`, because
-- that assignment triggers the `ShapeDrawable::build()` call that actually
-- makes the shape geometry.  Changes to `shape` after `drw.Shape=shape`
-- do not affect the displayed geometry even though they DO change the shape
-- properties in a way that is reflected when you later inspect the
-- scene graph!
shape.Radius=2
drw.Shape = shape   -- assign now that we're ready.

--frame=0

function updateCone(data, nv)
    -- TODO figure out how to get the FrameStamp.  I may need to add
    -- a function to the serializer for NodeVisitor.
    -- TODO figure out read-only properties.  I think I may just need to
    -- have a setter that always throws, or have a custom function that
    -- only returns the value and doesn't take any parameters.

    time = nv:getSimulationTime()

    -- Reporting
    -- frame = frame + 1
    -- if frame % 60 == 1 then
    --     print('sim time',time)
    -- end

    local scale = math.max(0.1, math.abs(2 * math.sin(time)))

    -- The following works
    xform.Matrix={scale,0,0,0,
                  0,scale,0,0,
                  0,0,scale,0,
                  0,0,0,1}

    --[[ -- This doesn't work, and I don't know why at the moment.
    xform.Matrix[0] = scale
    xform.Matrix[5] = scale
    xform.Matrix[10] = scale
    ]]--

    drw.Color = {time % 1.0, 1.0 - (time % 1.0), 0.5, 1.0}

end --updateCone

drw.UpdateCallback = updateCone

-- Axis markers
local axis_switch = new('osg::Switch')
grp:addChild(axis_switch)
axis_switch:addChild(Geometry.makeSphere({1,1,1,1}, {0,0,0}, 0.2))
axis_switch:addChild(Geometry.makeSphere({1,0,0,1}, {5,0,0}, 0.2, 'X'))
axis_switch:addChild(Geometry.makeSphere({0,1,0,1}, {0,5,0}, 0.2, 'Y'))
axis_switch:addChild(Geometry.makeSphere({0,0,1,1}, {0,0,5}, 0.2, 'Z'))

function axis(onoff)
    val = not not onoff     -- make sure it's a bool
    for i=0,3 do
        axis_switch:setValue(i, val)
    end
end
sethelp(axis,'axis(true) to turn axis markers on; axis(false) to turn them off')

print("Children:", grp:getNumChildren())

MODEL = grp     -- save it in the global namespace
S = nil         -- the last shape created

--=========================================================================--
-- Some random uniforms

local rss = ROOT:getOrCreateStateSet()
-- For rotation in the shader
uOrigin = Geometry.makeUniform('Origin','FLOAT_VEC2')
do
    rss:add(uOrigin)
    local function origin_update(sim_time)  -- called via CBK
        local radius = 0.5
        local speed = 1.0
        uOrigin:value({radius*math.cos(speed*sim_time),
                        radius*math.sin(speed*sim_time)})
        --print('Origin:', origin:value()[0], origin:value()[1])
    end
    CBK[#CBK+1] = origin_update
end
sethelp('uOrigin','uniform vec2 Origin')

-- For the HUD
uSpeed = Geometry.makeUniform('circle_speed', 'FLOAT', 4) --try 1
sethelp('uSpeed','uniform float circle_speed')

uInterval = Geometry.makeUniform('circle_interval', 'FLOAT', 1) --try 2.4
sethelp('uInterval','uniform float circle_interval')

uShift = Geometry.makeUniform('circle_shift', 'FLOAT', 0)
sethelp('uShift','uniform float circle_shift')

rss:add(uSpeed)
rss:add(uInterval)
rss:add(uShift)

--=========================================================================--
-- Live-coding functions
-- Global S holds the current shape.  All these functions either populate S
-- or operate on S, unless otherwise specified.
-- All shape-creation functions add the shape to MODEL, unless
-- otherwise specified.

-- Create a sphere and add it to the model.
function sph(color, pos, radius, name)
    S = Geometry.makeSphere(
        Util.vec2seq(color, 4),
        Util.vec2seq(pos, 3),
        radius or 1.0,
        name or '')
    MODEL:addChild(S)
end --sph

sethelp(sph, [[
Create a sphere and add it to the model.  Sets S to the new shape.
function sph(color, pos, radius, name)]])
sethelp('sph','See help(sph)')

-- Remove the last-added child
function drop()
    MODEL:removeChild(S)
end
sethelp(drop,'drop(): Remove the last-added child (S)')

print [[
Reminder --> camoff, then
CAM.ViewMatrix = UTIL:viewLookAt({1,-20,6},{2.5,2.5,2.5},{0,0,1})]]

return grp  -- whatever object(s) we return are what are displayed.

-- vi: set ts=4 sts=4 sw=4 et ai: --
