/*
 * Decompiled with CFR 0.152.
 */
package jkcemu.programming.assembler;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.EmptyStackException;
import java.util.HashMap;
import java.util.Map;
import java.util.Stack;
import jkcemu.base.EmuUtil;
import jkcemu.base.FileFormat;
import jkcemu.base.FileSaver;
import jkcemu.base.LoadData;
import jkcemu.programming.PrgException;
import jkcemu.programming.PrgLogger;
import jkcemu.programming.PrgOptions;
import jkcemu.programming.PrgSource;
import jkcemu.programming.TooManyErrorsException;
import jkcemu.programming.assembler.AsmArg;
import jkcemu.programming.assembler.AsmLabel;
import jkcemu.programming.assembler.AsmLine;
import jkcemu.programming.assembler.AsmStackEntry;
import jkcemu.programming.assembler.ExprParser;

public class Z80Assembler {
    private static String[] sortedReservedWords = new String[]{"ADC", "ADD", "AND", "BINCLUDE", "BIT", "CALL", "CCF", "CPD", "CPDR", "CPI", "CPIR", "CPL", "CPU", "DA", "DAA", "DB", "DEC", "DEFA", "DEFB", "DEFH", "DEFM", "DEFS", "DEFW", "DFB", "DFH", "DFS", "DFW", "DI", "DJNZ", "DW", "EI", "ELSE", "END", "ENDIF", "ENT", "EQU", "EX", "EXX", "HALT", "HEX", "IF1", "IF2", "IFDEF", "IFE", "IFF", "IFNDEF", "IM", "IN", "INC", "INCLUDE", "IND", "INDR", "INI", "INIR", "JP", "JR", "LD", "LDD", "LDDR", "LDI", "LDIR", "NEG", "NOP", "OR", "ORG", "OTDR", "OTIR", "OUT", "OUTD", "OUTI", "POP", "PUSH", "RES", "RET", "RETI", "RETN", "RL", "RLA", "RLC", "RLCA", "RLD", "RR", "RRA", "RRC", "RRCA", "RRD", "RST", "SBC", "SCF", "SET", "SLA", "SRA", "SRL", "SUB", "XOR", "Z80"};
    private static String[] sortedReservedRobotronWords = new String[]{"CAC", "CAM", "CANC", "CANZ", "CAP", "CAPE", "CAPO", "CAZ", "CMP", "EXAF", "INF", "JMP", "JPC", "JPM", "JPNC", "JPNZ", "JPP", "JPPE", "JPPO", "JPZ", "JRC", "JRNC", "JRNZ", "JRZ", "RC", "RM", "RNC", "RNZ", "RP", "RPE", "RPO", "RZ"};
    private static String BUILT_IN_LABEL = "__JKCEMU__";
    private PrgSource curSource = null;
    private PrgSource mainSource = null;
    private PrgOptions options;
    private PrgLogger logger;
    private Stack<AsmStackEntry> stack;
    private Map<File, byte[]> file2Bytes;
    private Map<File, PrgSource> file2Source;
    private Map<String, AsmLabel> labels;
    private AsmLabel[] sortedLabels;
    private StringBuilder srcOut;
    private ByteArrayOutputStream codeBuf;
    private byte[] codeOut;
    private boolean addrOverflow;
    private boolean endReached;
    private boolean interactive;
    private boolean orgOverlapped;
    private boolean relJumpsTooLong;
    private boolean suppressLineAddr;
    private boolean status;
    private volatile boolean execEnabled;
    private Integer entryAddr;
    private int begAddr;
    private int endAddr;
    private int curAddr;
    private int instBegAddr;
    private int passNum;
    private int errCnt;

    public Z80Assembler(String string, String string2, File file, PrgOptions prgOptions, PrgLogger prgLogger, boolean bl) {
        this.options = prgOptions;
        this.logger = prgLogger;
        this.interactive = bl;
        this.stack = new Stack();
        this.file2Bytes = new HashMap<File, byte[]>();
        this.file2Source = new HashMap<File, PrgSource>();
        this.labels = new HashMap<String, AsmLabel>();
        this.sortedLabels = null;
        this.srcOut = null;
        this.codeBuf = null;
        this.codeOut = null;
        this.addrOverflow = false;
        this.endReached = false;
        this.orgOverlapped = false;
        this.relJumpsTooLong = false;
        this.suppressLineAddr = false;
        this.status = true;
        this.execEnabled = true;
        this.entryAddr = null;
        this.begAddr = -1;
        this.endAddr = -1;
        this.curAddr = 0;
        this.instBegAddr = 0;
        this.passNum = 0;
        this.errCnt = 0;
        if (this.options.getFormatSource() && string != null) {
            this.srcOut = new StringBuilder(Math.max(string.length(), 16));
        }
        if (this.options.getCreateCode()) {
            this.codeBuf = new ByteArrayOutputStream(32768);
        }
        this.labels.put(BUILT_IN_LABEL, new AsmLabel(BUILT_IN_LABEL, 1));
        if (string != null) {
            this.mainSource = PrgSource.readText(string, string2, file);
        } else if (file != null) {
            try {
                this.mainSource = PrgSource.readFile(file);
            }
            catch (IOException iOException) {
                String string3 = iOException.getMessage();
                if (string3 != null && string3.trim().isEmpty()) {
                    string3 = null;
                }
                if (string3 == null) {
                    string3 = "Datei kann nicht ge\u00f6ffnet werden";
                }
                this.appendToErrLog(file.getPath() + ": " + string3);
            }
        }
    }

    public boolean addLabel(String string, int n) {
        if (!this.options.getLabelsCaseSensitive()) {
            string = string.toUpperCase();
        }
        return this.labels.put(string, new AsmLabel(string, n)) == null;
    }

    public boolean assemble(String string, boolean bl) throws IOException {
        this.errCnt = 0;
        this.passNum = 1;
        this.execEnabled = true;
        this.endReached = false;
        this.curSource = this.mainSource;
        try {
            this.parseAsm();
            this.computeMissingLabelValues();
            if (this.execEnabled && this.status) {
                if (this.mainSource != null) {
                    this.mainSource.reset();
                }
                this.passNum = 2;
                this.endReached = false;
                this.curSource = this.mainSource;
                this.parseAsm();
                if (this.codeBuf != null) {
                    this.codeBuf.close();
                    if (this.execEnabled && this.status) {
                        this.codeOut = this.codeBuf.toByteArray();
                    }
                }
                if (this.execEnabled && this.status) {
                    if (this.options.getPrintLabels()) {
                        this.printLabels();
                    }
                    if (this.options.getCodeToFile()) {
                        this.writeCodeToFile(string, bl);
                    }
                }
            }
        }
        catch (TooManyErrorsException tooManyErrorsException) {
            this.appendToErrLog("\nAbgebrochen aufgrund zu vieler Fehler\n");
            this.codeOut = null;
        }
        if (this.execEnabled && this.errCnt > 0) {
            this.appendToErrLog(String.format("%d Fehler\n", this.errCnt));
            this.codeOut = null;
        }
        return this.status;
    }

    public void cancel() {
        this.execEnabled = false;
    }

    public int getBegAddr() {
        return this.begAddr;
    }

    public int getEndAddr() {
        return this.endAddr;
    }

    public byte[] getCreatedCode() {
        return this.codeOut;
    }

    public Integer getEntryAddr() {
        return this.entryAddr;
    }

    public String getFormattedSourceText() {
        return this.srcOut != null ? this.srcOut.toString() : null;
    }

    public Integer getLabelValue(String string) {
        Object object;
        AsmLabel asmLabel;
        Integer n = null;
        if (string != null && (asmLabel = this.labels.get(string)) != null && (object = asmLabel.getLabelValue()) != null && object instanceof Integer) {
            n = (Integer)object;
        }
        return n;
    }

    public boolean getOrgOverlapped() {
        return this.orgOverlapped;
    }

    public PrgOptions getOptions() {
        return this.options;
    }

    public Collection<PrgSource> getPrgSources() {
        ArrayList<PrgSource> arrayList = new ArrayList<PrgSource>(this.file2Source.size() + 1);
        if (this.mainSource != null) {
            arrayList.add(this.mainSource);
        }
        arrayList.addAll(this.file2Source.values());
        return arrayList;
    }

    public boolean getRelJumpsTooLong() {
        return this.relJumpsTooLong;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public AsmLabel[] getSortedLabels() {
        int n;
        Object[] objectArray = this.sortedLabels;
        if (objectArray == null && (n = this.labels.size()) > 0) {
            Map<String, AsmLabel> map = this.labels;
            try {
                map = new HashMap<String, AsmLabel>(n);
                map.putAll(this.labels);
                map.remove(BUILT_IN_LABEL);
            }
            catch (ClassCastException | IllegalArgumentException | UnsupportedOperationException runtimeException) {
                map = this.labels;
            }
            try {
                Collection<AsmLabel> collection = map.values();
                if (collection != null) {
                    int n2 = collection.size();
                    if (n2 > 0) {
                        objectArray = collection.toArray(new AsmLabel[n2]);
                        if (objectArray != null) {
                            Arrays.sort(objectArray);
                        }
                    } else {
                        objectArray = new AsmLabel[]{};
                    }
                }
            }
            catch (ArrayStoreException arrayStoreException) {
            }
            catch (ClassCastException classCastException) {
            }
            finally {
                this.sortedLabels = objectArray;
            }
        }
        return objectArray;
    }

    public boolean isReservedWord(String string) {
        boolean bl;
        boolean bl2 = bl = Arrays.binarySearch(sortedReservedWords, string) >= 0;
        if (!bl) {
            Syntax syntax = this.options.getAsmSyntax();
            if ((syntax == Syntax.ALL || syntax == Syntax.ROBOTRON_ONLY) && Arrays.binarySearch(sortedReservedRobotronWords, string) >= 0) {
                bl = true;
            }
            if (syntax == Syntax.ALL || syntax == Syntax.ZILOG_ONLY) {
                bl |= string.equals("CP");
            }
        }
        if (this.options.getAllowUndocInst()) {
            bl |= string.equals("SLL");
        }
        return bl;
    }

    public void putWarning(String string) {
        if (this.passNum == 2) {
            this.appendLineNumMsgToErrLog(string, "Warnung");
        }
    }

    public void appendLineNumMsgToErrLog(String string, String string2) {
        int n;
        StringBuilder stringBuilder = new StringBuilder(128);
        if (this.curSource != null && (n = this.curSource.getLineNum()) > 0) {
            String string3 = this.curSource.getName();
            if (string3 != null && !string3.isEmpty()) {
                stringBuilder.append(string3);
                stringBuilder.append(": ");
            }
            if (string2 != null) {
                stringBuilder.append(string2);
                stringBuilder.append(" in ");
            }
            stringBuilder.append("Zeile ");
            stringBuilder.append(n);
            stringBuilder.append(": ");
        }
        if (string != null) {
            stringBuilder.append(string);
        }
        if (!string.endsWith("\n")) {
            stringBuilder.append('\n');
        }
        this.appendToErrLog(stringBuilder.toString());
    }

    private void appendToErrLog(String string) {
        if (this.logger != null) {
            this.logger.appendToErrLog(string);
        }
    }

    private void appendToOutLog(String string) {
        if (this.logger != null) {
            this.logger.appendToOutLog(string);
        }
    }

    private void checkPrint16BitWarning(int n) {
        if (n < Short.MIN_VALUE || n > 65535) {
            this.putWarning("Numerischer Wert au\u00dferhalb 16-Bit-Bereich:Bits gehen verloren");
        }
    }

    private void checkAddr() throws PrgException {
        if (!this.addrOverflow && this.curAddr > 65536) {
            this.addrOverflow = true;
            throw new PrgException("\u00dcberlauf: Adressz\u00e4hler > 0FFFFh");
        }
    }

    private void computeMissingLabelValues() {
        boolean bl = false;
        boolean bl2 = false;
        do {
            for (AsmLabel asmLabel : this.labels.values()) {
                String string;
                Object object = asmLabel.getLabelValue();
                if (object == null || object instanceof Integer || (string = object.toString()) == null) continue;
                try {
                    Integer n = ExprParser.parse(string, this.instBegAddr, this.labels, false, this.options.getLabelsCaseSensitive());
                    if (n != null) {
                        asmLabel.setLabelValue(n);
                        bl = true;
                        continue;
                    }
                    bl2 = true;
                }
                catch (PrgException prgException) {}
            }
        } while (bl && bl2);
    }

    private boolean isAssemblingEnabled() {
        boolean bl = true;
        for (AsmStackEntry asmStackEntry : this.stack) {
            if (asmStackEntry.isAssemblingEnabled()) continue;
            bl = false;
            break;
        }
        return bl;
    }

    private void parseAsm() throws IOException, TooManyErrorsException {
        this.begAddr = -1;
        this.endAddr = -1;
        this.curAddr = 0;
        this.instBegAddr = 0;
        this.stack.clear();
        while (this.execEnabled && !this.endReached && this.curSource != null) {
            String string = this.curSource.readLine();
            if (string != null) {
                this.parseLine(string);
                continue;
            }
            if (this.curSource != this.mainSource) {
                this.curSource = this.mainSource;
                continue;
            }
            this.curSource = null;
        }
        if (!this.stack.isEmpty()) {
            try {
                int n = this.stack.peek().getLineNum();
                StringBuilder stringBuilder = new StringBuilder(32);
                stringBuilder.append("Bedingung");
                if (n > 0) {
                    stringBuilder.append(" in Zeile ");
                    stringBuilder.append(n);
                }
                stringBuilder.append(" nicht geschlossen (ENDIF fehlt)");
                this.appendLineNumMsgToErrLog(stringBuilder.toString(), "Fehler");
                this.status = false;
                ++this.errCnt;
            }
            catch (EmptyStackException emptyStackException) {
                // empty catch block
            }
        }
    }

    private int parseExpr(String string) throws PrgException {
        int n = 0;
        Integer n2 = ExprParser.parse(string, this.instBegAddr, this.labels, this.passNum == 2, this.options.getLabelsCaseSensitive());
        if (n2 != null) {
            n = n2;
        } else if (this.passNum == 2) {
            throw new PrgException("Wert nicht ermittelbar");
        }
        return n;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void parseLine(String string) throws IOException, TooManyErrorsException {
        Object object;
        this.instBegAddr = this.curAddr;
        this.suppressLineAddr = false;
        String string2 = null;
        try {
            AsmLine asmLine = AsmLine.scanLine(this, string, this.options.getLabelsCaseSensitive());
            if (asmLine != null) {
                Object object2;
                string2 = asmLine.getLabel();
                if (string2 != null && this.passNum == 1) {
                    if (this.labels.containsKey(string2)) {
                        throw new PrgException("Marke " + string2 + " bereits vergeben");
                    }
                    this.labels.put(string2, new AsmLabel(string2, this.curAddr));
                    this.sortedLabels = null;
                }
                if ((object2 = asmLine.getInstruction()) != null && !((String)object2).isEmpty()) {
                    if (((String)object2).equals("IF") || ((String)object2).equals("IFT")) {
                        this.parseIF(asmLine, true);
                    } else if (((String)object2).equals("IFE") || ((String)object2).equals("IFF")) {
                        this.parseIF(asmLine, false);
                    } else if (((String)object2).equals("IF1")) {
                        this.parseIfInPass(asmLine, 1);
                    } else if (((String)object2).equals("IF2")) {
                        this.parseIfInPass(asmLine, 2);
                    } else if (((String)object2).equals("IFDEF")) {
                        this.parseIFDEF(asmLine);
                    } else if (((String)object2).equals("IFNDEF")) {
                        this.parseIFNDEF(asmLine);
                    } else if (((String)object2).equals("ELSE")) {
                        this.parseELSE(asmLine);
                    } else if (((String)object2).equals("ENDIF")) {
                        this.parseENDIF(asmLine);
                    } else if (this.isAssemblingEnabled() && !((String)object2).equals("Z80") && !((String)object2).equals(".Z80")) {
                        if (((String)object2).equals("ADD")) {
                            this.parseADD(asmLine);
                        } else if (((String)object2).equals("ADC")) {
                            this.parseADC_SBC(asmLine, 136, 74);
                        } else if (((String)object2).equals("AND")) {
                            this.parseBiOp8(asmLine, 160);
                        } else if (((String)object2).equals("BINCLUDE") || ((String)object2).equals(".BINCLUDE")) {
                            this.parseBINCLUDE(asmLine);
                        } else if (((String)object2).equals("BIT")) {
                            this.parseSingleBit(asmLine, 64);
                        } else if (((String)object2).equals("CAC")) {
                            this.parseInstDirectAddr(asmLine, 220);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("CALL")) {
                            this.parseCALL(asmLine);
                        } else if (((String)object2).equals("CAM")) {
                            this.parseInstDirectAddr(asmLine, 252);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("CANC")) {
                            this.parseInstDirectAddr(asmLine, 212);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("CANZ")) {
                            this.parseInstDirectAddr(asmLine, 196);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("CAP")) {
                            this.parseInstDirectAddr(asmLine, 244);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("CAPE")) {
                            this.parseInstDirectAddr(asmLine, 236);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("CAPO")) {
                            this.parseInstDirectAddr(asmLine, 228);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("CAZ")) {
                            this.parseInstDirectAddr(asmLine, 204);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("CCF")) {
                            asmLine.checkEOL();
                            this.putCode(63);
                        } else if (((String)object2).equals("CMP")) {
                            this.parseBiOp8(asmLine, 184);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("CP")) {
                            this.parseBiOp8(asmLine, 184);
                            this.zilogMnemonic();
                        } else if (((String)object2).equals("CPD")) {
                            asmLine.checkEOL();
                            this.putCode(237);
                            this.putCode(169);
                        } else if (((String)object2).equals("CPDR")) {
                            asmLine.checkEOL();
                            this.putCode(237);
                            this.putCode(185);
                        } else if (((String)object2).equals("CPI")) {
                            asmLine.checkEOL();
                            this.putCode(237);
                            this.putCode(161);
                        } else if (((String)object2).equals("CPIR")) {
                            asmLine.checkEOL();
                            this.putCode(237);
                            this.putCode(177);
                        } else if (((String)object2).equals("CPL")) {
                            asmLine.checkEOL();
                            this.putCode(47);
                        } else if (((String)object2).equals("CPU") || ((String)object2).equals(".CPU")) {
                            this.parseCPU(asmLine);
                        } else if (((String)object2).equals("DAA")) {
                            asmLine.checkEOL();
                            this.putCode(39);
                        } else if (((String)object2).equals("DEC")) {
                            this.parseINC_DEC(asmLine, 5, 11);
                        } else if (((String)object2).equals("DEFB") || ((String)object2).equals(".DEFB") || ((String)object2).equals("DEFM") || ((String)object2).equals(".DEFM") || ((String)object2).equals("DFB") || ((String)object2).equals(".DFB") || ((String)object2).equals("DB") || ((String)object2).equals(".DB")) {
                            this.parseDEFB(asmLine);
                        } else if (((String)object2).equals("DEFH") || ((String)object2).equals(".DEFH") || ((String)object2).equals("DFH") || ((String)object2).equals(".DFH") || ((String)object2).equals("HEX") || ((String)object2).equals(".HEX")) {
                            this.parseDEFH(asmLine);
                        } else if (((String)object2).equals("DEFS") || ((String)object2).equals(".DEFS") || ((String)object2).equals("DFS") || ((String)object2).equals(".DFS") || ((String)object2).equals("DS") || ((String)object2).equals(".DS")) {
                            this.parseDEFS(asmLine);
                        } else if (((String)object2).equals("DEFW") || ((String)object2).equals(".DEFW") || ((String)object2).equals("DFW") || ((String)object2).equals(".DFW") || ((String)object2).equals("DA") || ((String)object2).equals(".DA") || ((String)object2).equals("DW") || ((String)object2).equals(".DW")) {
                            this.parseDEFW(asmLine);
                        } else if (((String)object2).equals("DI")) {
                            asmLine.checkEOL();
                            this.putCode(243);
                        } else if (((String)object2).equals("DJNZ")) {
                            int n = this.getAddrDiff(asmLine.nextArg());
                            asmLine.checkEOL();
                            this.putCode(16);
                            this.putCode(n);
                        } else if (((String)object2).equals("EI")) {
                            asmLine.checkEOL();
                            this.putCode(251);
                        } else if (((String)object2).equals("END") || ((String)object2).equals(".END")) {
                            this.parseEND(asmLine);
                        } else if (((String)object2).equals("ENT") || ((String)object2).equals(".ENT")) {
                            this.parseENT(asmLine);
                        } else if (((String)object2).equals("EQU") || ((String)object2).equals(".EQU")) {
                            this.parseEQU(asmLine);
                        } else if (((String)object2).equals("EX")) {
                            this.parseEX(asmLine);
                        } else if (((String)object2).equals("EXAF")) {
                            asmLine.checkEOL();
                            this.putCode(8);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("EXX")) {
                            asmLine.checkEOL();
                            this.putCode(217);
                        } else if (((String)object2).equals("HALT")) {
                            asmLine.checkEOL();
                            this.putCode(118);
                        } else if (((String)object2).equals("IM")) {
                            this.parseIM(asmLine);
                        } else if (((String)object2).equals("IN")) {
                            this.parseIN(asmLine);
                        } else if (((String)object2).equals("INC")) {
                            this.parseINC_DEC(asmLine, 4, 3);
                        } else if (((String)object2).equals("INCLUDE") || ((String)object2).equals(".INCLUDE")) {
                            this.parseINCLUDE(asmLine);
                        } else if (((String)object2).equals("INF")) {
                            asmLine.checkEOL();
                            this.putCode(237);
                            this.putCode(112);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("IND")) {
                            asmLine.checkEOL();
                            this.putCode(237);
                            this.putCode(170);
                        } else if (((String)object2).equals("INDR")) {
                            asmLine.checkEOL();
                            this.putCode(237);
                            this.putCode(186);
                        } else if (((String)object2).equals("INI")) {
                            asmLine.checkEOL();
                            this.putCode(237);
                            this.putCode(162);
                        } else if (((String)object2).equals("INIR")) {
                            asmLine.checkEOL();
                            this.putCode(237);
                            this.putCode(178);
                        } else if (((String)object2).equals("JMP")) {
                            this.parseJMP(asmLine);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("JP")) {
                            this.parseJP(asmLine);
                        } else if (((String)object2).equals("JPC")) {
                            this.parseInstDirectAddr(asmLine, 218);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("JPM")) {
                            this.parseInstDirectAddr(asmLine, 250);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("JPNC")) {
                            this.parseInstDirectAddr(asmLine, 210);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("JPNZ")) {
                            this.parseInstDirectAddr(asmLine, 194);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("JPP")) {
                            this.parseInstDirectAddr(asmLine, 242);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("JPPE")) {
                            this.parseInstDirectAddr(asmLine, 234);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("JPPO")) {
                            this.parseInstDirectAddr(asmLine, 226);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("JPZ")) {
                            this.parseInstDirectAddr(asmLine, 202);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("JR")) {
                            this.parseJR(asmLine);
                        } else if (((String)object2).equals("JRC")) {
                            int n = this.getAddrDiff(asmLine.nextArg());
                            asmLine.checkEOL();
                            this.putCode(56);
                            this.putCode(n);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("JRNC")) {
                            int n = this.getAddrDiff(asmLine.nextArg());
                            asmLine.checkEOL();
                            this.putCode(48);
                            this.putCode(n);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("JRNZ")) {
                            int n = this.getAddrDiff(asmLine.nextArg());
                            asmLine.checkEOL();
                            this.putCode(32);
                            this.putCode(n);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("JRZ")) {
                            int n = this.getAddrDiff(asmLine.nextArg());
                            asmLine.checkEOL();
                            this.putCode(40);
                            this.putCode(n);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("LD")) {
                            this.parseLD(asmLine);
                        } else if (((String)object2).equals("LDD")) {
                            asmLine.checkEOL();
                            this.putCode(237);
                            this.putCode(168);
                        } else if (((String)object2).equals("LDDR")) {
                            asmLine.checkEOL();
                            this.putCode(237);
                            this.putCode(184);
                        } else if (((String)object2).equals("LDI")) {
                            asmLine.checkEOL();
                            this.putCode(237);
                            this.putCode(160);
                        } else if (((String)object2).equals("LDIR")) {
                            asmLine.checkEOL();
                            this.putCode(237);
                            this.putCode(176);
                        } else if (((String)object2).equals("NEG")) {
                            asmLine.checkEOL();
                            this.putCode(237);
                            this.putCode(68);
                        } else if (((String)object2).equals("NOP")) {
                            asmLine.checkEOL();
                            this.putCode(0);
                        } else if (((String)object2).equals("OR")) {
                            this.parseBiOp8(asmLine, 176);
                        } else if (((String)object2).equals("ORG") || ((String)object2).equals(".ORG")) {
                            this.parseORG(asmLine);
                        } else if (((String)object2).equals("OTDR")) {
                            asmLine.checkEOL();
                            this.putCode(237);
                            this.putCode(187);
                        } else if (((String)object2).equals("OTIR")) {
                            asmLine.checkEOL();
                            this.putCode(237);
                            this.putCode(179);
                        } else if (((String)object2).equals("OUT")) {
                            this.parseOUT(asmLine);
                        } else if (((String)object2).equals("OUTD")) {
                            asmLine.checkEOL();
                            this.putCode(237);
                            this.putCode(171);
                        } else if (((String)object2).equals("OUTI")) {
                            asmLine.checkEOL();
                            this.putCode(237);
                            this.putCode(163);
                        } else if (((String)object2).equals("POP")) {
                            this.parsePUSH_POP(asmLine, 193);
                        } else if (((String)object2).equals("PUSH")) {
                            this.parsePUSH_POP(asmLine, 197);
                        } else if (((String)object2).equals("RC")) {
                            asmLine.checkEOL();
                            this.putCode(216);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("RES")) {
                            this.parseSingleBit(asmLine, 128);
                        } else if (((String)object2).equals("RET")) {
                            this.parseRET(asmLine);
                        } else if (((String)object2).equals("RETI")) {
                            asmLine.checkEOL();
                            this.putCode(237);
                            this.putCode(77);
                        } else if (((String)object2).equals("RETN")) {
                            asmLine.checkEOL();
                            this.putCode(237);
                            this.putCode(69);
                        } else if (((String)object2).equals("RM")) {
                            asmLine.checkEOL();
                            this.putCode(248);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("RNC")) {
                            asmLine.checkEOL();
                            this.putCode(208);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("RNZ")) {
                            asmLine.checkEOL();
                            this.putCode(192);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("RL")) {
                            this.parseRotShift(asmLine, 16);
                        } else if (((String)object2).equals("RLA")) {
                            asmLine.checkEOL();
                            this.putCode(23);
                        } else if (((String)object2).equals("RLC")) {
                            this.parseRotShift(asmLine, 0);
                        } else if (((String)object2).equals("RLCA")) {
                            asmLine.checkEOL();
                            this.putCode(7);
                        } else if (((String)object2).equals("RLD")) {
                            this.parseRXD(asmLine, 111);
                        } else if (((String)object2).equals("RP")) {
                            asmLine.checkEOL();
                            this.putCode(240);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("RPE")) {
                            asmLine.checkEOL();
                            this.putCode(232);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("RPO")) {
                            asmLine.checkEOL();
                            this.putCode(224);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("RR")) {
                            this.parseRotShift(asmLine, 24);
                        } else if (((String)object2).equals("RRA")) {
                            asmLine.checkEOL();
                            this.putCode(31);
                        } else if (((String)object2).equals("RRC")) {
                            this.parseRotShift(asmLine, 8);
                        } else if (((String)object2).equals("RRCA")) {
                            asmLine.checkEOL();
                            this.putCode(15);
                        } else if (((String)object2).equals("RRD")) {
                            this.parseRXD(asmLine, 103);
                        } else if (((String)object2).equals("RST")) {
                            this.parseRST(asmLine);
                        } else if (((String)object2).equals("RZ")) {
                            asmLine.checkEOL();
                            this.putCode(200);
                            this.robotronMnemonic();
                        } else if (((String)object2).equals("SBC")) {
                            this.parseADC_SBC(asmLine, 152, 66);
                        } else if (((String)object2).equals("SCF")) {
                            asmLine.checkEOL();
                            this.putCode(55);
                        } else if (((String)object2).equals("SET")) {
                            this.parseSingleBit(asmLine, 192);
                        } else if (((String)object2).equals("SLA")) {
                            this.parseRotShift(asmLine, 32);
                        } else if (((String)object2).equals("SLL")) {
                            this.parseRotShift(asmLine, 48);
                            this.undocInst();
                        } else if (((String)object2).equals("SRA")) {
                            this.parseRotShift(asmLine, 40);
                        } else if (((String)object2).equals("SRL")) {
                            this.parseRotShift(asmLine, 56);
                        } else if (((String)object2).equals("SUB")) {
                            this.parseBiOp8(asmLine, 144);
                        } else if (((String)object2).equals("XOR")) {
                            this.parseBiOp8(asmLine, 168);
                        } else {
                            throw new PrgException("'" + (String)object2 + "': Unbekannte Mnemonik");
                        }
                    }
                }
                if (this.srcOut != null && this.passNum == 2) {
                    asmLine.appendFormattedTo(this.srcOut);
                    this.srcOut.append('\n');
                }
            }
        }
        catch (PrgException prgException) {
            object = prgException.getMessage();
            if (object == null) {
                object = "Unbekannter Fehler";
            }
            this.appendLineNumMsgToErrLog((String)object, "Fehler");
            this.status = false;
            ++this.errCnt;
            if (this.errCnt >= 100) {
                throw new TooManyErrorsException();
            }
        }
        finally {
            if (this.interactive && !this.suppressLineAddr && this.curSource != null && this.passNum == 2) {
                AsmLabel asmLabel;
                if (this.instBegAddr < this.curAddr) {
                    this.curSource.setLineAddr(this.instBegAddr);
                } else if (string2 != null && (asmLabel = this.labels.get(string2)) != null && (object = asmLabel.getLabelValue()) != null && object instanceof Integer && (Integer)object == this.instBegAddr) {
                    this.curSource.setLineAddr(this.instBegAddr);
                }
            }
        }
    }

    private void parseADD(AsmLine asmLine) throws PrgException {
        AsmArg asmArg = asmLine.nextArg();
        if (asmLine.hasMoreArgs()) {
            AsmArg asmArg2 = asmLine.nextArg();
            asmLine.checkEOL();
            if (asmArg.equalsUpper("HL")) {
                if (asmArg2.equalsUpper("BC")) {
                    this.putCode(9);
                } else if (asmArg2.equalsUpper("DE")) {
                    this.putCode(25);
                } else if (asmArg2.equalsUpper("HL")) {
                    this.putCode(41);
                } else if (asmArg2.equalsUpper("SP")) {
                    this.putCode(57);
                } else {
                    this.throwNoSuchInstArgs();
                }
            } else if (asmArg.equalsUpper("IX")) {
                if (asmArg2.equalsUpper("BC")) {
                    this.putCode(221);
                    this.putCode(9);
                } else if (asmArg2.equalsUpper("DE")) {
                    this.putCode(221);
                    this.putCode(25);
                } else if (asmArg2.equalsUpper("SP")) {
                    this.putCode(221);
                    this.putCode(57);
                } else if (asmArg2.equalsUpper("IX")) {
                    this.putCode(221);
                    this.putCode(41);
                } else {
                    this.throwNoSuchInstArgs();
                }
            } else if (asmArg.equalsUpper("IY")) {
                if (asmArg2.equalsUpper("BC")) {
                    this.putCode(253);
                    this.putCode(9);
                } else if (asmArg2.equalsUpper("DE")) {
                    this.putCode(253);
                    this.putCode(25);
                } else if (asmArg2.equalsUpper("SP")) {
                    this.putCode(253);
                    this.putCode(57);
                } else if (asmArg2.equalsUpper("IY")) {
                    this.putCode(253);
                    this.putCode(41);
                } else {
                    this.throwNoSuchInstArgs();
                }
            } else {
                this.parseBiOp8(asmArg, asmArg2, 128);
            }
        } else {
            this.parseBiOp8(asmArg, 128);
        }
    }

    private void parseADC_SBC(AsmLine asmLine, int n, int n2) throws PrgException {
        AsmArg asmArg = asmLine.nextArg();
        if (asmLine.hasMoreArgs()) {
            AsmArg asmArg2 = asmLine.nextArg();
            asmLine.checkEOL();
            if (asmArg.equalsUpper("HL")) {
                if (asmArg2.equalsUpper("BC")) {
                    this.putCode(237);
                    this.putCode(n2);
                } else if (asmArg2.equalsUpper("DE")) {
                    this.putCode(237);
                    this.putCode(n2 + 16);
                } else if (asmArg2.equalsUpper("HL")) {
                    this.putCode(237);
                    this.putCode(n2 + 32);
                } else if (asmArg2.equalsUpper("SP")) {
                    this.putCode(237);
                    this.putCode(n2 + 48);
                } else {
                    this.throwNoSuchInstArgs();
                }
            } else {
                this.parseBiOp8(asmArg, asmArg2, n);
            }
        } else {
            this.parseBiOp8(asmArg, n);
        }
    }

    private void parseBINCLUDE(AsmLine asmLine) throws PrgException {
        File file = this.getIncludeFile(asmLine);
        asmLine.checkEOL();
        byte[] byArray = this.file2Bytes.get(file);
        try {
            if (byArray == null) {
                byArray = EmuUtil.readFile(file, false, 65536);
                this.file2Bytes.put(file, byArray);
            }
            if (byArray == null) {
                throw new IOException();
            }
            for (byte by : byArray) {
                this.putCode(by);
            }
        }
        catch (IOException iOException) {
            String string = iOException.getMessage();
            if (string != null && string.trim().isEmpty()) {
                string = null;
            }
            if (string == null) {
                string = "Datei kann nicht ge\u00f6ffnet werden.";
            }
            throw new PrgException(string);
        }
    }

    private void parseCALL(AsmLine asmLine) throws PrgException {
        AsmArg asmArg = asmLine.nextArg();
        if (asmLine.hasMoreArgs()) {
            AsmArg asmArg2 = asmLine.nextArg();
            asmLine.checkEOL();
            if (asmArg.equalsUpper("NZ")) {
                this.zilogSyntax();
                this.putCode(196);
                this.putWord(this.getWord(asmArg2));
            } else if (asmArg.equalsUpper("Z")) {
                this.zilogSyntax();
                this.putCode(204);
                this.putWord(this.getWord(asmArg2));
            } else if (asmArg.equalsUpper("NC")) {
                this.zilogSyntax();
                this.putCode(212);
                this.putWord(this.getWord(asmArg2));
            } else if (asmArg.equalsUpper("C")) {
                this.zilogSyntax();
                this.putCode(220);
                this.putWord(this.getWord(asmArg2));
            } else if (asmArg.equalsUpper("PO")) {
                this.zilogSyntax();
                this.putCode(228);
                this.putWord(this.getWord(asmArg2));
            } else if (asmArg.equalsUpper("PE")) {
                this.zilogSyntax();
                this.putCode(236);
                this.putWord(this.getWord(asmArg2));
            } else if (asmArg.equalsUpper("P")) {
                this.zilogSyntax();
                this.putCode(244);
                this.putWord(this.getWord(asmArg2));
            } else if (asmArg.equalsUpper("M")) {
                this.zilogSyntax();
                this.putCode(252);
                this.putWord(this.getWord(asmArg2));
            } else {
                this.throwNoSuchInstArgs();
            }
        } else {
            this.putCode(205);
            this.putWord(this.getWord(asmArg));
        }
    }

    private void parseCPU(AsmLine asmLine) throws PrgException {
        boolean bl = false;
        String string = asmLine.nextArg().toString();
        if (string != null) {
            if ((string = string.toUpperCase()).equals("U880")) {
                this.robotronSyntax();
                bl = true;
            } else if (string.equals("Z80")) {
                this.zilogSyntax();
                bl = true;
            }
        }
        if (!bl) {
            throw new PrgException("'Z80' oder 'U880' erwartet");
        }
        asmLine.checkEOL();
    }

    private void parseDEFB(AsmLine asmLine) throws PrgException {
        do {
            int n;
            String string;
            if ((string = asmLine.nextArg().toString()) == null || (n = string.length()) <= 0) continue;
            char c = string.charAt(0);
            if (c == '\'' || c == '\"') {
                if (string.indexOf(c, 1) == 2) {
                    this.putChar(string.charAt(1));
                    continue;
                }
                int n2 = 1;
                char c2 = '\u0000';
                while (n2 < n && (c2 = string.charAt(n2++)) != c) {
                    this.putChar(c2);
                }
                if (c2 != c) {
                    throw new PrgException("Zeichenkette nicht geschlossen");
                }
                if (n2 >= n) continue;
                throw new PrgException("'" + string.charAt(n2) + "': Unerwartetes Zeichen hinter Zeichenkette");
            }
            this.putCode(this.getByte(string));
        } while (asmLine.hasMoreArgs());
    }

    private void parseDEFH(AsmLine asmLine) throws PrgException {
        do {
            String string;
            if ((string = asmLine.nextArg().toUpperString()) == null) continue;
            if (string.endsWith("H")) {
                string = string.substring(0, string.length() - 1);
            }
            try {
                int n = Integer.parseInt(string, 16);
                if (n < -128 || n > 255) {
                    this.putWarningOutOf8Bits();
                }
                this.putCode(n);
            }
            catch (NumberFormatException numberFormatException) {
                throw new PrgException("'" + string + "': Ung\u00fcltige Hexadezimalzahl");
            }
        } while (asmLine.hasMoreArgs());
    }

    private void parseDEFS(AsmLine asmLine) throws PrgException {
        do {
            AsmLabel asmLabel;
            String string;
            int n;
            if ((n = this.nextWordArg(asmLine)) > 0 && (string = asmLine.getLabel()) != null && (asmLabel = this.labels.get(string)) != null && asmLabel.getVarSize() <= 0) {
                asmLabel.setVarSize(n);
            }
            this.skipCode(n);
        } while (asmLine.hasMoreArgs());
    }

    private void parseDEFW(AsmLine asmLine) throws PrgException {
        do {
            this.putWord(this.nextWordArg(asmLine));
        } while (asmLine.hasMoreArgs());
    }

    private void parseELSE(AsmLine asmLine) throws PrgException {
        if (this.stack.isEmpty()) {
            throw new PrgException("ELSE ohne zugeh\u00f6riges IF...");
        }
        try {
            this.stack.peek().processELSE();
        }
        catch (EmptyStackException emptyStackException) {
            // empty catch block
        }
        asmLine.checkEOL();
    }

    private void parseEND(AsmLine asmLine) throws PrgException {
        this.endReached = true;
        asmLine.checkEOL();
    }

    private void parseENDIF(AsmLine asmLine) throws PrgException {
        if (this.stack.isEmpty()) {
            throw new PrgException("ENDIF ohne zugeh\u00f6riges IF...");
        }
        try {
            this.stack.pop();
        }
        catch (EmptyStackException emptyStackException) {
            // empty catch block
        }
        asmLine.checkEOL();
    }

    private void parseENT(AsmLine asmLine) throws PrgException {
        if (this.passNum == 1) {
            if (this.entryAddr != null) {
                throw new PrgException("Mehrfache ENT-Anweisungen nicht erlaubt");
            }
            this.entryAddr = this.curAddr;
        }
        asmLine.checkEOL();
    }

    private void parseEQU(AsmLine asmLine) throws PrgException {
        this.suppressLineAddr = true;
        String string = asmLine.getLabel();
        if (string != null) {
            AsmLabel asmLabel = this.labels.get(string);
            if (asmLabel != null) {
                String string2 = asmLine.nextArg().toString();
                Integer n = ExprParser.parse(string2, this.instBegAddr, this.labels, this.passNum == 2, this.options.getLabelsCaseSensitive());
                asmLabel.setLabelValue(n != null ? n : string2);
            }
        } else {
            throw new PrgException("EQU ohne Marke");
        }
        asmLine.checkEOL();
    }

    private void parseEX(AsmLine asmLine) throws PrgException {
        AsmArg asmArg = asmLine.nextArg();
        AsmArg asmArg2 = asmLine.nextArg();
        asmLine.checkEOL();
        if (asmArg.isIndirectSP()) {
            if (asmArg2.equalsUpper("HL")) {
                this.putCode(227);
            } else if (asmArg2.equalsUpper("IX")) {
                this.putCode(221);
                this.putCode(227);
            } else if (asmArg2.equalsUpper("IY")) {
                this.putCode(253);
                this.putCode(227);
            } else {
                this.throwNoSuchInstArgs();
            }
        } else if (asmArg.equalsUpper("AF")) {
            if (asmArg2.equalsUpper("AF'")) {
                this.zilogSyntax();
                this.putCode(8);
            } else {
                this.throwNoSuchInstArgs();
            }
        } else if (asmArg.equalsUpper("DE")) {
            if (asmArg2.equalsUpper("HL")) {
                this.putCode(235);
            } else {
                this.throwNoSuchInstArgs();
            }
        } else {
            this.throwNoSuchInstArgs();
        }
    }

    private void parseIFDEF(AsmLine asmLine) throws PrgException {
        AsmArg asmArg = asmLine.nextArg();
        this.stack.push(new AsmStackEntry(this.curSource.getLineNum(), this.labels.containsKey(this.options.getLabelsCaseSensitive() ? asmArg.toString() : asmArg.toUpperString())));
        asmLine.checkEOL();
    }

    private void parseIM(AsmLine asmLine) throws PrgException {
        String string = asmLine.nextArg().toString();
        asmLine.checkEOL();
        if (string.equals("0")) {
            this.putCode(237);
            this.putCode(70);
        } else if (string.equals("1")) {
            this.putCode(237);
            this.putCode(86);
        } else if (string.equals("2")) {
            this.putCode(237);
            this.putCode(94);
        } else {
            this.throwNoSuchInstArgs();
        }
    }

    private void parseIN(AsmLine asmLine) throws PrgException {
        AsmArg asmArg = asmLine.nextArg();
        if (asmArg.equalsUpper("A")) {
            if (asmLine.hasMoreArgs()) {
                AsmArg asmArg2 = asmLine.nextArg();
                if (asmArg2.isIndirectAddr()) {
                    this.putCode(219);
                    this.putCode(this.getByte(asmArg2.getIndirectText()));
                } else if (asmArg2.equalsUpper("(C)")) {
                    this.putCode(237);
                    this.putCode(120);
                } else {
                    this.throwNoSuchInstArgs();
                }
                asmLine.checkEOL();
                this.zilogSyntax();
            } else {
                this.putCode(237);
                this.putCode(120);
                this.robotronSyntax();
            }
        } else if (asmArg.isRegBtoL()) {
            if (asmLine.hasMoreArgs()) {
                if (!asmLine.nextArg().equalsUpper("(C)")) {
                    this.throwNoSuchInstArgs();
                }
                asmLine.checkEOL();
                this.zilogSyntax();
            } else {
                this.robotronSyntax();
            }
            this.putCode(237);
            this.putCode(0x40 | asmArg.getReg8Code() << 3);
        } else if (asmArg.equalsUpper("F")) {
            if (asmLine.hasMoreArgs()) {
                if (!asmLine.nextArg().equalsUpper("(C)")) {
                    this.throwNoSuchInstArgs();
                }
                asmLine.checkEOL();
            }
            this.putCode(237);
            this.putCode(112);
            this.undocInst();
        } else if (asmArg.equalsUpper("(C)")) {
            if (asmLine.hasMoreArgs()) {
                this.throwNoSuchInstArgs();
            }
            this.putCode(237);
            this.putCode(112);
            this.undocInst();
        } else {
            if (asmLine.hasMoreArgs()) {
                this.throwNoSuchInstArgs();
            }
            this.putCode(219);
            this.putCode(this.getByte(asmArg));
            this.robotronSyntax();
        }
    }

    private void parseINC_DEC(AsmLine asmLine, int n, int n2) throws PrgException {
        AsmArg asmArg = asmLine.nextArg();
        asmLine.checkEOL();
        if (asmArg.equalsUpper("BC")) {
            this.putCode(n2);
        } else if (asmArg.equalsUpper("DE")) {
            this.putCode(n2 + 16);
        } else if (asmArg.equalsUpper("HL")) {
            this.putCode(n2 + 32);
        } else if (asmArg.equalsUpper("SP")) {
            this.putCode(n2 + 48);
        } else if (asmArg.equalsUpper("IX")) {
            this.putCode(221);
            this.putCode(n2 + 32);
        } else if (asmArg.equalsUpper("IY")) {
            this.putCode(253);
            this.putCode(n2 + 32);
        } else if (asmArg.isRegAtoL()) {
            this.putCode(n | asmArg.getReg8Code() << 3);
        } else if (asmArg.isIndirectHL()) {
            this.putCode(n + 48);
        } else if (asmArg.equalsUpper("M")) {
            this.putCode(n + 48);
            this.robotronSyntax();
        } else if (asmArg.isIndirectIXDist()) {
            this.putCode(221);
            this.putCode(n + 48);
            this.putCode(this.getIndirectIXYDist(asmArg));
        } else if (asmArg.isIndirectIYDist()) {
            this.putCode(253);
            this.putCode(n + 48);
            this.putCode(this.getIndirectIXYDist(asmArg));
        } else if (asmArg.equalsUpper("IXH") || asmArg.equalsUpper("HX")) {
            this.putCode(221);
            this.putCode(n + 32);
            this.undocInst();
        } else if (asmArg.equalsUpper("IXL") || asmArg.equalsUpper("LX")) {
            this.putCode(221);
            this.putCode(n + 40);
            this.undocInst();
        } else if (asmArg.equalsUpper("IYH") || asmArg.equalsUpper("HY")) {
            this.putCode(253);
            this.putCode(n + 32);
            this.undocInst();
        } else if (asmArg.equalsUpper("IYL") || asmArg.equalsUpper("LY")) {
            this.putCode(253);
            this.putCode(n + 40);
            this.undocInst();
        } else {
            this.throwNoSuchInstArgs();
        }
    }

    private void parseINCLUDE(AsmLine asmLine) throws PrgException {
        File file = this.getIncludeFile(asmLine);
        if (this.curSource != this.mainSource) {
            throw new PrgException("In sich geschachtelte INCLUDE-Befehle nicht erlaubt");
        }
        asmLine.checkEOL();
        this.curSource = this.file2Source.get(file);
        if (this.curSource != null) {
            this.curSource.reset();
        } else {
            try {
                this.curSource = PrgSource.readFile(file);
                this.file2Source.put(file, this.curSource);
            }
            catch (IOException iOException) {
                String string = iOException.getMessage();
                if (string != null && string.trim().isEmpty()) {
                    string = null;
                }
                if (string == null) {
                    string = "Datei kann nicht ge\u00f6ffnet werden.";
                }
                throw new PrgException(string);
            }
        }
    }

    private void parseJMP(AsmLine asmLine) throws PrgException {
        AsmArg asmArg = asmLine.nextArg();
        asmLine.checkEOL();
        if (asmArg.isIndirectHL()) {
            this.putCode(233);
        } else if (asmArg.isIndirectIX()) {
            this.putCode(221);
            this.putCode(233);
        } else if (asmArg.isIndirectIY()) {
            this.putCode(253);
            this.putCode(233);
        } else {
            this.putCode(195);
            this.putWord(this.getWord(asmArg));
        }
    }

    private void parseJP(AsmLine asmLine) throws PrgException {
        AsmArg asmArg = asmLine.nextArg();
        if (asmLine.hasMoreArgs()) {
            AsmArg asmArg2 = asmLine.nextArg();
            asmLine.checkEOL();
            if (asmArg.equalsUpper("NZ")) {
                this.zilogSyntax();
                this.putCode(194);
                this.putWord(this.getWord(asmArg2));
            } else if (asmArg.equalsUpper("Z")) {
                this.zilogSyntax();
                this.putCode(202);
                this.putWord(this.getWord(asmArg2));
            } else if (asmArg.equalsUpper("NC")) {
                this.zilogSyntax();
                this.putCode(210);
                this.putWord(this.getWord(asmArg2));
            } else if (asmArg.equalsUpper("C")) {
                this.zilogSyntax();
                this.putCode(218);
                this.putWord(this.getWord(asmArg2));
            } else if (asmArg.equalsUpper("PO")) {
                this.zilogSyntax();
                this.putCode(226);
                this.putWord(this.getWord(asmArg2));
            } else if (asmArg.equalsUpper("PE")) {
                this.zilogSyntax();
                this.putCode(234);
                this.putWord(this.getWord(asmArg2));
            } else if (asmArg.equalsUpper("P")) {
                this.zilogSyntax();
                this.putCode(242);
                this.putWord(this.getWord(asmArg2));
            } else if (asmArg.equalsUpper("M")) {
                this.zilogSyntax();
                this.putCode(250);
                this.putWord(this.getWord(asmArg2));
            } else {
                this.throwNoSuchInstArgs();
            }
        } else if (asmArg.isIndirectHL()) {
            this.putCode(233);
        } else if (asmArg.isIndirectIX()) {
            this.putCode(221);
            this.putCode(233);
        } else if (asmArg.isIndirectIY()) {
            this.putCode(253);
            this.putCode(233);
        } else {
            this.putCode(195);
            this.putWord(this.getWord(asmArg));
        }
    }

    private void parseJR(AsmLine asmLine) throws PrgException {
        AsmArg asmArg = asmLine.nextArg();
        if (asmLine.hasMoreArgs()) {
            AsmArg asmArg2 = asmLine.nextArg();
            asmLine.checkEOL();
            if (asmArg.equalsUpper("NZ")) {
                int n = this.getAddrDiff(asmArg2);
                asmLine.checkEOL();
                this.putCode(32);
                this.putCode(n);
                this.zilogSyntax();
            } else if (asmArg.equalsUpper("Z")) {
                int n = this.getAddrDiff(asmArg2);
                asmLine.checkEOL();
                this.putCode(40);
                this.putCode(n);
                this.zilogSyntax();
            } else if (asmArg.equalsUpper("NC")) {
                int n = this.getAddrDiff(asmArg2);
                asmLine.checkEOL();
                this.putCode(48);
                this.putCode(n);
                this.zilogSyntax();
            } else if (asmArg.equalsUpper("C")) {
                int n = this.getAddrDiff(asmArg2);
                asmLine.checkEOL();
                this.putCode(56);
                this.putCode(n);
                this.zilogSyntax();
            } else {
                this.throwNoSuchInstArgs();
            }
        } else {
            int n = this.getAddrDiff(asmArg);
            this.putCode(24);
            this.putCode(n);
        }
    }

    private void parseLD(AsmLine asmLine) throws PrgException {
        AsmArg asmArg = asmLine.nextArg();
        AsmArg asmArg2 = asmLine.nextArg();
        asmLine.checkEOL();
        if (asmArg.equalsUpper("A")) {
            if (asmArg2.isRegAtoL()) {
                this.putCode(0x78 | asmArg2.getReg8Code());
            } else if (asmArg2.isIndirectHL()) {
                this.putCode(126);
            } else if (asmArg2.equalsUpper("M")) {
                this.robotronSyntax();
                this.putCode(126);
            } else if (asmArg2.isIndirectBC()) {
                this.putCode(10);
            } else if (asmArg2.isIndirectDE()) {
                this.putCode(26);
            } else if (asmArg2.isIndirectAddr()) {
                this.putCode(58);
                this.putWord(this.getWord(asmArg2.getIndirectText()));
            } else if (asmArg2.isIndirectIXDist()) {
                this.putCode(221);
                this.putCode(126);
                this.putCode(this.getIndirectIXYDist(asmArg2));
            } else if (asmArg2.isIndirectIYDist()) {
                this.putCode(253);
                this.putCode(126);
                this.putCode(this.getIndirectIXYDist(asmArg2));
            } else if (asmArg2.equalsUpper("I")) {
                this.putCode(237);
                this.putCode(87);
            } else if (asmArg2.equalsUpper("R")) {
                this.putCode(237);
                this.putCode(95);
            } else if (asmArg2.equalsUpper("IXH") || asmArg2.equalsUpper("HX")) {
                this.putCode(221);
                this.putCode(124);
            } else if (asmArg2.equalsUpper("IXL") || asmArg2.equalsUpper("LX")) {
                this.putCode(221);
                this.putCode(125);
            } else if (asmArg2.equalsUpper("IYH") || asmArg2.equalsUpper("HY")) {
                this.putCode(253);
                this.putCode(124);
            } else if (asmArg2.equalsUpper("IYL") || asmArg2.equalsUpper("LY")) {
                this.putCode(253);
                this.putCode(125);
            } else {
                this.putCode(62);
                this.putCode(this.getByte(asmArg2));
            }
        } else if (asmArg.isRegBtoL()) {
            if (asmArg2.isRegAtoL()) {
                this.putCode(0x40 | asmArg.getReg8Code() << 3 | asmArg2.getReg8Code());
            } else if (asmArg2.isIndirectHL()) {
                this.putCode(0x46 | asmArg.getReg8Code() << 3);
            } else if (asmArg2.equalsUpper("M")) {
                this.robotronSyntax();
                this.putCode(0x46 | asmArg.getReg8Code() << 3);
            } else if (asmArg2.isIndirectIXDist()) {
                this.putCode(221);
                this.putCode(0x46 | asmArg.getReg8Code() << 3);
                this.putCode(this.getIndirectIXYDist(asmArg2));
            } else if (asmArg2.isIndirectIYDist()) {
                this.putCode(253);
                this.putCode(0x46 | asmArg.getReg8Code() << 3);
                this.putCode(this.getIndirectIXYDist(asmArg2));
            } else if (asmArg2.equalsUpper("IXH") || asmArg2.equalsUpper("HX")) {
                this.putCode(221);
                this.putCode(0x44 | asmArg.getReg8Code() << 3);
                this.undocInst();
            } else if (asmArg2.equalsUpper("IXL") || asmArg2.equalsUpper("LX")) {
                this.putCode(221);
                this.putCode(0x45 | asmArg.getReg8Code() << 3);
                this.undocInst();
            } else if (asmArg2.equalsUpper("IYH") || asmArg2.equalsUpper("HY")) {
                this.putCode(253);
                this.putCode(0x44 | asmArg.getReg8Code() << 3);
                this.undocInst();
            } else if (asmArg2.equalsUpper("IYL") || asmArg2.equalsUpper("LY")) {
                this.putCode(253);
                this.putCode(0x45 | asmArg.getReg8Code() << 3);
                this.undocInst();
            } else {
                this.putCode(6 | asmArg.getReg8Code() << 3);
                this.putCode(this.getByte(asmArg2));
            }
        } else if (asmArg.isIndirectHL()) {
            if (asmArg2.isRegAtoL()) {
                this.putCode(0x70 | asmArg2.getReg8Code());
            } else {
                this.putCode(54);
                this.putCode(this.getByte(asmArg2));
            }
        } else if (asmArg.equalsUpper("M")) {
            this.robotronSyntax();
            if (asmArg2.isRegAtoL()) {
                this.putCode(0x70 | asmArg2.getReg8Code());
            } else {
                this.putCode(54);
                this.putCode(this.getByte(asmArg2));
            }
        } else if (asmArg.isIndirectBC()) {
            if (asmArg2.equalsUpper("A")) {
                this.putCode(2);
            } else {
                this.throwNoSuchInstArgs();
            }
        } else if (asmArg.isIndirectDE()) {
            if (asmArg2.equalsUpper("A")) {
                this.putCode(18);
            } else {
                this.throwNoSuchInstArgs();
            }
        } else if (asmArg.isIndirectAddr()) {
            if (asmArg2.equalsUpper("A")) {
                this.putCode(50);
                this.putWord(this.getWord(asmArg.getIndirectText()));
            } else if (asmArg2.equalsUpper("BC")) {
                this.putCode(237);
                this.putCode(67);
                this.putWord(this.getWord(asmArg.getIndirectText()));
            } else if (asmArg2.equalsUpper("DE")) {
                this.putCode(237);
                this.putCode(83);
                this.putWord(this.getWord(asmArg.getIndirectText()));
            } else if (asmArg2.equalsUpper("HL")) {
                this.putCode(34);
                this.putWord(this.getWord(asmArg.getIndirectText()));
            } else if (asmArg2.equalsUpper("SP")) {
                this.putCode(237);
                this.putCode(115);
                this.putWord(this.getWord(asmArg.getIndirectText()));
            } else if (asmArg2.equalsUpper("IX")) {
                this.putCode(221);
                this.putCode(34);
                this.putWord(this.getWord(asmArg.getIndirectText()));
            } else if (asmArg2.equalsUpper("IY")) {
                this.putCode(253);
                this.putCode(34);
                this.putWord(this.getWord(asmArg.getIndirectText()));
            } else {
                this.throwNoSuchInstArgs();
            }
        } else if (asmArg.isIndirectIXDist()) {
            this.parseLD8_IXYD_X(asmArg, asmArg2, 221);
        } else if (asmArg.isIndirectIYDist()) {
            this.parseLD8_IXYD_X(asmArg, asmArg2, 253);
        } else if (asmArg.equalsUpper("I") && asmArg2.equalsUpper("A")) {
            this.putCode(237);
            this.putCode(71);
        } else if (asmArg.equalsUpper("R") && asmArg2.equalsUpper("A")) {
            this.putCode(237);
            this.putCode(79);
        } else if (asmArg.equalsUpper("BC")) {
            this.parseLD16_BCDE_X(asmArg, asmArg2, 0);
        } else if (asmArg.equalsUpper("DE")) {
            this.parseLD16_BCDE_X(asmArg, asmArg2, 16);
        } else if (asmArg.equalsUpper("HL")) {
            this.parseLD16_HL_X(asmArg, asmArg2, -1);
        } else if (asmArg.equalsUpper("IX")) {
            this.parseLD16_HL_X(asmArg, asmArg2, 221);
        } else if (asmArg.equalsUpper("IY")) {
            this.parseLD16_HL_X(asmArg, asmArg2, 253);
        } else if (asmArg.equalsUpper("SP")) {
            if (asmArg2.equalsUpper("HL")) {
                this.putCode(249);
            } else if (asmArg2.equalsUpper("IX")) {
                this.putCode(221);
                this.putCode(249);
            } else if (asmArg2.equalsUpper("IY")) {
                this.putCode(253);
                this.putCode(249);
            } else if (asmArg2.isIndirectAddr()) {
                this.putCode(237);
                this.putCode(123);
                this.putWord(this.getWord(asmArg2.getIndirectText()));
            } else {
                this.putCode(49);
                this.putWord(this.getWord(asmArg2));
            }
        } else if (asmArg.isIndirectAddr()) {
            if (asmArg2.equalsUpper("BC")) {
                this.putCode(237);
                this.putCode(67);
                this.putWord(this.getWord(asmArg.getIndirectText()));
            } else if (asmArg2.equalsUpper("DE")) {
                this.putCode(237);
                this.putCode(83);
                this.putWord(this.getWord(asmArg.getIndirectText()));
            } else if (asmArg2.equalsUpper("HL")) {
                this.putCode(34);
                this.putWord(this.getWord(asmArg.getIndirectText()));
            } else if (asmArg2.equalsUpper("SP")) {
                this.putCode(237);
                this.putCode(115);
                this.putWord(this.getWord(asmArg.getIndirectText()));
            } else if (asmArg2.equalsUpper("IX")) {
                this.putCode(221);
                this.putCode(34);
                this.putWord(this.getWord(asmArg.getIndirectText()));
            } else if (asmArg2.equalsUpper("IY")) {
                this.putCode(253);
                this.putCode(34);
                this.putWord(this.getWord(asmArg.getIndirectText()));
            } else {
                this.throwNoSuchInstArgs();
            }
        } else if (asmArg.equalsUpper("IXH") || asmArg.equalsUpper("HX")) {
            this.parseLD_IXY8_X(asmArg2, 221, 0);
        } else if (asmArg.equalsUpper("IXL") || asmArg.equalsUpper("LX")) {
            this.parseLD_IXY8_X(asmArg2, 221, 8);
        } else if (asmArg.equalsUpper("IYH") || asmArg.equalsUpper("HY")) {
            this.parseLD_IXY8_X(asmArg2, 253, 0);
        } else if (asmArg.equalsUpper("IYL") || asmArg.equalsUpper("LY")) {
            this.parseLD_IXY8_X(asmArg2, 253, 8);
        } else {
            this.throwNoSuchInstArgs();
        }
    }

    private void parseLD8_IXYD_X(AsmArg asmArg, AsmArg asmArg2, int n) throws PrgException {
        this.putCode(n);
        if (asmArg2.isRegAtoL()) {
            this.putCode(0x70 | asmArg2.getReg8Code());
            this.putCode(this.getIndirectIXYDist(asmArg));
        } else {
            this.putCode(54);
            this.putCode(this.getIndirectIXYDist(asmArg));
            this.putCode(this.getByte(asmArg2));
        }
    }

    private void parseLD16_BCDE_X(AsmArg asmArg, AsmArg asmArg2, int n) throws PrgException {
        if (asmArg2.isIndirectAddr()) {
            int n2 = this.getWord(asmArg2.getIndirectText());
            this.putCode(237);
            this.putCode(75 + n);
            this.putWord(n2);
        } else {
            this.putCode(1 + n);
            this.putWord(this.getWord(asmArg2));
        }
    }

    private void parseLD16_HL_X(AsmArg asmArg, AsmArg asmArg2, int n) throws PrgException {
        if (n >= 0) {
            this.putCode(n);
        }
        if (asmArg2.isIndirectAddr()) {
            this.putCode(42);
            this.putWord(this.getWord(asmArg2.getIndirectText()));
        } else {
            this.putCode(33);
            this.putWord(this.getWord(asmArg2));
        }
    }

    private void parseLD_IXY8_X(AsmArg asmArg, int n, int n2) throws PrgException {
        if (asmArg.equalsUpper("A") || asmArg.equalsUpper("B") || asmArg.equalsUpper("C") || asmArg.equalsUpper("D") || asmArg.equalsUpper("E")) {
            this.putCode(n);
            this.putCode(96 + n2 + asmArg.getReg8Code());
        } else if (n == 221 && (asmArg.equalsUpper("IXH") || asmArg.equalsUpper("HX")) || n == 253 && (asmArg.equalsUpper("IYH") || asmArg.equalsUpper("HY"))) {
            this.putCode(n);
            this.putCode(100 + n2);
        } else if (n == 221 && (asmArg.equalsUpper("IXL") || asmArg.equalsUpper("LX")) || n == 253 && (asmArg.equalsUpper("IYL") || asmArg.equalsUpper("LY"))) {
            this.putCode(n);
            this.putCode(101 + n2);
        } else {
            this.putCode(n);
            this.putCode(38 + n2);
            this.putCode(this.getByte(asmArg));
        }
        this.undocInst();
    }

    private void parseIF(AsmLine asmLine, boolean bl) throws PrgException {
        Integer n = ExprParser.parse(asmLine.nextArg().toString(), this.instBegAddr, this.labels, true, this.options.getLabelsCaseSensitive());
        if (n == null) {
            throw new PrgException("Wert nicht ermittelbar (bei IF sind keine Vorw\u00e4rtsreferenzen auf Marken erlaubt.)");
        }
        this.stack.push(new AsmStackEntry(this.curSource.getLineNum(), bl == (n != 0)));
        asmLine.checkEOL();
    }

    private void parseIFNDEF(AsmLine asmLine) throws PrgException {
        AsmArg asmArg = asmLine.nextArg();
        this.stack.push(new AsmStackEntry(this.curSource.getLineNum(), !this.labels.containsKey(this.options.getLabelsCaseSensitive() ? asmArg.toString() : asmArg.toUpperString())));
        asmLine.checkEOL();
    }

    private void parseIfInPass(AsmLine asmLine, int n) throws PrgException {
        this.stack.push(new AsmStackEntry(this.curSource.getLineNum(), n == this.passNum));
        asmLine.checkEOL();
    }

    private void parseORG(AsmLine asmLine) throws PrgException {
        this.suppressLineAddr = true;
        int n = this.nextWordArg(asmLine);
        asmLine.checkEOL();
        if (n < this.curAddr) {
            this.orgOverlapped = true;
            throw new PrgException("Zur\u00fccksetzen des Addressz\u00e4hlers nicht erlaubt");
        }
        if (this.curAddr > 0 && n > this.curAddr) {
            this.skipCode(n - this.curAddr);
        }
        this.curAddr = n;
    }

    private void parseOUT(AsmLine asmLine) throws PrgException {
        AsmArg asmArg = asmLine.nextArg();
        if (asmArg.equalsUpper("(C)")) {
            AsmArg asmArg2 = asmLine.nextArg();
            asmLine.checkEOL();
            if (asmArg2.isRegAtoL()) {
                this.putCode(237);
                this.putCode(0x41 | asmArg2.getReg8Code() << 3);
            } else {
                this.throwNoSuchInstArgs();
            }
            this.zilogSyntax();
        } else if (asmArg.isRegAtoL()) {
            asmLine.checkEOL();
            this.putCode(237);
            this.putCode(0x41 | asmArg.getReg8Code() << 3);
            this.robotronSyntax();
        } else if (asmArg.isIndirectAddr()) {
            if (asmLine.hasMoreArgs()) {
                AsmArg asmArg3 = asmLine.nextArg();
                asmLine.checkEOL();
                if (!asmArg3.equalsUpper("A")) {
                    this.throwNoSuchInstArgs();
                }
                this.putCode(211);
                this.putCode(this.getByte(asmArg.getIndirectText()));
                this.zilogSyntax();
            } else {
                this.putCode(211);
                this.putCode(this.getByte(asmArg.getIndirectText()));
                this.undocSyntax();
            }
        } else {
            asmLine.checkEOL();
            this.putCode(211);
            this.putCode(this.getByte(asmArg));
            this.robotronSyntax();
        }
    }

    private void parsePUSH_POP(AsmLine asmLine, int n) throws PrgException {
        AsmArg asmArg = asmLine.nextArg();
        asmLine.checkEOL();
        if (asmArg.equalsUpper("BC")) {
            this.putCode(n);
        } else if (asmArg.equalsUpper("DE")) {
            this.putCode(n + 16);
        } else if (asmArg.equalsUpper("HL")) {
            this.putCode(n + 32);
        } else if (asmArg.equalsUpper("AF")) {
            this.putCode(n + 48);
        } else if (asmArg.equalsUpper("IX")) {
            this.putCode(221);
            this.putCode(n + 32);
        } else if (asmArg.equalsUpper("IY")) {
            this.putCode(253);
            this.putCode(n + 32);
        } else {
            this.throwNoSuchInstArgs();
        }
    }

    private void parseRET(AsmLine asmLine) throws PrgException {
        if (asmLine.hasMoreArgs()) {
            AsmArg asmArg = asmLine.nextArg();
            asmLine.checkEOL();
            if (asmArg.equalsUpper("NZ")) {
                this.zilogSyntax();
                this.putCode(192);
            } else if (asmArg.equalsUpper("Z")) {
                this.zilogSyntax();
                this.putCode(200);
            } else if (asmArg.equalsUpper("NC")) {
                this.zilogSyntax();
                this.putCode(208);
            } else if (asmArg.equalsUpper("C")) {
                this.zilogSyntax();
                this.putCode(216);
            } else if (asmArg.equalsUpper("PO")) {
                this.zilogSyntax();
                this.putCode(224);
            } else if (asmArg.equalsUpper("PE")) {
                this.zilogSyntax();
                this.putCode(232);
            } else if (asmArg.equalsUpper("P")) {
                this.zilogSyntax();
                this.putCode(240);
            } else if (asmArg.equalsUpper("M")) {
                this.zilogSyntax();
                this.putCode(248);
            } else {
                this.throwNoSuchInstArgs();
            }
        } else {
            this.putCode(201);
        }
    }

    private void parseRXD(AsmLine asmLine, int n) throws PrgException {
        if (asmLine.hasMoreArgs()) {
            AsmArg asmArg = asmLine.nextArg();
            asmLine.checkEOL();
            if (asmArg.isIndirectHL() || asmArg.equalsUpper("M")) {
                this.putCode(237);
                this.putCode(n);
                this.undocSyntax();
            } else {
                this.throwNoSuchInstArgs();
            }
        } else {
            this.putCode(237);
            this.putCode(n);
        }
    }

    private void parseRST(AsmLine asmLine) throws PrgException {
        int n = this.parseExpr(asmLine.nextArg().toString());
        asmLine.checkEOL();
        if ((n & 0xFFFFFFC7) != 0) {
            this.throwNoSuchInstArgs();
        }
        this.putCode(0xC7 | n);
    }

    private void parseBiOp8(AsmLine asmLine, int n) throws PrgException {
        AsmArg asmArg = asmLine.nextArg();
        if (asmLine.hasMoreArgs()) {
            AsmArg asmArg2 = asmLine.nextArg();
            asmLine.checkEOL();
            this.parseBiOp8(asmArg, asmArg2, n);
        } else {
            this.parseBiOp8(asmArg, n);
        }
    }

    private void parseBiOp8(AsmArg asmArg, AsmArg asmArg2, int n) throws PrgException {
        if (asmArg.equalsUpper("A")) {
            this.parseBiOp8Internal(asmArg2, n);
            if (n == 128 || n == 136 || n == 152) {
                this.zilogSyntax();
            } else {
                this.undocSyntax();
            }
        } else {
            this.throwNoSuchInstArgs();
        }
    }

    private void parseBiOp8(AsmArg asmArg, int n) throws PrgException {
        this.parseBiOp8Internal(asmArg, n);
        if (n == 128 || n == 136 || n == 152) {
            this.robotronSyntax();
        }
    }

    private void parseBiOp8Internal(AsmArg asmArg, int n) throws PrgException {
        if (asmArg.isRegAtoL()) {
            this.putCode(n + asmArg.getReg8Code());
        } else if (asmArg.isIndirectHL()) {
            this.putCode(n + 6);
        } else if (asmArg.equalsUpper("M")) {
            this.putCode(n + 6);
            this.robotronSyntax();
        } else if (asmArg.isIndirectIXDist()) {
            this.putCode(221);
            this.putCode(n + 6);
            this.putCode(this.getIndirectIXYDist(asmArg));
        } else if (asmArg.isIndirectIYDist()) {
            this.putCode(253);
            this.putCode(n + 6);
            this.putCode(this.getIndirectIXYDist(asmArg));
        } else if (asmArg.equalsUpper("IXH") || asmArg.equalsUpper("HX")) {
            this.putCode(221);
            this.putCode(n + 4);
            this.undocInst();
        } else if (asmArg.equalsUpper("IXL") || asmArg.equalsUpper("LX")) {
            this.putCode(221);
            this.putCode(n + 5);
            this.undocInst();
        } else if (asmArg.equalsUpper("IYH") || asmArg.equalsUpper("HY")) {
            this.putCode(253);
            this.putCode(n + 4);
            this.undocInst();
        } else if (asmArg.equalsUpper("IYL") || asmArg.equalsUpper("LY")) {
            this.putCode(253);
            this.putCode(n + 5);
            this.undocInst();
        } else {
            this.putCode(n + 70);
            this.putCode(this.getByte(asmArg));
        }
    }

    private void parseInstDirectAddr(AsmLine asmLine, int n) throws PrgException {
        int n2 = this.nextWordArg(asmLine);
        asmLine.checkEOL();
        this.putCode(n);
        this.putWord(n2);
    }

    private void parseRotShift(AsmLine asmLine, int n) throws PrgException {
        AsmArg asmArg = asmLine.nextArg();
        if (asmLine.hasMoreArgs()) {
            AsmArg asmArg2 = asmLine.nextArg();
            asmLine.checkEOL();
            if (asmArg.isIndirectIXDist() && asmArg2.isRegAtoL()) {
                this.putCode(221);
                this.putCode(203);
                this.putCode(this.getIndirectIXYDist(asmArg));
                this.putCode(n | asmArg2.getReg8Code());
                if (n != 48) {
                    this.undocInst();
                }
            } else {
                this.throwNoSuchInstArgs();
            }
        } else if (asmArg.isRegAtoL()) {
            this.putCode(203);
            this.putCode(n + asmArg.getReg8Code());
        } else if (asmArg.isIndirectHL()) {
            this.putCode(203);
            this.putCode(n + 6);
        } else if (asmArg.equalsUpper("M")) {
            this.putCode(203);
            this.putCode(n + 6);
            this.robotronSyntax();
        } else if (asmArg.isIndirectIXDist()) {
            this.putCode(221);
            this.putCode(203);
            this.putCode(this.getIndirectIXYDist(asmArg));
            this.putCode(n + 6);
        } else if (asmArg.isIndirectIYDist()) {
            this.putCode(253);
            this.putCode(203);
            this.putCode(this.getIndirectIXYDist(asmArg));
            this.putCode(n + 6);
        } else {
            this.throwNoSuchInstArgs();
        }
    }

    private void parseSingleBit(AsmLine asmLine, int n) throws PrgException {
        AsmArg asmArg = asmLine.nextArg();
        AsmArg asmArg2 = asmLine.nextArg();
        AsmArg asmArg3 = null;
        if (asmLine.hasMoreArgs()) {
            asmArg3 = asmLine.nextArg();
            asmLine.checkEOL();
        }
        int n2 = this.parseExpr(asmArg.toString()) << 3;
        if (asmArg3 != null) {
            if ((n == 128 || n == 192) && asmArg3.isRegAtoL()) {
                if (asmArg2.isIndirectIXDist()) {
                    this.putCode(221);
                    this.putCode(203);
                    this.putCode(this.getIndirectIXYDist(asmArg2));
                    this.putCode(n + asmArg3.getReg8Code() + n2);
                    this.undocInst();
                } else if (asmArg2.isIndirectIYDist()) {
                    this.putCode(253);
                    this.putCode(203);
                    this.putCode(this.getIndirectIXYDist(asmArg2));
                    this.putCode(n + asmArg3.getReg8Code() + n2);
                    this.undocInst();
                } else {
                    this.throwNoSuchInstArgs();
                }
            } else {
                this.throwNoSuchInstArgs();
            }
        } else if (asmArg2.isRegAtoL()) {
            this.putCode(203);
            this.putCode(n + asmArg2.getReg8Code() + n2);
        } else if (asmArg2.isIndirectHL()) {
            this.putCode(203);
            this.putCode(n + 6 + n2);
        } else if (asmArg2.equalsUpper("M")) {
            this.putCode(203);
            this.putCode(n + 6 + n2);
            this.robotronSyntax();
        } else if (asmArg2.isIndirectIXDist()) {
            this.putCode(221);
            this.putCode(203);
            this.putCode(this.getIndirectIXYDist(asmArg2));
            this.putCode(n + 6 + n2);
        } else if (asmArg2.isIndirectIYDist()) {
            this.putCode(253);
            this.putCode(203);
            this.putCode(this.getIndirectIXYDist(asmArg2));
            this.putCode(n + 6 + n2);
        } else {
            this.throwNoSuchInstArgs();
        }
    }

    private int nextWordArg(AsmLine asmLine) throws PrgException {
        return this.getWord(asmLine.nextArg());
    }

    private int getAddrDiff(AsmArg asmArg) throws PrgException {
        int n = 0;
        if (this.passNum == 2) {
            String string = asmArg.toString();
            if (string.endsWith("-#")) {
                string = string.substring(0, string.length() - 2);
            }
            if ((n = this.getWord(string) - (this.curAddr + 2 & 0xFFFF)) < -128 || n > 127) {
                this.relJumpsTooLong = true;
                throw new PrgException("Relative Sprungdistanz zu gro\u00df");
            }
        }
        return n;
    }

    private File getIncludeFile(AsmLine asmLine) throws PrgException {
        int n;
        String string = null;
        String string2 = asmLine.nextArg().toString();
        if (string2 != null && (n = string2.length()) > 0) {
            char c = string2.charAt(0);
            if (c == '\'' || c == '\"') {
                if (n < 2 || string2.charAt(n - 1) != c) {
                    throw new PrgException("Dateiname nicht mit " + c + " abgeschlossen");
                }
                if (n > 2) {
                    string = string2.substring(1, n - 1);
                }
            } else {
                string = string2;
            }
        }
        if (string == null) {
            throw new PrgException("Dateiname erwartet");
        }
        return PrgSource.getIncludeFile(this.curSource, string);
    }

    private int getIndirectIXYDist(AsmArg asmArg) throws PrgException {
        int n = 0;
        String string = asmArg.getIndirectIXYDist();
        if (string != null && !string.isEmpty()) {
            n = this.parseExpr(string);
            if (n < -128 || n > 255) {
                throw new PrgException("Distanzangabe zu gro\u00df");
            }
            if (string.startsWith("+") && n > 127) {
                this.putWarning("Distanz ist negativ (r\u00fcchw\u00e4rts) obwohl positiv angegeben");
            }
        }
        return n;
    }

    private int getByte(AsmArg asmArg) throws PrgException {
        return this.getByte(asmArg.toString());
    }

    private int getByte(String string) throws PrgException {
        int n = this.parseExpr(string);
        if (n < -128 || n > 255) {
            this.putWarningOutOf8Bits();
        }
        return n;
    }

    private int getWord(AsmArg asmArg) throws PrgException {
        return this.getWord(asmArg.toString());
    }

    private int getWord(String string) throws PrgException {
        int n = this.parseExpr(string);
        this.checkPrint16BitWarning(n);
        return n;
    }

    private void putChar(char c) throws PrgException {
        if (c > '\u00ff') {
            throw new PrgException(String.format("'%c': 16-Bit-Unicodezeichen nicht erlaubt", Character.valueOf(c)));
        }
        if (this.options.getWarnNonAsciiChars() && (c < ' ' || c > '\u007f')) {
            this.putWarning(String.format("'%c': kein ASCII-Zeichen, entsprechend Unicode mit %02XH \u00fcbersetzt", Character.valueOf(c), (int)c));
        }
        this.putCode(c);
    }

    private void putCode(int n) throws PrgException {
        if (this.codeBuf != null && this.passNum == 2) {
            if (this.begAddr < 0) {
                this.begAddr = this.curAddr;
            } else if (this.endAddr + 1 < this.curAddr) {
                int n2 = this.curAddr - this.endAddr - 1;
                for (int i = 0; i < n2; ++i) {
                    this.codeBuf.write(0);
                }
            }
            this.endAddr = this.curAddr;
            this.codeBuf.write(n);
        }
        ++this.curAddr;
        this.checkAddr();
    }

    private void putWord(int n) throws PrgException {
        this.putCode(n);
        this.putCode(n >> 8);
    }

    private void skipCode(int n) throws PrgException {
        this.curAddr += n;
        this.checkAddr();
    }

    private void robotronMnemonic() {
        if (this.options.getAsmSyntax() == Syntax.ZILOG_ONLY) {
            this.putWarning("Robotron-Mnemonik");
        }
    }

    private void robotronSyntax() {
        if (this.options.getAsmSyntax() == Syntax.ZILOG_ONLY) {
            this.putWarning("Robotron-Syntax");
        }
    }

    private void zilogSyntax() {
        if (this.options.getAsmSyntax() == Syntax.ROBOTRON_ONLY) {
            this.putWarning("Zilog-Syntax");
        }
    }

    private void zilogMnemonic() {
        if (this.options.getAsmSyntax() == Syntax.ROBOTRON_ONLY) {
            this.putWarning("Zilog-Mnemonik");
        }
    }

    private void undocInst() {
        if (!this.options.getAllowUndocInst()) {
            this.putWarning("Undokumentierter Befehl");
        }
    }

    private void undocSyntax() {
        if (!this.options.getAllowUndocInst()) {
            this.putWarning("Undokumentierte Syntax");
        }
    }

    private void throwNoSuchInstArgs() throws PrgException {
        throw new PrgException("Die Anweisung existiert nicht f\u00fcr die angegebenen Argumente.");
    }

    private void printLabels() {
        boolean bl = true;
        AsmLabel[] asmLabelArray = this.getSortedLabels();
        if (asmLabelArray != null) {
            for (int i = 0; i < asmLabelArray.length; ++i) {
                if (bl) {
                    bl = false;
                    this.appendToOutLog("\nMarkentabelle:\n");
                }
                String string = "????";
                Object object = asmLabelArray[i].getLabelValue();
                if (object != null && object instanceof Integer) {
                    string = String.format("%04X", (Integer)object & 0xFFFF);
                }
                this.appendToOutLog(String.format("    %s   %s\n", string, asmLabelArray[i].getLabelName()));
            }
        }
        if (bl) {
            this.appendToOutLog("Markentabelle ist leer.\n");
        } else {
            this.appendToOutLog("\n");
        }
    }

    private void putWarningOutOf8Bits() {
        this.putWarning("Numerischer Wert au\u00dferhalb 8-Bit-Bereich: Bits gehen verloren");
    }

    private boolean writeCodeToFile(String string, boolean bl) {
        boolean bl2 = false;
        byte[] byArray = this.codeOut;
        if (byArray != null && byArray.length == 0) {
            byArray = null;
        }
        if (byArray != null) {
            File file = this.options.getCodeFile();
            if (file != null) {
                int n = this.begAddr;
                Integer n2 = this.entryAddr;
                if (n2 == null) {
                    n2 = n;
                }
                FileFormat fileFormat = FileFormat.BIN;
                String string2 = null;
                String string3 = "";
                String string4 = file.getName();
                if (string4 != null) {
                    int n3;
                    String string5 = string4.toUpperCase();
                    if (string3.isEmpty() && (n3 = string4.lastIndexOf(46)) > 0) {
                        string3 = string5.substring(0, n3);
                    }
                    if (string5.endsWith(".HEX")) {
                        fileFormat = FileFormat.INTELHEX;
                    } else if (string5.endsWith(".KCC") || string5.endsWith(".KCM")) {
                        fileFormat = FileFormat.KCC;
                    } else if (string5.endsWith(".TAP")) {
                        if (bl) {
                            fileFormat = FileFormat.KCTAP_Z9001;
                            if (n2 != null) {
                                string2 = "COM";
                            }
                        } else {
                            fileFormat = FileFormat.KCTAP_KC85;
                        }
                    } else if (string5.endsWith(".Z80")) {
                        fileFormat = FileFormat.HEADERSAVE;
                        String string6 = string2 = n2 != null ? "C" : "M";
                    }
                    if ((fileFormat.equals(FileFormat.KCC) || fileFormat.equals(FileFormat.KCTAP_KC85) || fileFormat.equals(FileFormat.KCTAP_Z9001)) && string != null && !string.isEmpty()) {
                        string3 = string;
                    }
                }
                try {
                    if (this.interactive) {
                        this.appendToOutLog("Speichere Programmcode in Datei '" + file.getPath() + "'...\n");
                    }
                    FileSaver.saveFile(file, fileFormat, new LoadData(byArray, 0, byArray.length, this.begAddr, -1, fileFormat), this.begAddr, n + byArray.length - 1, false, this.begAddr, n2, string3, string2, null);
                    bl2 = true;
                }
                catch (IOException iOException) {
                    StringBuilder stringBuilder = new StringBuilder(256);
                    stringBuilder.append("Ein-/Ausgabefehler");
                    String string7 = iOException.getMessage();
                    if (string7 != null && !string7.isEmpty()) {
                        stringBuilder.append(":\n");
                        stringBuilder.append(string7);
                    }
                    stringBuilder.append('\n');
                    this.appendToErrLog(stringBuilder.toString());
                }
            } else {
                this.appendToErrLog("Programmcode kann nicht gespeichert werden,\nda kein Dateiname ausgew\u00e4hlt wurde.\n");
            }
        } else {
            this.appendToErrLog("Programmcode kann nicht gespeichert werden,\nda kein einziges Byte erzeugt wurde.\n");
        }
        return bl2;
    }

    public static enum Syntax {
        ALL,
        ZILOG_ONLY,
        ROBOTRON_ONLY;

    }
}

