// Runge-Kutta RK4 algorithm
// https://github.com/skeeto/double-pendulum

float G = -1.0; // gravitational acceleration
float M = 2.0; // mass
float L; /// 0.9*height*3; //300.0; // length of arm
float L2 = 1.2; 
float rubberBand = 0.5; // No rubber banding at 0.5

float[] derivative(float a1, float a2, float p1, float p2) {
    L = height/2*0.8;
    float ml2 = M * L * L;
    float cos12 = (float) Math.cos(a1 - a2);
    float sin12 = (float) Math.sin(a1 - a2);
    float da1 = 6 / ml2 * (2 * p1 - 3 * cos12 * p2) / (16 - 9 * cos12 * cos12);
    float da2 = 6 / ml2 * (8 * p2 - 3 * cos12 * p1) / (16 - 9 * cos12 * cos12);
    float dp1 = ml2 / -2 * (+da1 * da2 * sin12 + 3 * G / L * ((float) Math.sin(a1)));
    float dp2 = ml2 / -2 * (-da1 * da2 * sin12 + 3 * G / L * ((float) Math.sin(a2)));
    float[] returnArray = {da1, da2, dp1, dp2};
    return returnArray;
}

// Update pendulum by timestep
float[] rk4(float k1a1, float k1a2, float k1p1, float k1p2, float dt) {
    float[] der1 = derivative(k1a1, k1a2, k1p1, k1p2);
    float k1da1 = der1[0];
    float k1da2 = der1[1];
    float k1dp1 = der1[2];
    float k1dp2 = der1[3];

    float k2a1 = k1a1 + k1da1 * dt / 2;
    float k2a2 = k1a2 + k1da2 * dt / 2;
    float k2p1 = k1p1 + k1dp1 * dt / 2;
    float k2p2 = k1p2 + k1dp2 * dt / 2;

    float[] der2 = derivative(k2a1, k2a2, k2p1, k2p2);
    float k2da1 = der2[0];
    float k2da2 = der2[1];
    float k2dp1 = der2[2];
    float k2dp2 = der2[3];

    float k3a1 = k1a1 + k2da1 * dt / 2;
    float k3a2 = k1a2 + k2da2 * dt / 2;
    float k3p1 = k1p1 + k2dp1 * dt / 2;
    float k3p2 = k1p2 + k2dp2 * dt / 2;

    float[] der3  = derivative(k3a1, k3a2, k3p1, k3p2);
    float k3da1 = der3[0];
    float k3da2 = der3[1];
    float k3dp1 = der3[2];
    float k3dp2 = der3[3];

    float k4a1 = k1a1 + k3da1 * dt;
    float k4a2 = k1a2 + k3da2 * dt;
    float k4p1 = k1p1 + k3dp1 * dt;
    float k4p2 = k1p2 + k3dp2 * dt;

    float[] der4 = derivative(k4a1, k4a2, k4p1, k4p2);
    float k4da1 = der4[0];
    float k4da2 = der4[1];
    float k4dp1 = der4[2];
    float k4dp2 = der4[3];

    float[] returnArray = {
    k1a1 + (k1da1 + 2*k2da1 + 2*k3da1 + k4da1) * dt / 6,
    k1a2 + (k1da2 + 2*k2da2 + 2*k3da2 + k4da2) * dt / 6,
    k1p1 + (k1dp1 + 2*k2dp1 + 2*k3dp1 + k4dp1) * dt / 6,
    k1p2 + (k1dp2 + 2*k2dp2 + 2*k3dp2 + k4dp2) * dt / 6
    };
        
    return returnArray;
}


float[] firstJoint(float t1, float t2) {
  float x = (L/2)*((float) Math.sin(t1));
  float y = (-L/2)*((float) Math.cos(t1));
  float[] returnArray = {x,y};
  return returnArray;
}

float[] secondJoint(float t1, float t2) {
  float x = L*rubberBand*((float) Math.sin(t1) + L2 * (float) Math.sin(t2));
  float y = -L*rubberBand*((float) Math.cos(t1) + L2 * (float) Math.cos(t2));
  float[] returnArray = {x,y};
  return returnArray;
}
