//>>> _using
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SharpDX;
using SharpDX.Direct3D11;
using SharpDX.Windows;
//<<< _using

using System.Runtime.InteropServices;
using Framefield.Core.Rendering;
using SharpDX.DXGI;
using SharpDX.D3DCompiler;
using Buffer = SharpDX.Direct3D11.Buffer;
using System.CodeDom.Compiler;
using System.Diagnostics;


namespace Framefield.Core.IDa83e988e_f959_4453_8940_0bc65cd08ab7
{
    public class Class_FluidSolver : FXSourceCodeFunction, Framefield.Core.OperatorPartTraits.ITimeAccessor
    {

        //>>> _inputids
        private enum InputId
        {
            Code = 0,
            BoundryVolume = 1
        }
        //<<< _inputids

        public override void Dispose()
        {
            for (int i = 0; i < 3; ++i)
            {
                Utilities.DisposeObj(ref _velocityVolumes[i]);
                Utilities.DisposeObj(ref _velocitySRVs[i]);
                Utilities.DisposeObj(ref _velocityUAVs[i]);
            }

            for (int i = 0; i < 2; ++i)
            {
                Utilities.DisposeObj(ref _pressureVolumes[i]);
                Utilities.DisposeObj(ref _pressureSRVs[i]);
                Utilities.DisposeObj(ref _pressureUAVs[i]);
            }

            Utilities.DisposeObj(ref _divergenceVolume);
            Utilities.DisposeObj(ref _divergenceSRV);
            Utilities.DisposeObj(ref _divergenceUAV);
            Utilities.DisposeObj(ref _velocitySizeVolume);
            Utilities.DisposeObj(ref _velocitySizeSRV);
            Utilities.DisposeObj(ref _velocitySizeUAV);

            Utilities.DisposeObj(ref _linearSamplerState);
            Utilities.DisposeObj(ref _constBuffer);

            Utilities.DisposeObj(ref _visuTexture);
            Utilities.DisposeObj(ref _csAdvection);
            Utilities.DisposeObj(ref _csAdvectionBackwards);
            Utilities.DisposeObj(ref _csAdvectionMacCormack);
            Utilities.DisposeObj(ref _csDrawSphere);
            Utilities.DisposeObj(ref _csDivergence);
            Utilities.DisposeObj(ref _csJacobi);
            Utilities.DisposeObj(ref _csProjection);
            Utilities.DisposeObj(ref _csRender);
            Utilities.DisposeObj(ref _csVisu);
            base.Dispose();
        }


        [StructLayout(LayoutKind.Explicit, Size = 144)]
        public struct ConstBufferLayout
        {
            [FieldOffset(0)] 
            public Vector4 Dim;
            [FieldOffset(16)] 
            public int width;
            [FieldOffset(20)] 
            public int height;
            [FieldOffset(24)] 
            public int viewSlice;
            [FieldOffset(28)] 
            public int viewOrientation;
            [FieldOffset(32)] 
            public Vector4 mouse;
            [FieldOffset(48)] 
            public Vector4 dragDirection;
            [FieldOffset(64)] 
            public Matrix orientation;
            [FieldOffset(128)] 
            public float zoom;
            [FieldOffset(132)] 
            public int smoky;
        }

        private readonly int SIZE = 256;

        protected bool BuildRenderTarget(OperatorPartContext context)
        {
            if (_visuTexture == null ||
                (int) context.Viewport.Width != _visuTexture.Description.Width ||
                (int) context.Viewport.Height != _visuTexture.Description.Height)
            {
                var uavVisuDesc = new Texture2DDescription
                                      {
                                          BindFlags = BindFlags.ShaderResource | BindFlags.UnorderedAccess,
                                          Format = Format.R8G8B8A8_UNorm,
                                          Width = (int) context.Viewport.Width,
                                          Height = (int) context.Viewport.Height,
                                          MipLevels = 1,
                                          SampleDescription = new SampleDescription(1, 0),
                                          Usage = ResourceUsage.Default,
                                          OptionFlags = ResourceOptionFlags.None,
                                          CpuAccessFlags = CpuAccessFlags.None,
                                          ArraySize = 1
                                      };
                Utilities.DisposeObj(ref _visuTexture);
                Utilities.DisposeObj(ref _visuUAV);
                _visuTexture = new Texture2D(context.D3DDevice, uavVisuDesc);
                _visuUAV = new UnorderedAccessView(context.D3DDevice, _visuTexture);
            }

            if (_velocityVolumes[0] != null)
            {
                return false;
            }

            // generate velocity volumes   
            var volumeDesc = new Texture3DDescription
                                 {
                                     BindFlags = BindFlags.ShaderResource | BindFlags.UnorderedAccess,
                                     Format = Format.R16G16B16A16_Float,
                                     Width = SIZE,
                                     Height = SIZE,
                                     Depth = SIZE,
                                     MipLevels = 1,
                                     Usage = ResourceUsage.Default,
                                     OptionFlags = ResourceOptionFlags.None,
                                     CpuAccessFlags = CpuAccessFlags.None,
                                 };

            for (int i = 0; i < 4; ++i)
            {
                Utilities.DisposeObj(ref _velocityVolumes[i]);
                Utilities.DisposeObj(ref _velocitySRVs[i]);
                Utilities.DisposeObj(ref _velocityUAVs[i]);
                _velocityVolumes[i] = new Texture3D(context.D3DDevice, volumeDesc);
                _velocitySRVs[i] = new ShaderResourceView(context.D3DDevice, _velocityVolumes[i]);
                _velocityUAVs[i] = new UnorderedAccessView(context.D3DDevice, _velocityVolumes[i]);
            }

            // generate pressure volumes
            volumeDesc.Width = SIZE/4;

            for (int i = 0; i < 2; ++i)
            {
                Utilities.DisposeObj(ref _pressureVolumes[i]);
                Utilities.DisposeObj(ref _pressureSRVs[i]);
                Utilities.DisposeObj(ref _pressureUAVs[i]);
                _pressureVolumes[i] = new Texture3D(context.D3DDevice, volumeDesc);
                _pressureSRVs[i] = new ShaderResourceView(context.D3DDevice, _pressureVolumes[i]);
                _pressureUAVs[i] = new UnorderedAccessView(context.D3DDevice, _pressureVolumes[i]);
            }

            // generate divergence volume
            Utilities.DisposeObj(ref _divergenceVolume);
            Utilities.DisposeObj(ref _divergenceSRV);
            Utilities.DisposeObj(ref _divergenceUAV);
            _divergenceVolume = new Texture3D(context.D3DDevice, volumeDesc);
            _divergenceSRV = new ShaderResourceView(context.D3DDevice, _divergenceVolume);
            _divergenceUAV = new UnorderedAccessView(context.D3DDevice, _divergenceVolume);


            // velocity size (length velocity vector)
            volumeDesc.Format = Format.R16_Float;
            volumeDesc.Width = SIZE;

            Utilities.DisposeObj(ref _velocitySizeVolume);
            Utilities.DisposeObj(ref _velocitySizeSRV);
            Utilities.DisposeObj(ref _velocitySizeUAV);
            _velocitySizeVolume = new Texture3D(context.D3DDevice, volumeDesc);
            _velocitySizeSRV = new ShaderResourceView(context.D3DDevice, _velocitySizeVolume);
            _velocitySizeUAV = new UnorderedAccessView(context.D3DDevice, _velocitySizeVolume);

            // create sampler state
            var samplerDesc = new SamplerStateDescription()
                                  {
                                      Filter = Filter.MinMagMipLinear,
                                      AddressU = TextureAddressMode.Border,
                                      AddressV = TextureAddressMode.Border,
                                      AddressW = TextureAddressMode.Border,
                                      BorderColor = new Color4(0, 0, 0, 0),
                                      MipLodBias = 0,
                                      ComparisonFunction = Comparison.Never,
                                      MaximumAnisotropy = 16,
                                      MinimumLod = 0,
                                      MaximumLod = Single.MaxValue
                                  };
            Utilities.DisposeObj(ref _linearSamplerState);
            _linearSamplerState = new SamplerState(context.D3DDevice, samplerDesc);

            return true;
        }

        static int _counter = 0;

        private void SetupConstBuffer(OperatorPartContext context)
        {
            var mc = new ConstBufferLayout();

            mc.Dim[0] = SIZE;
            mc.Dim[1] = SIZE;
            mc.Dim[2] = SIZE;

            mc.width  = (int) context.Viewport.Width;
            mc.height = (int) context.Viewport.Height;

            mc.viewSlice = SIZE/2;
            mc.viewOrientation = 0;
  
            var mod = _counter++ % 80;
            var cos = (float)Math.Cos(_counter * 2.0f * 3.1415f / 180.0f);
            var sin = (float)Math.Sin(_counter * 2.0f * 3.1415f / 180.0f);
            mc.mouse[0] = mc.width/2.0f;// + cos*mc.width/6.9f;
            mc.mouse[1] = mc.height/2.0f;// + sin*mc.height/6.5f;
            mc.mouse[2] = SIZE/2; 
            mc.mouse[3] = 10 ; // radius in voxels;
 
            mc.dragDirection[0] = cos;//-cos*0.5f;
            mc.dragDirection[1] = sin;//(mod < 20) ? -2 : 0;
            mc.dragDirection[2] = 0;
            mc.orientation = Matrix.Identity;//RotationY(context.Time);
            mc.zoom = 1.2f;
            mc.smoky = 0;

            BaseRenderer.SetupConstBufferForCS(context, mc, ref _constBuffer, 0);
        }

        void RunShader(DeviceContext immediateContext, ComputeShader shader, ShaderResourceView in0, ShaderResourceView in1, ShaderResourceView in2, UnorderedAccessView out0, UnorderedAccessView out1, int dx, int dy, int dz)
        {
            // Set volumes as textures for cached read
            immediateContext.ComputeShader.SetShaderResource(0, in0);
            immediateContext.ComputeShader.SetShaderResource(1, in1);
            immediateContext.ComputeShader.SetShaderResource(2, in2);

            // Set unordered access views for writees
            immediateContext.ComputeShader.SetUnorderedAccessView(0, out0);
            immediateContext.ComputeShader.SetUnorderedAccessView(1, out1);

           // Set compute shader
            immediateContext.ComputeShader.Set(shader);

            // Run compute shader
            immediateContext.Dispatch(dx, dy, dz);

            immediateContext.ComputeShader.SetShaderResource(0, null);
            immediateContext.ComputeShader.SetShaderResource(1, null);
            immediateContext.ComputeShader.SetShaderResource(2, null);
            immediateContext.ComputeShader.SetUnorderedAccessView(0, null);
            immediateContext.ComputeShader.SetUnorderedAccessView(1, null);
        }

        private Stopwatch _stopwatch = new Stopwatch();
        public override OperatorPartContext Eval(OperatorPartContext context, List<OperatorPart> inputs, int outputIdx) 
        {
            //>>> _params
            var Code = inputs[(int)InputId.Code].Eval(context).Text;
            var BoundryVolume = inputs[(int)InputId.BoundryVolume].Eval(context).Volume;
            //<<< __params

            if (!Changed)
            {
                context.Image = _visuTexture;
                context.Volume = _velocitySizeVolume;
                return context;
            }

            if (_csAdvection == null)
            {
                Utilities.DisposeObj(ref _csAdvection);
                Utilities.DisposeObj(ref _csDrawSphere);
                Utilities.DisposeObj(ref _csDivergence);
                Utilities.DisposeObj(ref _csJacobi);
                Utilities.DisposeObj(ref _csProjection);
                Utilities.DisposeObj(ref _csRender);
                Utilities.DisposeObj(ref _csVisu);
                var errors = new CompilerErrorCollection();
                try
                {
                    using (var bytecode = ShaderBytecode.Compile(GetCode(0), "CSAdvection", "cs_5_0", ShaderFlags.Debug))
                        _csAdvection = new ComputeShader(D3DDevice.Device, bytecode);
                    using (var bytecode = ShaderBytecode.Compile(GetCode(0), "CSAdvectionBackwards", "cs_5_0", ShaderFlags.Debug))
                        _csAdvectionBackwards = new ComputeShader(D3DDevice.Device, bytecode);
                    using (var bytecode = ShaderBytecode.Compile(GetCode(0), "CSAdvectionMacCormack", "cs_5_0", ShaderFlags.Debug))
                        _csAdvectionMacCormack = new ComputeShader(D3DDevice.Device, bytecode);
                    using (var bytecode = ShaderBytecode.Compile(GetCode(0), "CSDrawSphere", "cs_5_0", ShaderFlags.Debug))
                        _csDrawSphere = new ComputeShader(D3DDevice.Device, bytecode);
                    using (var bytecode = ShaderBytecode.Compile(GetCode(0), "CSDivergence", "cs_5_0", ShaderFlags.Debug))
                        _csDivergence = new ComputeShader(D3DDevice.Device, bytecode);
                    using (var bytecode = ShaderBytecode.Compile(GetCode(0), "CSJacobi", "cs_5_0", ShaderFlags.Debug))
                        _csJacobi = new ComputeShader(D3DDevice.Device, bytecode);
                    using (var bytecode = ShaderBytecode.Compile(GetCode(0), "CSProjection", "cs_5_0", ShaderFlags.Debug))
                        _csProjection = new ComputeShader(D3DDevice.Device, bytecode);
                    using (var bytecode = ShaderBytecode.Compile(GetCode(0), "CSRender", "cs_5_0", ShaderFlags.Debug))
                        _csRender = new ComputeShader(D3DDevice.Device, bytecode);
/*                        
                    using (var bytecode = ShaderBytecode.CompileFromFile("fluidsolver.hlsl", "CSAdvection", "cs_5_0", ShaderFlags.Debug))
                        _csAdvection = new ComputeShader(D3DDevice.Device, bytecode);
                    _csAdvection.DebugName = "CSAdvection4123";
                    using (var bytecode = ShaderBytecode.CompileFromFile("fluidsolver.hlsl", "CSAdvectionBackwards", "cs_5_0", ShaderFlags.Debug))
                        _csAdvectionBackwards = new ComputeShader(D3DDevice.Device, bytecode);
                    using (var bytecode = ShaderBytecode.CompileFromFile("fluidsolver.hlsl", "CSAdvectionMacCormack", "cs_5_0", ShaderFlags.Debug))
                        _csAdvectionMacCormack = new ComputeShader(D3DDevice.Device, bytecode);
                    using (var bytecode = ShaderBytecode.CompileFromFile("fluidsolver.hlsl", "CSDrawSphere", "cs_5_0", ShaderFlags.Debug))
                        _csDrawSphere = new ComputeShader(D3DDevice.Device, bytecode);
                    using (var bytecode = ShaderBytecode.CompileFromFile("fluidsolver.hlsl", "CSDivergence", "cs_5_0", ShaderFlags.Debug))
                        _csDivergence = new ComputeShader(D3DDevice.Device, bytecode);
                    using (var bytecode = ShaderBytecode.CompileFromFile("fluidsolver.hlsl", "CSJacobi", "cs_5_0", ShaderFlags.Debug))
                        _csJacobi = new ComputeShader(D3DDevice.Device, bytecode);
                    using (var bytecode = ShaderBytecode.CompileFromFile("fluidsolver.hlsl", "CSProjection", "cs_5_0", ShaderFlags.Debug))
                        _csProjection = new ComputeShader(D3DDevice.Device, bytecode);
                    using (var bytecode = ShaderBytecode.CompileFromFile("fluidsolver.hlsl", "CSRender", "cs_5_0", ShaderFlags.Debug))
                        _csRender = new ComputeShader(D3DDevice.Device, bytecode);
*/                        
                }
                catch (CompilationException ex)
                {
                    errors = ErrorsFromString(ex.Message);
                    Logger.Error(this,"CS compile error: {0}", ex.Message);
                }
            }

            BuildRenderTarget(context);

            //_effect.GetVariableByName("RenderTargetSize").AsVector().Set(new Vector2(_usedViewport.Width, _usedViewport.Height));
//            _stopwatch.Restart();

            var deviceContext = context.D3DDevice.ImmediateContext;

            SetupConstBuffer(context);
            deviceContext.ComputeShader.SetSampler(0, _linearSamplerState);

            RunShader(deviceContext, _csAdvection, _velocitySRVs[0], null, null, _velocityUAVs[1], null, (SIZE + 15) / 16, SIZE / 4, SIZE / 4);

            //  second order correction
            RunShader(deviceContext, _csAdvectionBackwards, _velocitySRVs[1], null, null, _velocityUAVs[2], null, (SIZE + 15) / 16, SIZE / 4, SIZE / 4);
            RunShader(deviceContext, _csAdvectionMacCormack, _velocitySRVs[0], _velocitySRVs[1], _velocitySRVs[2], _velocityUAVs[3], null, (SIZE + 15) / 16, SIZE / 4, SIZE / 4);

            // velocity injection
            RunShader(deviceContext, _csDrawSphere, _velocitySRVs[3], null, null, _velocityUAVs[0], null, (SIZE + 15) / 16, SIZE / 4, SIZE / 4);

            // calc speed divergence
            RunShader(deviceContext, _csDivergence, _velocitySRVs[0], null, null, _divergenceUAV, null, (SIZE + 63) / 64, SIZE / 4, SIZE / 4);

            // jacobi
            for (int i = 0; i < 20; i++)
            {
                RunShader(deviceContext, _csJacobi, _divergenceSRV, _pressureSRVs[1], null, _pressureUAVs[0], null, (SIZE + 63) / 64, SIZE / 4, SIZE / 4);
                RunShader(deviceContext, _csJacobi, _divergenceSRV, _pressureSRVs[0], null, _pressureUAVs[1], null, (SIZE + 63) / 64, SIZE / 4, SIZE / 4);
            }


            // project
            if (BoundryVolume != null)
            {
                var boundriesSRV = new ShaderResourceView(context.D3DDevice, context.Volume);
                deviceContext.ComputeShader.SetShaderResource(2, boundriesSRV);
                RunShader(deviceContext, _csProjection, _velocitySRVs[0], _pressureSRVs[1], boundriesSRV, _velocityUAVs[1], _velocitySizeUAV, (SIZE + 63) / 64, SIZE / 4, SIZE / 4);
                deviceContext.ComputeShader.SetShaderResource(2, null);
                boundriesSRV.Dispose();
                //Logger.Info("jkfldjfl");
            }
            else
                RunShader(deviceContext, _csProjection, _velocitySRVs[0], _pressureSRVs[1], null, _velocityUAVs[1], _velocitySizeUAV, (SIZE + 63) / 64, SIZE / 4, SIZE / 4);
            
            deviceContext.PixelShader.SetShaderResource(0, null);
            // render output
            RunShader(deviceContext, _csRender, _velocitySizeSRV, _pressureSRVs[1], null, _visuUAV, null, ((int)context.Viewport.Width + 15) / 16, ((int)context.Viewport.Height + 15) / 16, 1);

            Utilities.Swap(ref _velocityVolumes[0], ref _velocityVolumes[1]);
            Utilities.Swap(ref _velocitySRVs[0], ref _velocitySRVs[1]);
            Utilities.Swap(ref _velocityUAVs[0], ref _velocityUAVs[1]);


            //            _stopwatch.Stop();
            //            Logger.Info(this,"update took: {0}ms", _stopwatch.ElapsedMilliseconds);
            Changed = false;
            context.Image = _visuTexture;
            context.Volume = _velocitySizeVolume;

            return context;
        }


        private Texture2D _visuTexture;
        private UnorderedAccessView _visuUAV;
        
        private Texture3D[] _velocityVolumes = { null, null, null, null };
        private ShaderResourceView[] _velocitySRVs = { null, null, null, null };
        private UnorderedAccessView[] _velocityUAVs = { null, null, null, null };
        private Texture3D[] _pressureVolumes = { null, null };
        private ShaderResourceView[] _pressureSRVs = { null, null };
        private UnorderedAccessView[] _pressureUAVs = { null, null };
        private Texture3D _divergenceVolume;
        private ShaderResourceView _divergenceSRV;
        private UnorderedAccessView _divergenceUAV;
        private Texture3D _velocitySizeVolume;
        private ShaderResourceView _velocitySizeSRV;
        private UnorderedAccessView _velocitySizeUAV;


        private SamplerState _linearSamplerState;
        private Buffer _constBuffer;

        private ComputeShader _csAdvection;
        private ComputeShader _csAdvectionBackwards;
        private ComputeShader _csAdvectionMacCormack;
        private ComputeShader _csDrawSphere;
        private ComputeShader _csDivergence;
        private ComputeShader _csJacobi;
        private ComputeShader _csProjection;
        private ComputeShader _csRender;
        private ComputeShader _csVisu;
    }
}

