// On the 1-bit train from NYC to Montreal
// Avoid logs rolled onto the tracks by 3-bit beavers!
// made by Mike Erwin (Danger Cobra M)
// for Synchrony 2018 demo party
// released under Mozilla Public License

#include <Arduboy2.h>
#include <Sprites.h>

Arduboy2Base ab;

typedef uint8_t u8;

class Image {
private:
  const byte* data;
  const byte* mask;

public:
  Image(const byte* d, const byte* m = nullptr)
    : data(d), mask(m) { }

  int width() const { u8 value = data[0]; return (int) value; }
  int height() const { return data[1]; }

  const byte* pxls() const { return data + 2; }

//  void draw(byte color, int x = 0, int y = 0) const
//  {
//    ab.drawBitmap(x, y, pxls(), width(), height(), color);
////    Arduboy2Base::drawBitmap(x, y, pxls(), w, h, color);
//  }

  void draw(int x = 0, int y = 0) const
  {
    if (!mask && width() == 128 && height() == 64 && x == 0 && y == 0)
      ab.paintScreen(pxls());
    else if (mask)
      Sprites::drawExternalMask(x, y, data, mask, 0, 0);
    else
      Sprites::drawOverwrite(x, y, data, 0);
  }
};

const byte sync_data[] PROGMEM = {
138, 8,
0x30,0x32,0x32,0x2,0xff,0xfc,0xf3,0xf,0xf3,0xfc,0xff,0,0xf3,0xcf,0,0xff,0xc3,0x9d,0x9d,0x9d,0xff,0,0xf3,0xf3,0,0xff,0,0xe6,0xc6,0x39,0xff,0xc3,0x9d,0x9d,0x9d,0xc3,0xff,0,0xf3,0xcf,0,0xff,0xfc,0xf3,0xf,0xf3,0xfc,0xff,0xff,0xff,0xff,0x2,0x32,0x30,0xff,0xc3,0x9d,0x9d,0x9d,0xc3,0xfe,0,0xff,0xc9,0x32,0x32,0xc9,0xff,0xff,0xff,0xff,0,0x3e,0x3e,0xc1,0xff,0x3,0xe4,0xe6,0x1,0xff,0,0xf3,0xcf,0,0xff,0xc3,0x3c,0x26,0x6,0xff,0,0x32,0x32,0xff,0,0xe6,0xc6,0x39,0xff,0xff,0xff,0xc3,0x9d,0x9d,0x9d,0xff,0xc3,0x9d,0x9d,0x9d,0xc3,0xff,0,0x66,0x66,0x99,0xff,0,0xe6,0xc6,0x39,0xff,0x3,0xe4,0xe6,0x1,0xff,0xff,0xff,0,0xf9,0xf3,0xe7,0xcf,0xe7,0xf3,0x1,
};
const Image sync { sync_data };

const byte dogcow_data[] PROGMEM = {
26, 24,
0,0x40,0x60,0x70,0x78,0xee,0xf8,0xf8,0xcc,0x80,0,0,0,0,0,0,0,0,0,0,0,0,0,0x80,0xc0,0,0,0,0,0,0,0,0x3,0xc3,0xe7,0xff,0x7f,0x7f,0x7c,0x78,0x78,0x70,0x70,0x70,0xf8,0xfe,0xfe,0xfe,0xf,0x3,0x1,0,0,0,0,0,0,0,0x10,0x1f,0xf,0,0,0,0,0,0,0,0,0,0,0x1,0x13,0x1f,0xf,0,0,0,
};
const byte dogcow_mask[] PROGMEM = {
0x40,0xe0,0xf0,0xf8,0xfe,0xff,0xfe,0xfc,0xfe,0xce,0x80,0x80,0,0,0,0,0,0,0,0,0,0,0x80,0xc0,0xf0,0xe0,0,0,0,0,0,0x3,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xf,0x3,0x1,0,0,0,0,0,0x30,0x3f,0x3f,0x1f,0xf,0,0,0,0,0,0,0,0,0x1,0x33,0x3f,0x3f,0x1f,0xf,0,0,
};
const Image dogcow { dogcow_data, dogcow_mask };

const byte rails_data[] PROGMEM = {
7, 16,
0x38,0x8,0,0x60,0x39,0x49,0x71,0xf8,0xfe,0xc2,0xd8,0xda,0xc2,0xe0, // normal
//0xc7,0xf7,0xff,0x9f,0xc6,0xb6,0x8e,0x7,0x1,0x3d,0x27,0x25,0x3d,0x1f, // inverse
};
const Image rails { rails_data };

void draw_rails(u16 phase)
{
  // rails.width() is giving me trouble... hard code
  static const u16 rail_ct = 19;
  for (int x = -phase; x < 128; x += 7)
  {
    rails.draw(x, 42);
  }
}

const byte train_engine_data[] PROGMEM = {
42, 24,
0,0,0,0,0x30,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x1c,0x1c,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x1e,0x3e,0x3e,0x3c,0x3c,0x3c,0x5c,0x6c,0xf0,0xf8,0xf0,0xe0,0xc0,0x80,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0x1,0x3,0x7,0xb,0x7d,0xfe,0xe0,0,0,0,0,0,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0x8,0,0,0xf,0x3,0,
};
const byte train_engine_mask[] PROGMEM = {
0,0,0xc0,0xf0,0xfc,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xfe,0xfe,0xfe,0xfe,0xfe,0xfc,0xfc,0xf8,0xf0,0xe0,0xc0,0x80,0,0,0,0,0x80,0xff,0xff,0xff,0xe3,0xc1,0xc1,0xc1,0xc1,0xff,0xc1,0xc1,0xc1,0xc1,0xe3,0xff,0xff,0xe3,0xc1,0xc1,0xc1,0xc1,0xff,0xc1,0xc1,0xc1,0xc1,0xe3,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xfe,0xe0,0x30,0x30,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x1f,0xf,0x3,
};
const Image train_engine { train_engine_data, train_engine_mask };

const byte train_car_data[] PROGMEM = {
38, 24,
0,0,0,0x30,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0x3c,0xcc,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0x7f,0xff,0xe0,0xc0,0xc0,0,0,0,0,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0xa,0,0,0xf,0xf,0x3,0,0,
};
const byte train_car_mask[] PROGMEM = {
0,0xc0,0xf0,0xfc,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfe,0xfc,0,0,0,0,0x80,0xff,0xff,0xff,0xe3,0xc1,0xc1,0xc1,0xc1,0xff,0xc1,0xc1,0xc1,0xc1,0xe3,0xff,0xff,0xe3,0xc1,0xc1,0xc1,0xc1,0xff,0xc1,0xc1,0xc1,0xc1,0xe3,0xff,0xff,0xff,0xff,0xff,0xff,0xe0,0xe0,0xc0,0x30,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0xf,0x3,0,
};
const Image train_car { train_car_data, train_car_mask };

const byte train_wheel_data[] PROGMEM = {
5, 8,
0,0xe,0xa,0xe,0,
};
const byte train_wheel_mask[] PROGMEM = {
0xe,0x1f,0x1f,0x1f,0xe,
};
const Image train_wheel { train_wheel_data, train_wheel_mask };

const byte log_data[] PROGMEM = {
9, 16,
0,0x80,0xc0,0xc0,0x80,0,0,0,0,0,0x1,0x3,0x3,0x1,0,0,0,0,
};
const byte log_mask[] PROGMEM = {
0x80,0xe0,0xf8,0xfe,0xff,0xff,0xff,0x3f,0xe,0x3,0x7,0x7,0x7,0x7,0x3,0,0,0,
};
const Image ob_log { log_data, log_mask };

#define JMP_HISTORY_SZ 64
int jump_history_table[JMP_HISTORY_SZ] = { 0 };
int jump_history_now = 0;
int jump_dy = 0;

int jump()
{
  if (jump_dy == 0 && jump_height() == 0)
    jump_dy = 15;
}

int jump_height_at_time(int dt)
{
  return jump_history_table[(jump_history_now - dt) % JMP_HISTORY_SZ];
}

int jump_height()
{
  return jump_history_table[jump_history_now];
}

void jump_tick()
{
  int h = jump_height() + (jump_dy-- / 4);
  if (h <= 0) { h = 0; jump_dy = 0; };
  jump_history_now = (jump_history_now + 1) % JMP_HISTORY_SZ;
  jump_history_table[jump_history_now] = h;
}

int wheel_height(int h)
{
  return 45 - 3 * h / 4;
}

void light(byte value)
{
   ab.setRGBled(value, value, value); 
}

bool has_crashed = false;
void crash()
{
  Arduboy2Base::invert(true);
  // todo: LED to 100%, then fade to black
  light(32);
  has_crashed = true;
}

void uncrash()
{
  Arduboy2Base::invert(false);
  light(0);
  has_crashed = false;
}

bool ob_exists = false;
Rect ob_rect;
Image* ob_img = nullptr;

void ob_spawn()
{
  if (ob_exists) return; // only one at a time

  ob_rect.x = 127; // right side of screen;
  ob_rect.y = 45;
  ob_rect.width = 6; // collision rect, not image rect!
  ob_rect.height = 6;
  ob_exists = true;
}

void ob_tick()
{
  ob_rect.x -= 2; // move left (rather train keeps moving right)
  if (ob_rect.x + ob_rect.width < 0)
  {
    ob_exists = false;
    return;
  }

  ob_img->draw(ob_rect.x, ob_rect.y - 5);

  Rect train_engine_rect { 22, 25 - jump_height(), 32, 24 };
  // don't include visual overhangs
  // if the back wheels cleared it that's good enough!

  if (ab.collide(ob_rect, train_engine_rect))
  {
    crash();
    ob_exists = false;
  }
}

bool sc_exists = false; // scenery
bool sc_foreground = false;
bool sc_background = false;
Image* sc_img;
Point sc_pos;

void sc_spawn()
{
  if (sc_exists) return;
  if (random(4) == 0)
  {
    sc_img = &dogcow;
    sc_foreground = true;
    sc_pos.y = random(44, 54); // bottom part of screen
  }
  else
  {
    sc_img = &sync;
    sc_background = true;
    sc_pos.y = random(5, 38); // upper part of screen
  }

  sc_pos.x = 255;
  sc_exists = true;
}

void sc_tick()
{
  if (sc_background)
    sc_pos.x -= 1; // x dim is higher res for parallax
  else
    sc_pos.x -= 3;

  if (sc_pos.x + sc_img->width() < 0)
  {
    sc_exists = false;
    sc_foreground = sc_background = false;
    return;
  }

  sc_img->draw(sc_pos.x / 2, sc_pos.y);
}

void setup() {
//  randomSeed(analogRead(3));
  ab.boot();
//  ab.setFrameRate(15);
  ab.clear();
  ob_img = &ob_log;
}

void loop() {
  if (!ab.nextFrame()) return;
  ab.pollButtons();

  if (has_crashed)
    if (ab.justPressed(A_BUTTON) || ab.justPressed(B_BUTTON))
        uncrash();
    else
      return;
 
  ab.fillScreen(WHITE);

  // scenery
  if (sc_exists && sc_background)
    sc_tick();
  else if (random(100) > 98)
    sc_spawn();

  // rails
  static u16 phase = 0;
  draw_rails(phase++);
  if (phase > 7) phase = 0;

  // action
  jump_tick();
  if (ab.justPressed(UP_BUTTON))
    jump();

  // handle obstacles
  if (ob_exists)
    ob_tick();
  else if (random(100) > 98)
    ob_spawn();
 
  // draw passenger car
  int h = jump_height_at_time(10);
  train_car.draw(-21, 25 - h);
  train_wheel.draw(0, wheel_height(h));

  // draw engine
  h = jump_height();
  train_engine.draw(15, 25 - h);
  train_wheel.draw(24, wheel_height(h));
  train_wheel.draw(36, wheel_height(h));

  // scenery
  if (sc_exists && sc_foreground)
    sc_tick();
  else if (random(100) > 98)
    sc_spawn();

  ab.display();
}
