// 1-bit image converter
// Reads a grayscale PNG, writes a C++ array of pixels.
// Can handle simple images or masked sprites.
// Created during Synchrony demoparty as a bridge between Affinity Designer & Arduboy.
// 2018 Mike Erwin
// released under Mozilla Public License

// export like this: (NOTE: out of date info!)
// struct Image { u16 width, u16 height, byte* data };
// const byte[] dogcow_data = { ... };
// Image dogcow { 26, 22, dogcow_data };

// === TODO ===
// black = 1 mode

#include <png.h>
#include <stdint.h>
#include <stdbool.h>
#include <string.h>

typedef uint8_t byte;
typedef uint32_t u32;

const char* color_type_name(int color_type)
{
	switch (color_type)
	{
		case PNG_COLOR_TYPE_GRAY: return "gray";
		case PNG_COLOR_TYPE_GRAY_ALPHA: return "gray + alpha";
		case PNG_COLOR_TYPE_PALETTE: return "palette";
		case PNG_COLOR_TYPE_RGB: return "RGB";
		case PNG_COLOR_TYPE_RGB_ALPHA: return "RGB + alpha";
		default: return "<invalid color type>";
	}
}

void base_name(const char* filename, char* name)
{
	// trim everything up to last / and file extension
	// leaving only base name
	const char* slash = strrchr(filename, '/');
	const char* dot = strrchr(filename, '.');
	if (slash && dot && slash < dot)
	{
		for (const char* c = slash + 1; c < dot; ++c)
		{
			*name++ = *c;
		}
	}
}

bool white(byte pixel)
{
	return pixel > 127;
}

bool black(byte pixel)
{
	return !white(pixel);
}

bool mask(byte pixel)
{
	return pixel < 63 || pixel >= 96;
}

bool needs_mask(u32 width, u32 height, const png_bytepp rows)
{
	for (u32 y = 0; y < height; ++y)
		for (u32 x = 0; x < width; ++x)
			if (!mask(rows[y][x]))
				return true;
	return false;
}

void bitify(u32 width, u32 height, const png_bytepp rows, bool (*func)(byte), const char* name, const char* suffix)
{
	printf("const byte %s_%s[] PROGMEM = {\n", name, suffix);
	if (func != mask)
		printf("%d, %d,\n", width, height);

	const u32 pixels_per_chunk = 8;
	const u32 chunky_rows = height / pixels_per_chunk;

	for (u32 row = 0; row < chunky_rows; ++row)
	{
		for (u32 x = 0; x < width; ++x)
		{
			byte bits = 0;
			for (u32 y = row * pixels_per_chunk; y < row * pixels_per_chunk + pixels_per_chunk; ++y)
			{
				const byte pixel = rows[y][x];
				byte new_bit = func(pixel) ? 0x80 : 0;
				bits = (bits >> 1) | new_bit;
			}

			// output each byte as we go
			printf("%#x,", bits);
		}
	}

	printf("\n};\n");
}

int main(int argc, const char* argv[])
{
	if (argc != 2)
	{
		printf("usage: %s filename.png\n", argv[0]);
		return 1;
	}

	const char* filename = argv[1];
	#define NAME_LEN 256
	char name[NAME_LEN] = { 0 };
	base_name(filename, name);

	FILE *fp = fopen(argv[1], "rb");
	if (!fp)
	{
		printf("could not open '%s'\n", filename);
		return 1;
	}

	#define SIG_SIZE 8
	byte sig[SIG_SIZE];
	if (fread(sig, 1, SIG_SIZE, fp) != SIG_SIZE)
	{
		printf("could not read PNG signature from '%s'\n", filename);
		return 1;
	}

	const bool is_png = !png_sig_cmp(sig, 0, SIG_SIZE);
	if (!is_png)
	{
		printf("'%s' is not in PNG format\n", filename);
		return 1;
	}

	png_structp png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);

	if (!png_ptr)
	{
		printf("failed on line %d\n", __LINE__);
		return 1;
	}

	png_infop info_ptr = png_create_info_struct(png_ptr);

	if (!info_ptr)
	{
		png_destroy_read_struct(&png_ptr, NULL, NULL);
		printf("failed on line %d\n", __LINE__);
		return 1;
	}

	png_init_io(png_ptr, fp);
	png_set_sig_bytes(png_ptr, SIG_SIZE);
	
	png_read_png(png_ptr, info_ptr, PNG_TRANSFORM_PACKING | PNG_TRANSFORM_SCALE_16, NULL);

	u32 width, height;
	int bit_depth, color_type;

	png_get_IHDR(png_ptr, info_ptr, &width, &height, &bit_depth, &color_type, NULL, NULL, NULL);

	printf("%ux%u %d-bit %s\n", width, height, bit_depth, color_type_name(color_type));

	const u32 max_width = 255, max_height = 248;
	if (width > max_width || height > max_height || color_type != PNG_COLOR_TYPE_GRAY)
	{
		printf("this program supports up to %dx%d grayscale\n", max_width, max_height);
		return 1;
	}

	if (height % 8)
	{
		// simplifies bytes --> bits conversion
		// I think only height should have this restriction...
		printf("image height must be multiple of 8\n");
		return 1;
	}

	const png_bytepp rows = png_get_rows(png_ptr, info_ptr);

	const char* data_suffix = "data";
	const char* mask_suffix = "mask";

	bitify(width, height, rows, white, name, data_suffix);

	if (needs_mask(width, height, rows))
	{
		bitify(width, height, rows, mask, name, mask_suffix);
		printf("const Image %s { %s_%s, %s_%s };\n", name, name, data_suffix, name, mask_suffix);
	}
	else
	{
		printf("const Image %s { %s_%s };\n", name, name, data_suffix);
	}

	return 0;
}
