//>>> _using
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SharpDX;
using SharpDX.Direct3D11;
using SharpDX.Windows;
//<<< _using

using System.Runtime.InteropServices;
using Framefield.Core;
using Framefield.Core.Rendering;
using SharpDX.DXGI;
using SharpDX.Direct3D;
using SharpDX.D3DCompiler;
using System.CodeDom.Compiler;
using Buffer = SharpDX.Direct3D11.Buffer;

namespace Framefield.Core.IDc47e6d4a_5447_4bb0_9896_4cedc0619a37
{
    public class Class_GradientChains : FXSourceCodeFunction, IFXSceneSourceCode, Framefield.Core.OperatorPartTraits.ITimeAccessor
    {
        //structured buffer element type. size must be multiple of 4
        [StructLayout(LayoutKind.Explicit, Size = 2*3*4 + 2*4)]
        public struct ParticleStateLayout
        {
            [FieldOffset(0)]     public  Vector3 Position;
            [FieldOffset(1*3*4)] public  Vector3 Velocity;
            [FieldOffset(2*3*4)] public  float Gradient;
            [FieldOffset(2*3*4+1)] public  float  Padding;
        }

        // Init Parameters / size must be multiple of 16
        [StructLayout(LayoutKind.Explicit, Size = 2*3*4 + 4 + 4)]
        public struct ParticleInitParametersBufferLayout
        {
            [FieldOffset(0)]         public Vector3 CenterPosition;
            [FieldOffset(1*3*4)]     public Vector3 Spacing;
            [FieldOffset(2*3*4)]     public int Reset;
            [FieldOffset(2*3*4 + 4)] int Padding;
        }
        private Buffer _particleInitParametersBuffer;

        // Update parameters / Size must be multiple of 16
        [StructLayout(LayoutKind.Explicit, Size = 20*4)]
        public struct ParticleUpdateParametersBufferLayout
        {
            [FieldOffset(0 * 4)]  public float DeltaTime;
            [FieldOffset(1 * 4)]  public float Friction;
            [FieldOffset(2 * 4)]  public float Acceleration;
            [FieldOffset(3 * 4)]  public float Stiffness;
            [FieldOffset(4 * 4)]  public float SampleRadius;
            [FieldOffset(5 * 4)]  public int SampleCount;
            [FieldOffset(6 * 4)]  public int ChainCount;
            [FieldOffset(7 * 4)]  public int ChainSegmentCount;
            [FieldOffset(8 * 4)]  public float SpacingY;
            [FieldOffset(9 * 4)]  public float Time;
            [FieldOffset(10 * 4)]  public float Alignment;
            [FieldOffset(11 * 4)]  public float Scale;
            [FieldOffset(12 * 4)]  public float AspectRatio;
            [FieldOffset(13 * 4)]  public Vector3 _padding;

        }
        private Buffer _particleUpdateParametersBuffer;

        //>>> _inputids
        private enum InputId
        {
            InitCode = 0,
            UpdateCode = 1,
            RenderCode = 2,
            Image = 3,
            ChainCount = 4,
            ChainSegmentCount = 5,
            SampleRadius = 6,
            Acceleration = 7,
            Friction = 8,
            Stiffness = 9,
            SpacingX = 10,
            SpacingY = 11,
            Scale = 12,
            ResetTrigger = 13,
            CenterPositionX = 14,
            CenterPositionY = 15,
            CenterPositionZ = 16,
            Alignment = 17,
            SampleCount = 18,
            ColorR = 19,
            ColorG = 20,
            ColorB = 21,
            ColorA = 22
        }
        //<<< _inputids

        public override void Dispose()
        {
            Utilities.DisposeObj(ref _csInitParticles);
            Utilities.DisposeObj(ref _csUpdateParticles);
            Utilities.DisposeObj(ref _fxRenderParticles);
            Utilities.DisposeObj(ref _particleStateBuffer);
            Utilities.DisposeObj(ref _imageView);
            base.Dispose();
        }

        bool _firstEval = true;
        public override OperatorPartContext Eval(OperatorPartContext context, List<OperatorPart> inputs, int outputIdx)
        {
            if (_firstEval)
            {
            
                for (int i = 0; i < NumCodes(); ++i) {
                    Compile(i);
                }
                
                _firstEval = false;
                Changed = true;
            }

            //>>> _params
            var InitCode = inputs[(int)InputId.InitCode].Eval(context).Text;
            var UpdateCode = inputs[(int)InputId.UpdateCode].Eval(context).Text;
            var RenderCode = inputs[(int)InputId.RenderCode].Eval(context).Text;
            var Image = inputs[(int)InputId.Image].Eval(context).Image; // Needs to be checked for null!
            var ChainCount = inputs[(int)InputId.ChainCount].Eval(context).Value;
            var ChainSegmentCount = inputs[(int)InputId.ChainSegmentCount].Eval(context).Value;
            var SampleRadius = inputs[(int)InputId.SampleRadius].Eval(context).Value;
            var Acceleration = inputs[(int)InputId.Acceleration].Eval(context).Value;
            var Friction = inputs[(int)InputId.Friction].Eval(context).Value;
            var Stiffness = inputs[(int)InputId.Stiffness].Eval(context).Value;
            var SpacingX = inputs[(int)InputId.SpacingX].Eval(context).Value;
            var SpacingY = inputs[(int)InputId.SpacingY].Eval(context).Value;
            var Spacing = new Vector2(SpacingX, SpacingY);
            var Scale = inputs[(int)InputId.Scale].Eval(context).Value;
            var ResetTrigger = inputs[(int)InputId.ResetTrigger].Eval(context).Value;
            var CenterPositionX = inputs[(int)InputId.CenterPositionX].Eval(context).Value;
            var CenterPositionY = inputs[(int)InputId.CenterPositionY].Eval(context).Value;
            var CenterPositionZ = inputs[(int)InputId.CenterPositionZ].Eval(context).Value;
            var CenterPosition = new Vector3(CenterPositionX, CenterPositionY, CenterPositionZ);
            var Alignment = inputs[(int)InputId.Alignment].Eval(context).Value;
            var SampleCount = inputs[(int)InputId.SampleCount].Eval(context).Value;
            var ColorR = inputs[(int)InputId.ColorR].Eval(context).Value;
            var ColorG = inputs[(int)InputId.ColorG].Eval(context).Value;
            var ColorB = inputs[(int)InputId.ColorB].Eval(context).Value;
            var ColorA = inputs[(int)InputId.ColorA].Eval(context).Value;
            var Color = new Color4(ColorR, ColorG, ColorB, ColorA);
            //<<< _params

            //Logger.Info("aspect:" + context.Viewport.Width / context.Viewport.Height * 1.0f);

            _chainCount = (int)ChainCount;
            _chainSegmentCount = (int)ChainSegmentCount;
            _centerPosition = CenterPosition;
            _spacing = new Vector3(Spacing.X, Spacing.Y, 0);
            
            float deltaTime = context.Time - _time;
            _time = context.Time;

            var newParticleCount = (int)(Math.Max(512, ChainCount * ChainSegmentCount));
            InitializeParticleStateBuffer(context, newParticleCount );

            var uavDesc = new UnorderedAccessViewDescription
                              {
                                  Format = Format.Unknown, 
                                  Dimension = UnorderedAccessViewDimension.Buffer,
                                  Buffer = new UnorderedAccessViewDescription.BufferResource
                                               {
                                                   FirstElement = 0,
                                                   ElementCount = _maxNumParticles,
                                                   Flags = UnorderedAccessViewBufferFlags.None
                                               }
                              };

            var deviceContext = context.D3DDevice.ImmediateContext;

            // Particle initializing -----------------------------------------------            
            var particleInitParameterLayout = new ParticleInitParametersBufferLayout() {
                CenterPosition= Vector3.Zero,
                Spacing= _spacing,
                Reset= (int)ResetTrigger,                 
            };
            
            
            if(Changed || _imageView != null && Image != null) 
            {
                 Utilities.DisposeObj(ref _imageView);
                _imageView = new ShaderResourceView(context.D3DDevice, Image);
            }

            
            BaseRenderer.SetupConstBufferForCS(
                context, 
                particleInitParameterLayout , 
                ref _particleInitParametersBuffer,
                0);

            using (var particleStateUAV = new UnorderedAccessView(context.D3DDevice, _particleStateBuffer, uavDesc))
            {
                deviceContext.ComputeShader.Set(_csInitParticles);
                deviceContext.ComputeShader.SetUnorderedAccessView(0, particleStateUAV);

                deviceContext.Dispatch(_maxNumParticles/512, 1, 1);

                deviceContext.ComputeShader.SetUnorderedAccessView(0, null);
                deviceContext.ComputeShader.SetConstantBuffer(0, null);
            }

            // -- updating ----------------------------------------------------
            BaseRenderer.SetupConstBufferForCS(
                context, 
                new ParticleUpdateParametersBufferLayout() {
                    DeltaTime = deltaTime,
                    Friction = Friction,
                    Acceleration = Acceleration,
                    Stiffness = Stiffness,
                    SampleRadius = SampleRadius,                    
                    SampleCount = (int)SampleCount,
                    ChainCount = (int)ChainCount,
                    ChainSegmentCount = (int)ChainSegmentCount,
                    Time = context.Time,
                    SpacingY = SpacingY,
                    Alignment = Alignment,
                    Scale = Scale,
                    AspectRatio = context.Viewport.Width / context.Viewport.Height * 1.0f,
                }, 
                ref _particleUpdateParametersBuffer,
                0);     

            using (var particleStateUAV = new UnorderedAccessView(context.D3DDevice, _particleStateBuffer, uavDesc))
            {
                deviceContext.ComputeShader.Set(_csUpdateParticles);
                deviceContext.ComputeShader.SetUnorderedAccessView(0, particleStateUAV);
                deviceContext.ComputeShader.SetShaderResource(0, _imageView);

                deviceContext.Dispatch(_maxNumParticles/512, 1, 1);

                deviceContext.ComputeShader.SetShaderResource(0, null);
                deviceContext.ComputeShader.SetUnorderedAccessView(0, null);
                deviceContext.ComputeShader.SetConstantBuffer(0, null);
            }

            // Particle rendering ----------------------------------------------
            using (var particleStateSRV = new ShaderResourceView(context.D3DDevice, _particleStateBuffer))
            using (var imageSRV = new ShaderResourceView(context.D3DDevice, Image))
            {
                try
                {
                    deviceContext.ClearState();

                    _fxRenderParticles.GetVariableByName("ParticleStates").AsShaderResource().SetResource(particleStateSRV);
                    _fxRenderParticles.GetVariableByName("Image").AsShaderResource().SetResource(imageSRV);
                    _fxRenderParticles.GetVariableByName("Time").AsScalar().Set((float)_time);
                    _fxRenderParticles.GetVariableByName("Scale").AsScalar().Set((float)Scale);
                    _fxRenderParticles.GetVariableByName("Color").AsVector().Set(Color);
                    
                    var previousEffect = context.Effect;
                    context.Effect = _fxRenderParticles;
                    context.Renderer.SetupEffect(context);

                    if (context.DepthStencilView != null)
                        deviceContext.OutputMerger.SetTargets(context.DepthStencilView, context.RenderTargetView);
                    else
                        deviceContext.OutputMerger.SetTargets(context.RenderTargetView);

                    if (context.BlendState != null) {
                        deviceContext.OutputMerger.BlendState = context.BlendState;
                        deviceContext.OutputMerger.BlendFactor = context.BlendFactor;
                    }

                    if (context.DepthStencilState != null) {
                        deviceContext.OutputMerger.DepthStencilState = context.DepthStencilState;
                    }

                    if (context.RasterizerState != null) {
                        deviceContext.Rasterizer.State = context.RasterizerState;
                    }

                    deviceContext.Rasterizer.SetViewports(new [] { context.Viewport });
                    deviceContext.InputAssembler.InputLayout = context.InputLayout;
                    deviceContext.InputAssembler.PrimitiveTopology = PrimitiveTopology.PointList;
                    deviceContext.InputAssembler.SetVertexBuffers(0, new VertexBufferBinding(null, 0, 0));

                    var technique = context.Effect.GetTechniqueByIndex(0);
                    for (int i = 0; i < technique.Description.PassCount; ++i) {
                        technique.GetPassByIndex(i).Apply(deviceContext);
                        deviceContext.Draw(_maxNumParticles, 0);
                    }

                    // remove target views that they are no longer bound as output and can be used also as input
                    DepthStencilView dsv = null;
                    RenderTargetView rtv = null;
                    deviceContext.OutputMerger.SetTargets(dsv, rtv);
                    deviceContext.VertexShader.SetShaderResource(0, null);
                    deviceContext.PixelShader.SetShaderResource(0, null);
                    context.Effect = previousEffect;
                }
                catch (Exception exception)
                {
                    Logger.Error(this,"render error: {0}", exception.Message);
                }
            }

            Changed = false;
            return context;
        }


        void InitializeParticleStateBuffer(OperatorPartContext context, int newParticleCount)
        {
            //var newMaxiNumParticles = _maxNumParticles;
            newParticleCount = Math.Max(512, newParticleCount); //at least we need to have 512 particles

            if (_particleStateBuffer == null || _maxNumParticles != newParticleCount)
            {
                using (var data = new DataStream(Marshal.SizeOf(typeof(ParticleStateLayout)) * newParticleCount, true, true))
                {
                var topLeft = _centerPosition - _spacing * new Vector3(_chainCount * 0.5f, _chainSegmentCount * 0.5f,1);
                    for (int i = 0; i < _chainCount; ++i) 
                    {
                        for (int j = 0; j < _chainSegmentCount; ++j) 
                        {
                            var pData = new ParticleStateLayout() {                        
                                Position= topLeft + _spacing * new Vector3(i,j,0), 
                                Velocity= new Vector3(0 ,0, 0)
                            };
                            data.Write(pData);
                        }
                    }
                    data.Position = 0;

                    var bufferDesc = new BufferDescription
                                         {
                                             Usage = ResourceUsage.Default,
                                             SizeInBytes = Marshal.SizeOf(typeof(ParticleStateLayout)) * newParticleCount,
                                             StructureByteStride = Marshal.SizeOf(typeof(ParticleStateLayout)),
                                             BindFlags = BindFlags.ShaderResource | BindFlags.UnorderedAccess,
                                             OptionFlags = ResourceOptionFlags.BufferStructured
                                         };
                    _particleStateBuffer = new Buffer(context.D3DDevice, data, bufferDesc);
                }
                _maxNumParticles = newParticleCount;
            }
        }

        public override int NumCodes()
        {
            return 3;
        }

        public override CompilerErrorCollection Compile(int codeIdx)
        {
            var errors = new CompilerErrorCollection();
            try
            {
                switch (codeIdx)
                {
                    case 0:
                    {
                        Utilities.DisposeObj(ref _csInitParticles);
                        var t = GetCode(codeIdx);
                        using (var bytecode = ShaderBytecode.Compile(GetCode(codeIdx), "CSInitParticles", "cs_5_0", ShaderFlags.Debug))
                            _csInitParticles = new ComputeShader(D3DDevice.Device, bytecode);
                        break;
                    }
                    case 1:
                    {
                        Utilities.DisposeObj(ref _csUpdateParticles);
                        using (var bytecode = ShaderBytecode.Compile(GetCode(codeIdx), "CSUpdateParticles", "cs_5_0", ShaderFlags.Debug))
                            _csUpdateParticles = new ComputeShader(D3DDevice.Device, bytecode);
                        break;
                    }
                    case 2:
                    {
                        Utilities.DisposeObj(ref _fxRenderParticles);
                        using (var bytecode = ShaderBytecode.Compile(GetCode(codeIdx), "fx_5_0", ShaderFlags.Debug, EffectFlags.None, null, null))
                            _fxRenderParticles = new Effect(D3DDevice.Device, bytecode);
                        break;
                    }
                }
            }
            catch (SharpDX.CompilationException ex)
            {
                errors = ErrorsFromString(ex.Message);
                Logger.Error(this,"Fx compile error: {0}", ex.Message);
            }
            return errors;
        }

        ShaderResourceView _imageView;
        ComputeShader _csInitParticles;
        ComputeShader _csUpdateParticles;
        Effect _fxRenderParticles;
        Buffer _particleStateBuffer;
        Vector3 _centerPosition;
        Vector3 _spacing;
        int _chainCount;
        int _chainSegmentCount;        
        int _maxNumParticles;
        float _time;
    }
}
